/*
 * ====================================================================
 *
 * Copyright (c) 2000 Attila Szegedi.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes the Expose library
 *        (http://www.szegedi.org/expose)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Expose" and "Attila Szegedi" must not be used to endorse
 *    or promote products derived from this software without prior written
 *    permission. For written permission, please contact expose@szegedi.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR ITS CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 */

package org.szegedi.expose.model;

import freemarker.template.*;

import java.beans.*;
import java.lang.reflect.*;
import java.util.*;

/**
 * A class that will wrap an arbitrary object into
 * {@link TemplateHashModel} interface. It uses Beans {@link Introspector}
 * to dinamically discover the properties and methods. Properties will be wrapped
 * into {@link ReflectionMethodModel} and {@link ReflectionScalarModel} class
 * instances. The models are cached (meaning requesting a model for same object
 * twice will return the same model instance) unless the system property
 * <tt>expose.reflection.nocache</tt> is set to true.
 * @author Attila Szegedi, attila@szegedi.org
 * @version 1.0
 */

public class ReflectionObjectModel
extends
	ReflectionObjectModelBase
implements
	TemplateHashModel
{
	private static final Integer GENERIC_GET_KEY = new Integer(0);
	private static final String GENERIC_GET = "get";
	private static final Class[] STRING_PARAMETER = new Class[] { java.lang.String.class };
	private static final Class[] OBJECT_PARAMETER = new Class[] { java.lang.Object.class };

	// Cache of hashtables that contain already discovered properties and methods
	// for a specified class. Each key is a Class, each value is a Hashtable. In
	// that Hashtable, each key is a property/method name, each value is a
	// MethodDescriptor or a PropertyDescriptor assigned to that property/method.
	private static final Map classCache = new HashMap();

	// Cached template models that implement member properties and methods for this
	// instance. Keys are FeatureDescriptor instances (from classCache values),
	// values are either ReflectionMethodModels/ReflectionScalarModels
	private final HashMap memberCache = new HashMap();

	/**
	 * Creates a new model that wraps the specified object.
	 * Note that there are specialized subclasses of this class for wrapping arrays,
	 * collections, enumeration, iterators, and maps. Note also that the superclass can
	 * be used to wrap String objects if only scalar functionality is needed.
	 * You can also choose to delegate the choice over which model class is used for
	 * wrapping to {@link ReflectionUtilities#wrap(Object)}.
	 * @param object the object to wrap into a model.
	 */
	public ReflectionObjectModel(Object object)
	{
		super(object);
		Class clazz = object.getClass();
		synchronized(classCache)
		{
			if(!classCache.containsKey(clazz))
				classCache.put(clazz, new HashMap());
		}
	}

	/**
	 * Returns a model wrapping the specified object. If there is already
	 * a cached model instance for this object, returns the cached model instance.
	 * Models are cached using {@link WeakReference} objects. The caching can be turned
	 * off by setting the <tt>expose.reflection.nocache</tt> system property to
	 * true. In this case calling this method is equivalent to constructing a new model.
	 * Note that there are specialized subclasses of this class for wrapping arrays,
	 * collections, enumeration, iterators, and maps. Note also that the superclass can
	 * be used to wrap String objects if only scalar functionality is needed.
	 * You can also choose to delegate the choice over which model class is used for
	 * wrapping to {@link ReflectionUtilities#wrap(Object)}.
	 * @param object the object to wrap into a model.
	 * @return the model for the object.
	 */
	public static final ReflectionObjectModel getInstance(Object object)
	{
		if(noCache)
			return new ReflectionObjectModel(object);

		ReflectionObjectModel model = (ReflectionObjectModel)lookup(object);
		if(model == null)
		{
			model = new ReflectionObjectModel(object);
			register(model, object);
		}
		return model;
	}

	/**
	 * Returns the type of this object (which is TYPE_OBJECT)
	 */
	public int getType()
	{
		return TYPE_OBJECT;
	}

	/**
	 * Uses Beans introspection to locate a property or method with name matching the
	 * key name. If a method or property is found, it is wrapped into
	 * {@link ReflectionMethodModel} or {@link ReflectionScalarModel} instance and
	 * returned. Models for various properties and methods are cached on a per-class
	 * basis, so the costly introspection is performed only once per property or method
	 * of a class. If no method or propery matching the key is found, the framework
	 * will try to invoke methods with signature <tt>get(java.lang.String)</tt>,
	 * then <tt>get(java.lang.Object)</tt>.
	 * @throws TemplateModelException if there was no property nor method nor
	 * a generic <tt>get</tt> method to invoke.
	 */
	public TemplateModel get(String key)
	throws
		TemplateModelException
	{
		Map keyMap = null;
		Class clazz = object.getClass();
		synchronized(classCache)
		{
			keyMap = (Map)classCache.get(clazz);
		}

		try
		{
			FeatureDescriptor fd = null;
			synchronized(keyMap)
			{
				fd = (FeatureDescriptor)keyMap.get(key);
				if(fd == null)
				{
					if(!keyMap.containsKey(key))
					{
						// Lookup the descriptor using beans introspection
						fd = lookupDescriptor(clazz, key);
						// Note that if nothing was found, null will still be bound into
						// the keyMap. This will allow for fail-fast next time a
						// nonexistant key is requested due to !containsKey condition.
						keyMap.put(key, fd);
					}
				}
			}
			return
				fd == null
				? invokeGenericGet(keyMap, clazz, key)
				: invokeThroughDescriptor(fd);
		}
		catch(TemplateModelException e)
		{
			throw e;
		}
		catch(Exception e)
		{
			java.io.StringWriter sw = new java.io.StringWriter();
			java.io.PrintWriter pw = new java.io.PrintWriter(sw);
			pw.println("Failed on " + object.getClass().getName());
			e.printStackTrace(pw);
			throw new TemplateModelException(sw.toString());
		}
	}

	private static FeatureDescriptor lookupDescriptor(Class clazz, String key)
	throws
		IntrospectionException
	{
		// Look if we have a property with this type
		BeanInfo beanInfo = Introspector.getBeanInfo(clazz);
		PropertyDescriptor[] pda = beanInfo.getPropertyDescriptors();
		for(int i = pda.length - 1; i >= 0; --i)
		{
			PropertyDescriptor pd = pda[i];
			if(pd.getName().equals(key))
				// OK, there's a property
				return pd;
		}

		// No property, look for method
		MethodDescriptor[] mda = beanInfo.getMethodDescriptors();
		for(int i = mda.length - 1; i >= 0; --i)
		{
			MethodDescriptor md = mda[i];
			if(md.getName().equals(key))
				// OK, there's a method
				return md;
		}

		// Nothing found
		return null;
	}

	private TemplateModel invokeThroughDescriptor(FeatureDescriptor fd)
	throws
		NoSuchMethodException,
		IllegalAccessException,
		InvocationTargetException
	{
		// See if this particular instance has a cached implementation
		// for the requested feature descriptor
		TemplateModel member = null;
		synchronized(memberCache)
		{
			member = (TemplateModel)memberCache.get(fd);
		}

		if(member != null)
			return member;
		
		TemplateModel retval = null;
		if(fd instanceof IndexedPropertyDescriptor)
		{
			retval = member = new ReflectionMethodModel(object, ReflectionUtilities.getPublicMethod(((IndexedPropertyDescriptor)fd).getIndexedReadMethod()));
		}
		else if(fd instanceof PropertyDescriptor)
		{
			PropertyDescriptor pd = (PropertyDescriptor)fd;
			if(ReflectionUtilities.isScalar(pd.getPropertyType()))
			{
				retval = member = new ReflectionScalarModel(object, ReflectionUtilities.getPublicMethod(pd.getReadMethod()));
			}
			else
			{
				retval = ReflectionUtilities.wrap(ReflectionUtilities.getPublicMethod(pd.getReadMethod()).invoke(object, null));
				// (member == null) condition remains, as we don't cache these
			}
		}
		else // if(fd instanceof MethodDescriptor)
		{
			retval = member = new ReflectionMethodModel(object, ReflectionUtilities.getPublicMethod(((MethodDescriptor)fd).getMethod()));
		}
		// If new cacheable member was created, cache it
		if(member != null)
		{
			synchronized(memberCache)
			{
				memberCache.put(fd, member);
			}
		}
    return retval;
	}

	private TemplateModel invokeGenericGet(Map keyMap, Class clazz, String key)
	throws
		NoSuchMethodException,
		IllegalAccessException,
		InvocationTargetException,
		TemplateModelException
	{
		Method genericGet = null;
		synchronized(keyMap)
		{
			// Lookup the cached get() method
			genericGet = (Method)keyMap.get(GENERIC_GET_KEY);
			// If it is not cached, and has not yet been looked up, look it up now
			if(genericGet == null && !keyMap.containsKey(GENERIC_GET_KEY))
			{
				// First try "get(java.lang.String)"
				genericGet = ReflectionUtilities.getPublicMethodOrNull(clazz.getDeclaredMethod(GENERIC_GET, STRING_PARAMETER));
				if(genericGet == null)
					// Next try "get(java.lang.Object)"
					genericGet = ReflectionUtilities.getPublicMethodOrNull(clazz.getDeclaredMethod(GENERIC_GET, OBJECT_PARAMETER));
				keyMap.put(GENERIC_GET_KEY, genericGet);
			}
		}
		if(genericGet == null)
			throw new TemplateModelException("No such key: " + key + " on instance of " + object.getClass().getName());

		return ReflectionUtilities.wrap(genericGet.invoke(object, new Object[] { key }));
	}
}
