/*
 * ====================================================================
 *
 * Copyright (c) 2000 Attila Szegedi.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes the Expose library
 *        (http://www.szegedi.org/expose)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Expose" and "Attila Szegedi" must not be used to endorse 
 *    or promote products derived from this software without prior written
 *    permission. For written permission, please contact expose@szegedi.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR ITS CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 */

package org.szegedi.expose.model;

import freemarker.template.*;

import java.lang.reflect.*;
import java.io.*;
import java.util.*;

/**
 * A class that will wrap a reflected method call into a
 * {@link TemplateMethodModel} interface. It can be used directly, and is
 * heavily used by {@link ReflectionObjectModel} to wrap reflected method calls.
 * @author Attila Szegedi, attila@szegedi.org
 * @version 1.0
 */

public final class ReflectionMethodModel
implements
	TemplateMethodModel
{
	private Object wrap;
	private Method getter;
	private boolean scalar;

	/**
	 * Creates a model for a specific method on a specific object.
	 * @param object the object to call the method on. Can be
	 * <tt>null</tt> for static methods.
	 * @param method the method that will be invoked.
	 */
	public ReflectionMethodModel(Object object, Method method)
	{
		this.wrap = object;
		this.getter = method;
		scalar = ReflectionUtilities.isScalar(getter.getReturnType());
	}

	/**
	 * Invokes the method, passing it the arguments from the list.
	 * As in current release of Freemarker (1.52), all elements of
	 * the argument list are Strings, the method must be
	 * callable with all-String parameters.
	 */
	public TemplateModel exec(List arguments)
	throws
		TemplateModelException
	{
		try
		{
			Object[] args = null;
			//TODO: check if we actually receive anything except string in the list.
			if(arguments != null)
			{
				int size = arguments.size();
				args = new Object[size];
				Iterator it = arguments.iterator();
				int i = 0;
				while(it.hasNext())
					args[i++] = it.next();
			}
      try
			{
				//TODO: Coerce arguments into right types
				return ReflectionUtilities.wrap(getter.invoke(wrap, args), scalar);
			}
			catch(IllegalArgumentException e)
			{
				// HACK: Maybe the method is overloaded. Try to find one with right arguments.
				Class[] paramClasses = new Class[args.length];
				for(int i = args.length - 1; i >= 0; --i)
					paramClasses[i] = args[i] == null ? java.lang.Object.class : args[i].getClass();
				Class scope = wrap == null ? getter.getDeclaringClass() : wrap.getClass();
				try
				{
					Method method2 = scope.getMethod(getter.getName(), paramClasses);
					if(method2 == null)
						throw e;
					// Set the new method as default getter
					getter = ReflectionUtilities.getPublicMethod(method2);
					// Retry the call
					return ReflectionUtilities.wrap(getter.invoke(wrap, args), scalar);
				}
				catch(NoSuchMethodException e2)
				{
					throw e;
				}
				catch(SecurityException e2)
				{
					throw e;
				}
			}
		}
		catch(Exception e)
		{
			StringWriter sw = new StringWriter();
			PrintWriter pw = new PrintWriter(sw);
			pw.println("Method" + getter.getName() + " on " + wrap.getClass().getName());
			e.printStackTrace(pw);
			throw new TemplateModelException(sw.toString());
		}
	}

	/**
	 * Returns true if the wrapped method's return type is <tt>void</tt>.
	 */
	public boolean isEmpty()
	{
		return getter.getReturnType() == Void.TYPE;
	}

}
