/*
 * ====================================================================
 *
 * Copyright (c) 2000 Attila Szegedi.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes the Expose library
 *        (http://www.szegedi.org/expose)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Expose" and "Attila Szegedi" must not be used to endorse 
 *    or promote products derived from this software without prior written
 *    permission. For written permission, please contact expose@szegedi.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR ITS CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 */

package org.szegedi.expose.model;

import freemarker.template.*;
import java.util.*;

/**
 * A special case of {@link ReflectionObjectModel} that adds implementation
 * for {@link TemplateListModel} and {@link TemplateMethodModel} on map objects.
 * List model will iterate through entries, while method model is a shortcut for the
 * <tt>Map.get()</tt> method.
 * Using the model as a list model is thread-safe, as it maintains an iterator
 * on a per-thread basis.
 * @author Attila Szegedi, attila@szegedi.org
 * @version 1.0
 */
public class ReflectionMapModel
extends
	ReflectionObjectModel
implements
	TemplateListModel,
	TemplateMethodModel
{
	private ThreadLocal iterator = new ThreadLocal();
	
	/**
	 * Creates a new model that wraps the specified map object.
	 * @param object the map object to wrap into a model.
	 */
	public ReflectionMapModel(Map map)
	{
		super(map);
	}
	
	/**
	 * Returns a model wrapping the specified map object. If there is already
	 * a cached model instance for this map, returns the cached model instance.
	 * Models are cached using {@link WeakReference} objects. The caching can be turned
	 * off by setting the <tt>expose.reflection.nocache</tt> system property to
	 * true. In this case calling this method is equivalent to constructing a new model.
	 * @param object the map to wrap into a model.
	 * @return the model for the map
	 */
	public static final ReflectionMapModel getInstance(Map object)
	{
		if(noCache)
			return new ReflectionMapModel(object);
		
		ReflectionMapModel model = (ReflectionMapModel)lookup(object);
		if(model == null)
		{
			model = new ReflectionMapModel(object);
			register(model, object);
		}
		return model;
	}

	/**
	 * Returns the type of this object (which is TYPE_MAP)
	 */
	public int getType()
	{
		return TYPE_MAP;
	}
	
	/**
	 * Calls {@link Iterator#hasNext()}.
	 */
	public boolean hasNext()
	{
		boolean retval = ((Iterator)iterator.get()).hasNext();
		if(!retval)
			iterator.set(null);
		return retval;
	}
	
	/**
	 * Always return false to force calling rewind.
	 */
	public boolean isRewound()
	{
		return false;
	}
	
	/**
	 * Calls {@link Iterator#next()} and wraps the returned {@link Map.Entry} into a
	 * template model.
	 */
	public TemplateModel next()
	{
		return ReflectionObjectModel.getInstance(((Iterator)iterator.get()).next());
	}
	
	/**
	 * Creates a new iterator on the map's entry set that will be used to implement 
	 * other {@link TemplateListModel} methods.
	 */
	public void rewind()
	{
		iterator.set(((Map)object).entrySet().iterator());
	}

	/**
	 * The first argument is used as a key to call the map's <tt>get</tt> method.
	 */
	public TemplateModel exec(List arguments)
	{
		return ReflectionUtilities.wrap(((Map)object).get(arguments.get(0)));
	}

	/**
	 * Returns true if the underlying map is empty.
	 */
	public boolean isEmpty()
	{
		return ((Map)object).isEmpty();
	}

	/**
	 * Throws an UnsupportedOperationException as map entry set elements can not
	 * be accessed at random.
	 */
	public TemplateModel get(int index)
	{
		throw new UnsupportedOperationException();
	}
	
	/**
	 * Returns a Simple scalar with a string containing the decimal 
	 * representation of the map size.
	 */
	public TemplateModel listSize()
	{
		return new SimpleScalar(Integer.toString(((Map)object).size()));
	}
}
