/*
 * ====================================================================
 *
 * Copyright (c) 2000 Attila Szegedi.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes the Expose library
 *        (http://www.szegedi.org/expose)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Expose" and "Attila Szegedi" must not be used to endorse 
 *    or promote products derived from this software without prior written
 *    permission. For written permission, please contact expose@szegedi.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR ITS CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 */

package org.szegedi.expose.model;

import freemarker.template.*;
import java.util.*;

/**
 * A class that adds {@link TemplateListModel} functionality to the
 * {@link Iterator} interface implementers.
 * Using the model as a list model is NOT thread-safe, as iterators are
 * inherently not thread-safe.
 * @author Attila Szegedi, attila@szegedi.org
 * @version 1.0
 */

public final class ReflectionIteratorModel
extends
	ReflectionObjectModel
implements
	TemplateListModel
{
	private boolean rewound;

	/**
	 * Creates a new model that wraps the specified iterator object.
	 * @param object the iterator object to wrap into a model.
	 */
	public ReflectionIteratorModel(Iterator iterator)
	{
		super(iterator);
		rewound = true;
	}
	
	/**
	 * Returns a model wrapping the specified iterator object. If there is already
	 * a cached model instance for this iterator, returns the cached model instance.
	 * Models are cached using {@link WeakReference} objects. The caching can be turned
	 * off by setting the <tt>expose.reflection.nocache</tt> system property to
	 * true. In this case calling this method is equivalent to constructing a new model.
	 * @param object the iterator to wrap into a model.
	 * @return the model for the iterator
	 */
	public static final ReflectionIteratorModel getInstance(Iterator object)
	{
		if(noCache)
			return new ReflectionIteratorModel(object);
		
		ReflectionIteratorModel model = (ReflectionIteratorModel)lookup(object);
		if(model == null)
		{
			model = new ReflectionIteratorModel(object);
			register(model, object);
		}
		return model;
	}

	/**
	 * Returns the type of this object (which is TYPE_ITERATOR)
	 */
	public int getType()
	{
		return TYPE_ITERATOR;
	}
	
	/**
	 * Calls {@link Iterator#hasNext()}.
	 */
	public boolean hasNext()
	{
		return ((Iterator)object).hasNext();
	}
	
	/**
	 * True if the iterator's <tt>next</tt> method has not been called since last
	 * rewind, or since the object creation.
	 */
	public boolean isRewound()
	{
		return rewound;
	}
	
	/**
	 * Calls {@link Iterator#next()}.
	 */
	public TemplateModel next()
	{
		rewound = false;
		return ReflectionUtilities.wrap(((Iterator)object).next());
	}
	
	/**
	 * If the iterator is a {@link ListIterator}, rewinds it to the first position.
	 * @throws TemplateModelException if the iterator is not a list iterator (and thus
	 * cannot be rewound).
	 */
	public void rewind()
	throws
		TemplateModelException
	{
		if(!rewound)
		{
			if(object instanceof ListIterator)
			{
				ListIterator listIterator = (ListIterator)object;
				while(listIterator.hasPrevious())
					listIterator.previous();
				rewound = true;
			}
			else
				throw new TemplateModelException("Cannot rewind an Iterator that is not a ListIterator.");
		}
	}

	/**
	 * Returns <tt>iterator == null || !iterator.hasNext()</tt>
	 */
	public boolean isEmpty()
	{
		return object == null || !((Iterator)object).hasNext();
	}

	/**
	 * Throws an UnsupportedOperationException as iterator size is unknown
	 */
	public TemplateModel listSize()
	{
		throw new UnsupportedOperationException();
	}
	
	/**
	 * Throws an UnsupportedOperationException as iterator elements can not
	 * be accessed at random.
	 */
	public TemplateModel get(int index)
	{
		throw new UnsupportedOperationException();
	}
}
