/*
 * ====================================================================
 *
 * Copyright (c) 2000 Attila Szegedi.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes the Expose library
 *        (http://www.szegedi.org/expose)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Expose" and "Attila Szegedi" must not be used to endorse 
 *    or promote products derived from this software without prior written
 *    permission. For written permission, please contact expose@szegedi.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR ITS CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 */

package org.szegedi.expose.model;

import freemarker.template.*;
import java.util.*;

/**
 * A class that adds {@link TemplateListModel} functionality to the
 * {@link Iterator} interface implementers.
 * Using the model as a list model is NOT thread-safe, as the enumerations are
 * inherently not thread-safe.
 * @author Attila Szegedi, attila@szegedi.org
 * @version 1.0
 */

public final class ReflectionEnumerationModel
extends
	ReflectionObjectModel
implements
	TemplateListModel
{
	private boolean rewound;

	/**
	 * Creates a new model that wraps the specified enumeration object.
	 * @param object the enumeration object to wrap into a model.
	 */
	public ReflectionEnumerationModel(Enumeration enumeration)
	{
		super(enumeration);
		rewound = true;
	}
	
	/**
	 * Returns a model wrapping the specified enumeration object. If there is already
	 * a cached model instance for this enumeration, returns the cached model instance.
	 * Models are cached using {@link WeakReference} objects. The caching can be turned
	 * off by setting the <tt>expose.reflection.nocache</tt> system property to
	 * true. In this case calling this method is equivalent to constructing a new model.
	 * @param object the enumeration to wrap into a model.
	 * @return the model for the enumeration
	 */
	public static final ReflectionEnumerationModel getInstance(Enumeration object)
	{
		if(noCache)
			return new ReflectionEnumerationModel(object);
		
		ReflectionEnumerationModel model = (ReflectionEnumerationModel)lookup(object);
		if(model == null)
		{
			model = new ReflectionEnumerationModel(object);
			register(model, object);
		}
		return model;
	}

	/**
	 * Returns the type of this object (which is TYPE_ENUMERATION)
	 */
	public int getType()
	{
		return TYPE_ENUMERATION;
	}
	
	/**
	 * Calls {@link Iterator#hasNext()}.
	 */
	public boolean hasNext()
	{
		return ((Enumeration)object).hasMoreElements();
	}
	
	/**
	 * True if the iterator's <tt>next</tt> method has not been called since last
	 * rewind, or since the object creation.
	 */
	public boolean isRewound()
	{
		return rewound;
	}
	
	/**
	 * Calls {@link Iterator#next()}.
	 */
	public TemplateModel next()
	{
		rewound = false;
		return ReflectionUtilities.wrap(((Enumeration)object).nextElement());
	}
	
	/**
	 * Throws an UnsupportedOperationException as enumeration cannot be rewound
	 */
	public void rewind()
	{
		throw new UnsupportedOperationException();
	}

	/**
	 * Throws an UnsupportedOperationException as enumeration size is unknown
	 */
	public TemplateModel listSize()
	{
		throw new UnsupportedOperationException();
	}
	
	/**
	 * Throws an UnsupportedOperationException as enumeration elements can not
	 * be accessed at random.
	 */
	public TemplateModel get(int index)
	{
		throw new UnsupportedOperationException();
	}
	
	/**
	 * Returns <tt>enumeration == null || !enumeration.hasMoreElements()</tt>
	 */
	public boolean isEmpty()
	{
		return object == null || !((Enumeration)object).hasMoreElements();
	}
}
