/*
 * ====================================================================
 *
 * Copyright (c) 2000 Attila Szegedi.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes the Expose library
 *        (http://www.szegedi.org/expose)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Expose" and "Attila Szegedi" must not be used to endorse 
 *    or promote products derived from this software without prior written
 *    permission. For written permission, please contact expose@szegedi.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR ITS CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 */

package org.szegedi.expose.model;

import freemarker.template.*;
import java.util.*;

/**
 * A special case of {@link ReflectionObjectModel} that adds implementation
 * for {@link TemplateListModel} on collection objects.
 * Using the model as a list model is thread-safe, as it maintains an iterator 
 * on a per-thread basis.
 * @author Attila Szegedi, attila@szegedi.org
 * @version 1.0
 */

public final class ReflectionCollectionModel
extends
	ReflectionObjectModel
implements
	TemplateListModel
{
	private ThreadLocal iterator = new ThreadLocal();
	
	/**
	 * Creates a new model that wraps the specified collection object.
	 * @param object the collection object to wrap into a model.
	 */
	public ReflectionCollectionModel(Collection collection)
	{
		super(collection);
	}
	
	/**
	 * Returns a model wrapping the specified collection object. If there is already
	 * a cached model instance for this collection, returns the cached model instance.
	 * Models are cached using {@link WeakReference} objects. The caching can be turned
	 * off by setting the <tt>expose.reflection.nocache</tt> system property to
	 * true. In this case calling this method is equivalent to constructing a new model.
	 * @param object the collection to wrap into a model.
	 * @return the model for the collection
	 */
	public static final ReflectionCollectionModel getInstance(Collection object)
	{
		if(noCache)
			return new ReflectionCollectionModel(object);
		
		ReflectionCollectionModel model = (ReflectionCollectionModel)lookup(object);
		if(model == null)
		{
			model = new ReflectionCollectionModel(object);
			register(model, object);
		}
		return model;
	}

	/**
	 * Returns the type of this object (which is TYPE_COLLECTION)
	 */
	public int getType()
	{
		return TYPE_COLLECTION;
	}
	
	/**
	 * Calls {@link Iterator#hasNext()}.
	 */
	public boolean hasNext()
	{
		boolean retval = ((Iterator)iterator.get()).hasNext();
		if(!retval)
			iterator.set(null);
		return retval;
	}
	
	/**
	 * Always return false to force calling rewind.
	 */
	public boolean isRewound()
	{
		return false;
	}
	
	/**
	 * Calls {@link Iterator#next()}.
	 */
	public TemplateModel next()
	{
		return ReflectionUtilities.wrap(((Iterator)iterator.get()).next());
	}
	
	/**
	 * Creates a new iterator on the collection that will be used to implement other
	 * {@link TemplateListModel} methods.
	 */
	public void rewind()
	{
		iterator.set(((Collection)object).iterator());
	}

	/**
	 * If the contained collection is a list, returns its element by position,
	 * wrapped into appropriate template model. Otherwise, an
	 * <tt>UnsupportedOperationException</tt> is thrown.
	 */
	public TemplateModel get(int index)
	{
		if(object instanceof List)
			return ReflectionUtilities.wrap(((List)object).get(index));
		
		throw new UnsupportedOperationException();
	}
	
	/**
	 * Returns a Simple scalar with a string containing the decimal 
	 * representation of the array size.
	 */
	public TemplateModel listSize()
	{
		return new SimpleScalar(Integer.toString(((Collection)object).size()));
	}
	
	/**
	 * Returns true if the underlying collection contains no elements.
	 */
	public boolean isEmpty()
	{
		return ((Collection)object).isEmpty();
	}
}
