/*
 * ====================================================================
 *
 * Copyright (c) 2000 Attila Szegedi.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes the Expose library
 *        (http://www.szegedi.org/expose)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Expose" and "Attila Szegedi" must not be used to endorse 
 *    or promote products derived from this software without prior written
 *    permission. For written permission, please contact expose@szegedi.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR ITS CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 */

package org.szegedi.expose.model;

import freemarker.template.*;

import java.lang.ref.*;
import java.lang.reflect.*;
import java.util.*;

/**
 * A class that will wrap an arbitrary array into
 * {@link TemplateHashModel}, {@link TemplateMethodModel}, and {@link TemplateListModel}
 * interfaces. The models are cached (meaning requesting a model for same object 
 * twice will return the same model instance) unless the system property <tt>expose.reflection.nocache</tt> 
 * is set to true. Except supporting retrieval through <tt>array.index</tt> syntax
 * and <tt>array("index")</tt> syntax, it also supports the <tt>array.length</tt>
 * syntax for retrieving the length of the array.
 * Using the model as a list model is thread-safe, as it maintains the list position 
 * on a per-thread basis.
 * @author Attila Szegedi, attila@szegedi.org
 * @version 1.0
 */
public final class ReflectionArrayModel
extends
	ReflectionObjectModelBase
implements
	TemplateHashModel,
	TemplateMethodModel,
	TemplateListModel
{
	// True if the array contains scalar values
	private boolean scalar;
	// Current list index for list operations. It is thread local which is important
	// in server environment.
	private ThreadLocal listIndex = new ThreadLocal()
	{
		protected Object initialValue() { return new MutableInteger(); }
	};
	
	/**
	 * Creates a new model that wraps the specified array object.
	 * @param object the array object to wrap into a model.
	 * @throws IllegalArgumentException if the passed object is not a Java array.
	 */
	public ReflectionArrayModel(Object array)
	{
		super(array);
		Class clazz = array.getClass();
		if(!clazz.isArray())
			throw new IllegalArgumentException("Object is not an array, it is " + array.getClass().getName());
		scalar = ReflectionUtilities.isScalar(clazz.getComponentType());
	}
	
	/**
	 * Returns a model wrapping the specified array object. If there is already
	 * a cached model instance for this array, returns the cached model instance.
	 * Models are cached using {@link WeakReference} objects. The caching can be turned
	 * off by setting the <tt>expose.reflection.nocache</tt> system property to
	 * true. In this case calling this method is equivalent to constructing a new model.
	 * @param object the array to wrap into a model.
	 * @throws IllegalArgumentException if the passed object is not a Java array.
	 * @return the model for the array
	 */
	public static final ReflectionArrayModel getInstance(Object object)
	{
		if(noCache) 
			return new ReflectionArrayModel(object);
		
		ReflectionObjectModelBase modelbase = lookup(object);
		if(modelbase instanceof ReflectionArrayModel)
			return (ReflectionArrayModel)modelbase;

		ReflectionArrayModel model = new ReflectionArrayModel(object);
		register(model, object);
		return model;
	}
	
	/**
	 * Returns the type of this object (which is TYPE_ARRAY)
	 */
	public int getType()
	{
		return TYPE_ARRAY;
	}
	
	/**
	 * If the key can be parsed by the {@link java.lang.Integer#parseInt(java.lang.String)} method into an
	 * array index, the array element at the parsed index is returned. If the key
	 * is named "length", the length of the array is returned.
	 */
	public TemplateModel get(String key)
	{
		try
		{
			return get(Integer.parseInt(key));
		}
		catch(NumberFormatException e)
		{
			if(key.equals("length"))
				return ReflectionUtilities.wrap(new Integer(Array.getLength(object)), ReflectionUtilities.WRAP_AS_SCALAR);
			else
				throw e;
		}
	}

	/**
	 * The first argument of the list is interpreted as an array index (it can be
	 * either a Number, or a String containing parseable integer). The element at 
	 * the specified index is returned.
	 */
	public TemplateModel exec(java.util.List arguments)
	throws
		TemplateModelException
	{
		Object index = arguments.get(0);
		if(index instanceof Number)
			return get(((Number)index).intValue());
		else 
			return get(Integer.parseInt(index.toString()));
	}

	/**
	 * Retrieves an array element by its index, wrapped into an appropriate
	 * template model.
	 */
	public TemplateModel get(int index)
	{
		return ReflectionUtilities.wrap(Array.get(object, index), scalar);
	}
	
	/**
	 * Returns true if the wrapped array is null, or its length is 0.
	 */
	public boolean isEmpty()
	{
		return object == null || Array.getLength(object) == 0;
	}

	/**
	 * True if the list pointer is not past the last element of the array
	 */
	public boolean hasNext()
	{
		return ((MutableInteger)listIndex.get()).i < Array.getLength(object);
	}
	
	/**
	 * True if the list pointer points to the first element of the array
	 */
	public boolean isRewound()
	{
		return ((MutableInteger)listIndex.get()).i == 0;
	}
	
	/**
	 * Returns the array element at the list pointer, advances list pointer by one
	 */
	public TemplateModel next()
	{
		return get(((MutableInteger)listIndex.get()).i++);
	}
	
	/**
	 * Rewinds the list pointer to the first element of the array
	 */
	public void rewind()
	{
		((MutableInteger)listIndex.get()).i = 0;
	}

	/**
	 * Returns a Simple scalar with a string containing the decimal 
	 * representation of the array size.
	 */
	public TemplateModel listSize()
	{
		return new SimpleScalar(Integer.toString(Array.getLength(object)));
	}
	
	/**
	 * Used as a thread-local list index
	 */
	private static class MutableInteger
	{
		int i = 0;
	}
}
