/*
 * ====================================================================
 *
 * Copyright (c) 2000 Attila Szegedi.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes the Expose library
 *        (http://www.szegedi.org/expose)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Expose" and "Attila Szegedi" must not be used to endorse 
 *    or promote products derived from this software without prior written
 *    permission. For written permission, please contact expose@szegedi.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR ITS CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 */

package org.szegedi.expose.model;

import freemarker.template.*;

/**
 * Provides singleton template models for representing boolean values.
 * @author Attila Szegedi, attila@szegedi.org
 * @version 1.0
 */

public final class BooleanScalar
implements
	TemplateScalarModel
{
	/**
	 * TemplateScalarModel for representing boolean true value
	 */
	public static final BooleanScalar TRUE = new BooleanScalar(true);
	/**
	 * TemplateScalarModel for representing boolean false value
	 */
	public static final BooleanScalar FALSE = new BooleanScalar(false);

	private Boolean value;
	
	private BooleanScalar(Boolean value)
	{
		this.value = value;
	}
	
	private BooleanScalar(boolean value)
	{
		this.value = value ? Boolean.TRUE : Boolean.FALSE;
	}
	
	/**
	 * Returns the boolean singleton TemplateScalarModel appropriate for 
	 * representing the specified boolean value.
	 */
	public static final BooleanScalar getScalar(boolean value)
	{
		return value ? TRUE : FALSE;
	}
	
	/**
	 * Returns the boolean singleton TemplateScalarModel appropriate for 
	 * representing the specified boolean value.
	 */
	public static final BooleanScalar getScalar(Boolean value)
	{
		return value.booleanValue() ? TRUE : FALSE;
	}

	/**
	 * Returns true if the passed object is the TRUE singleton.
	 */
	public static final boolean getBoolean(Object value)
	{
		return value == TRUE;
	}

	/**
	 * Returns {@link Boolean.TRUE} if the passed object is the TRUE singleton,
	 * otherwise returns {@link Boolean.FALSE}.
	 */
	public static final Boolean getBooleanObject(Object value)
	{
		return value == TRUE ? Boolean.TRUE : Boolean.FALSE;
	}

	/**
	 * "true" for the TRUE singleton, "false" for the FALSE singleton.
	 */
	public String getAsString()
	{
		return value.toString();
	}
	
	/**
	 * Returns true if this is the FALSE object. As the <tt>Expression.isTrue()</tt> returns true if 
	 * the model is </tt>!isEmpty()</tt>, this allows for usual semantics of 
	 * booleans (you can write <tt>&lt;if cond></tt> instead of 
	 * <tt>&lt;if cond == "true"></tt>).
	 */
	public boolean isEmpty()
	{
		return this == FALSE;
	}
}
