unit YearCal;

interface

uses Classes, Controls, Messages, WinTypes, Forms, Graphics, WinProcs, StdCtrls,
  Grids, SysUtils;

type
  TDayOfWeek = 0..6;

  TYearCal = class(TCustomGrid)
  private
    FDate: TDateTime;
    FOnChange: TNotifyEvent;
    FReadOnly: Boolean;
    FStartOfWeek: TDayOfWeek;
    FUpdating: Boolean;
    FUseCurrentDate: Boolean;
    function GetCellText(ACol, ARow: Integer): string;
    function GetDateElement(Index: Integer): Integer;
    function GetMonthOffset(Index: Word): Integer;
    procedure SetCalendarDate(Value: TDateTime);
    procedure SetDateElement(Index: Integer; Value: Integer);
    procedure SetStartOfWeek(Value: TDayOfWeek);
    function StoreCalendarDate: Boolean;
  protected
    procedure Change; dynamic;
    procedure Click; override;
    function DaysPerMonth(AYear, AMonth: Integer): Integer; virtual;
    procedure DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState); override;
    function IsLeapYear(AYear: Integer): Boolean; virtual;
    function SelectCell(ACol, ARow: Longint): Boolean; override;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
  public
    constructor Create(AOwner: TComponent); override;
    property CalendarDate: TDateTime  read FDate write SetCalendarDate stored StoreCalendarDate;
    property CellText[ACol, ARow: Integer]: string read GetCellText;
    procedure NextYear;
    procedure PrevYear;
    procedure UpdateCalendar; virtual;
  published
    property Align;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property Day: Integer index 3  read GetDateElement write SetDateElement stored False;
    property Enabled;
    property Font;
    property GridLineWidth;
    property Month: Integer index 2  read GetDateElement write SetDateElement stored False;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property ReadOnly: Boolean read FReadOnly write FReadOnly default False;
    property ShowHint;
    property StartOfWeek: TDayOfWeek read FStartOfWeek write SetStartOfWeek;
    property TabOrder;
    property TabStop;
    property UseCurrentDate: Boolean read FUseCurrentDate write FUseCurrentDate default True;
    property Visible;
    property Year: Integer index 1  read GetDateElement write SetDateElement stored False;
    property OnClick;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
  end;

procedure Register;

implementation

constructor TYearCal.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  { defaults }
  FUseCurrentDate := True;
  FixedCols := 1;
  FixedRows := 1;
  ColCount := 38;
  RowCount := 13;
  ScrollBars := ssNone;
  Options := Options - [goRangeSelect] + [goDrawFocusSelected];
  FDate := Date;
  UpdateCalendar;
end;

procedure TYearCal.Change;
begin
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TYearCal.Click;
var
  TheCellText: string;
begin
  inherited Click;
  TheCellText := CellText[Col, Row];
  if (row > 0) and (col > 0) and (TheCellText <> '') then
  begin
       CalendarDate := EnCodeDate(Year,Row,StrToInt(TheCellText));
  end;
end;

function TYearCal.IsLeapYear(AYear: Integer): Boolean;
begin
  Result := (AYear mod 4 = 0) and ((AYear mod 100 <> 0) or (AYear mod 400 = 0));
end;

function TYearCal.DaysPerMonth(AYear, AMonth: Integer): Integer;
const
  DaysInMonth: array[1..12] of Integer = (31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);
begin
  Result := DaysInMonth[AMonth];
  if (AMonth = 2) and IsLeapYear(AYear) then Inc(Result); { leap-year Feb is special }
end;

procedure TYearCal.DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
var
  TheText: string;
begin
  TheText := CellText[ACol, ARow];
  with ARect, Canvas do
    TextRect(ARect, Left + (Right - Left - TextWidth(TheText)) div 2,
      Top + (Bottom - Top - TextHeight(TheText)) div 2, TheText);
end;

function TYearCal.GetMonthOffset(Index: Word): integer;
begin
     Result := DayOfWeek(EncodeDate(Year,Index,1)) - (StartOfWeek+1);
     if Result < 0 then Result := 7 + Result;
end;

function TYearCal.GetCellText(ACol, ARow: Integer): string;
var
  DayNum: Integer;
begin
  Result := '';
  if (ARow = 0) or (ACol = 0) then  { day names at tops of columns }
  begin
    Result := IntToStr(Year);
    if aCol > 0 then
       Result := copy(ShortDayNames[(StartOfWeek + ACol - 1) mod 7 + 1],1,1);
    if aRow > 0 then
       Result := ShortMonthNames[ARow];
  end
  else
  begin
    DayNum := ACol - GetMonthOffset(aRow);
    if (DayNum < 1) or (DayNum > DaysPerMonth(Year, aRow)) then Result := ''
    else Result := IntToStr(DayNum);
  end;
end;

function TYearCal.SelectCell(ACol, ARow: Longint): Boolean;
begin
  if ((not FUpdating) and FReadOnly) or (CellText[ACol, ARow] = '') or
     (ACol = 0) or (ARow = 0) then
    Result := False
  else Result := inherited SelectCell(ACol, ARow);
end;

procedure TYearCal.SetCalendarDate(Value: TDateTime);
var
  AYear, AMonth, ADay, OldYear: Word;
begin
  DecodeDate(FDate, OldYear, AMonth, ADay);
  FDate := Value;
  DecodeDate(FDate, AYear, AMonth, ADay);
  if AYear <> OldYear then Invalidate;
  UpdateCalendar;
  Change;
end;

function TYearCal.StoreCalendarDate: Boolean;
begin
  Result := not FUseCurrentDate;
end;

function TYearCal.GetDateElement(Index: Integer): Integer;
var
  AYear, AMonth, ADay: Word;
begin
  DecodeDate(FDate, AYear, AMonth, ADay);
  case Index of
    1: Result := AYear;
    2: Result := AMonth;
    3: Result := ADay;
    else Result := -1;
  end;
end;

procedure TYearCal.SetDateElement(Index: Integer; Value: Integer);
var
  AYear, AMonth, ADay: Word;
begin
  if Value > 0 then
  begin
    DecodeDate(FDate, AYear, AMonth, ADay);
    case Index of
      1: if AYear <> Value then
         begin
              AYear := Value;
              Invalidate;
         end
         else Exit;
      2: if (Value <= 12) and (Value <> AMonth) then AMonth := Value else Exit;
      3: if (Value <= DaysPerMonth(AYear, AMonth)) and (Value <> ADay) then ADay := Value else Exit;
      else Exit;
    end;
    FDate := EncodeDate(AYear, AMonth, ADay);
    FUseCurrentDate := False;
    UpdateCalendar;
    Change;
  end;
end;

procedure TYearCal.SetStartOfWeek(Value: TDayOfWeek);
begin
  if Value <> FStartOfWeek then
  begin
    FStartOfWeek := Value;
    Invalidate;
    UpdateCalendar;
  end;
end;

procedure TYearCal.NextYear;
begin
  if IsLeapYear(Year+1) and (Month = 2) and (Day = 29) then Day := 28;
  Year := Year + 1;
  Invalidate;
end;

procedure TYearCal.PrevYear;
begin
  if IsLeapYear(Year-1) and (Month = 2) and (Day = 29) then Day := 28;
  Year := Year - 1;
  Invalidate;
end;

procedure TYearCal.UpdateCalendar;
var
  AYear, AMonth, ADay: Word;
  FirstDate: TDateTime;
begin
  FUpdating := True;
  try
    DecodeDate(FDate, AYear, AMonth, ADay);
    MoveColRow((ADay + GetMonthOffset(aMonth)), AMonth,False, False);
  finally
    FUpdating := False;
  end;
end;

procedure TYearCal.WMSize(var Message: TWMSize);
var
   aMonth : Word;
   sText  : string;
   iWidth : integer;
begin
  iWidth := Canvas.TextWidth(IntToStr(Year));
  for aMonth := 1 to 12 do
  begin
       sText := ShortMonthNames[aMonth];
       if Canvas.TextWidth(sText) > iWidth then
          iWidth := Canvas.TextWidth(sText);
  end;
  DefaultColWidth := ((Message.Width-iWidth) - 37 * GridLineWidth) div 37;
  DefaultRowHeight := (Message.Height - 12 * GridLineWidth) div 13;
  ColWidths[0] := Message.Width - (DefaultColWidth+GridLineWidth) * 37;
  RowHeights[0] := Message.Height - (DefaultRowHeight+GridLineWidth) * 12;
end;

procedure Register;
begin
  RegisterComponents('Mytools', [TYearCal]);
end;

end.