unit ParaCode;

{ Description:  Parallel port component with direct hardware access
{ Author:       Richard Shotbolt (100327,2305@compuserve.com)
{ Date:         23 Apr 1996

{ Opens a LPT port in a Win3.1/95-friendly way, then uses brute force
{ to access the port. Not tested under NT, in all probability it would fail as
{ a VxD should be used here.

{ All True/False properties refer to positive logic on the port pins i.e. True
{ always represents +5V, and False means 0V.

{ Source included for 16 and 32-bit versions }

interface

uses
   SysUtils, WinTypes, WinProcs, Classes, DsgnIntf;

const
	PORTNAME: array [0..3] of string = ('None', 'LPT1', 'LPT2', 'LPT3');
   STB = 1;
   AUTOFD = 2;
   INITL = 4;
   SLCI = 8;
   IRQEN = 16;
   ERR = 8;
   SLCO = 16;
   PE = 32;
   ACK = 64;
   BSY = 128;
{$IFDEF VER80}
	INVALID_HANDLE_VALUE = 0;
{$ENDIF}

type
   TNotifyEvent = procedure(Sender: TObject) of object;
   TPortNumber = (None, LPT1, LPT2, LPT3);

{ ************************************************************************* }
   TParallelPort = class(TComponent)
   private
   	FPort: TPortNumber;
       FPortAddress: word;
   	PortHandle: THandle;
       FDummyBool: Boolean;
       FDummyByte: Byte;
       FDummyWord: Word;
       FData: byte;
       FControl: byte;
       FStatus: byte;
       FStrobe: boolean;
       FAutoFd: boolean;
       FInit: boolean;
       FSlctIn: boolean;
       FError: boolean;
       FSlct: boolean;
       FPaperEnd: boolean;
       FAcknlg: boolean;
       FBusy: boolean;
   	procedure SetPort(Value: TPortNumber);
       function GetData: byte;
       procedure SetData(Value: byte);
       function GetControl: byte;
       procedure SetControl(Value: byte);
       function GetStatus: byte;
       function GetStrobe: boolean;
       procedure SetStrobe(Value: boolean);
       function GetAutoFd: boolean;
       procedure SetAutoFd(Value: boolean);
       function GetInit: boolean;
       procedure SetInit(Value: boolean);
       function GetSlctIn: boolean;
       procedure SetSlctIn(Value: boolean);
       function GetError: boolean;
       function GetSlct: boolean;
       function GetPaperEnd: boolean;
       function GetAcknlg: boolean;
       function GetBusy: boolean;
   protected
   public
       constructor Create(AOwner: TComponent); override;
       destructor Destroy; override;
       {methods}
       function OpenPort(PortNo: Byte): boolean;
       function ClosePort: boolean;
       {properties}
       property Control: byte read GetControl write SetControl;
       {individual line control}
   published
   	property Port: TPortNumber read FPort write SetPort;
       property BaseAddress: word read FPortAddress write FDummyWord;
       property Status: byte read GetStatus write FDummyByte;
   	property Data: byte read GetData write SetData;
       property Strobe: boolean read GetStrobe write SetStrobe;
       property AutoFeed: boolean read GetAutofd write SetAutofd;
       property Initialize: boolean read GetInit write SetInit;
       property SlctIn: boolean read GetSlctIn write SetSlctIn;
       property Error: boolean read GetError write FDummyBool;
       property Slct: boolean read GetSlct write FDummyBool;
       property PaperEnd: boolean read GetPaperEnd write FDummyBool;
       property Acknlg: boolean read GetAcknlg write FDummyBool;
       property Busy: boolean read GetBusy write FDummyBool;
   end;

procedure Register;

implementation



{ ************************************************************************* }
{ Get a byte from the port }
function InPort(PortAddr: word): byte;
{$IFDEF VER90}
assembler; stdcall;
asm
	mov dx,PortAddr
	in al,dx
end;
{$ELSE}
begin
Result := Port[PortAddr];
end;
{$ENDIF}

{ ************************************************************************* }
{ Write a byte to the port }
procedure OutPort(PortAddr: word; Databyte: byte);
{$IFDEF VER90}
assembler; stdcall;
asm
   mov al,Databyte
   mov dx,PortAddr
   out dx,al
end;
{$ELSE}
begin
Port[PortAddr] := DataByte;
end;
{$ENDIF}

{ ************************************************************************* }
{ Given a value of 1..3, get port base address from BIOS table }
function GetPortAddress(PortNo: integer): word;
{$IFDEF VER90}
assembler; stdcall;
asm
	push es
	mov ebx, PortNo
	shl ebx,1
	mov eax,40h
	mov es,ax
	mov ax,es:[ebx+6]
	pop es
end;
{$ELSE}
begin
Result := Word(Ptr($0040,($0008 + 2*(PortNo - 1)))^);
end;
{$ENDIF}

{ ************************************************************************* }
{ Do (Action) to b(Bit) of (PortAddr) }
procedure SetBitState(PortAddr: Word; Action: Boolean; Bit: Byte);
begin
if Action = True then
	OutPort(PortAddr, InPort(PortAddr) or Bit)
else
	OutPort(PortAddr, InPort(PortAddr) and not Bit);
end;

{ ************************************************************************* }
{ Return status of b(Bit) of (PortAddr) }
function GetBitState(PortAddr: Word; Bit: Byte): boolean;
begin
if (InPort(PortAddr) and Bit) <> 0 then
	Result := True
else
	Result := False;
end;

{----------  TParallelPort ----------}

{ ************************************************************************* }
constructor TParallelPort.Create(AOwner:TComponent);
begin
inherited Create(AOwner);
FPort := None;
FPortAddress := 0;
PortHandle := INVALID_HANDLE_VALUE;
end;

{ ************************************************************************* }
destructor TParallelPort.Destroy;
begin
ClosePort;
inherited Destroy;
end;

{ ************************************************************************* }
function TParallelPort.OpenPort(PortNo: Byte): boolean;
{ Open LPT 1..3
{ Return True if successful }
var
	PH: THandle;
   i: integer;
   s: string;
   pn: string;
begin
i := 0;
s := '';
pn := '';
Result := False;
{ If a port is already open, close it }
ClosePort;
{ Port number must be 1..3 }
if (PortNo < 1) or (PortNo > 3) then
	Exit;
{ OK, so open LPT(PortNo) }
{$IFDEF VER90}
PH := CreateFile(
   PChar('\\.\LPT' + IntToStr(PortNo)),
   GENERIC_READ or GENERIC_WRITE,
   0,
   nil,
   OPEN_EXISTING,
   0,
   0
   );
{$ELSE}
pn := '123';
s := 'LPT' + pn[PortNo];
s[ord(s[0]) + 1] := #0;
i := OpenComm(@s[1],0,0);
if (i < 0) then
	PH := INVALID_HANDLE_VALUE
else
   PH := i;
{$ENDIF}
{ Validity checks }
if PH <> INVALID_HANDLE_VALUE then
	begin
   case PortNo of
   	1: FPort := LPT1;
       2: FPort := LPT2;
       3: FPort := LPT3;
   	end;
   PortHandle := PH;
   { Port base address valid if handle returned }
   FPortAddress := GetPortAddress(PortNo);
   { Output data = 0 }
   OutPort(FPortAddress, 0);
   { Control reg = 0c }
   OutPort(FPortAddress + 2, (InPort(FPortAddress + 2) and $F0) or $0C);
   Result := True;
   end;
end;

{ ************************************************************************* }
procedure TParallelPort.SetPort(Value: TPortNumber);
begin
if Value <> FPort then
	begin
	{ Set default output data }
	OpenPort(Ord(Value));
	{ Dont leave port open at design time }
	if (csDesigning in ComponentState) and (PortHandle <> INVALID_HANDLE_VALUE) then
       {$IFDEF VER90}
		CloseHandle(PortHandle);
       {$ELSE}
       CloseComm(PortHandle);
       {$ENDIF}
   end;
end;

{ ************************************************************************* }
function TParallelPort.GetData: byte;
begin
{ Send a byte to the data port }
if PortHandle <> INVALID_HANDLE_VALUE then
   FData := InPort(FPortAddress)
else
   FData := 0;
Result := FData;
end;

{ ************************************************************************* }
procedure TParallelPort.SetData(Value: byte);
begin
{ Send a byte to the data port }
if PortHandle <> INVALID_HANDLE_VALUE then
	OutPort(FPortAddress, Value);
end;

{ ************************************************************************* }
function TParallelPort.GetControl: byte;
begin
{ Send a byte to the data port }
if PortHandle <> INVALID_HANDLE_VALUE then
   FControl := InPort(FPortAddress + 2)
else
   FControl := 0;
Result := FControl;
end;

{ ************************************************************************* }
procedure TParallelPort.SetControl(Value: byte);
begin
{ Send a byte to the data port }
if PortHandle <> INVALID_HANDLE_VALUE then
	OutPort(FPortAddress + 2, (InPort(FPortAddress + 2) and $f0) or (Value and $0f));
end;

{ ************************************************************************* }
function TParallelPort.GetStatus: byte;
begin
{ Read port status, inverting B7 (busy) }
if PortHandle <> INVALID_HANDLE_VALUE then
	FStatus := InPort(FPortAddress + 1)
else
   FStatus := 0;
Result := FStatus;
end;

{ ************************************************************************* }
function TParallelPort.GetStrobe: boolean;
begin
{ Read the STROBE output level }
if PortHandle <> INVALID_HANDLE_VALUE then
   FStrobe := not GetBitState(FPortAddress + 2, STB)
else
   FStrobe := False;
Result := FStrobe;
end;

{ ************************************************************************* }
procedure TParallelPort.SetStrobe(Value: boolean);
begin
{ Condition the STROBE output }
if PortHandle <> INVALID_HANDLE_VALUE then
   SetBitState(FPortAddress + 2, not Value, STB);
end;

{ ************************************************************************* }
function TParallelPort.GetAutoFd: boolean;
begin
{ Read the AUTOFD output level }
if PortHandle <> INVALID_HANDLE_VALUE then
   FAutoFd := not GetBitState(FPortAddress + 2, AUTOFD)
else
   FAutoFd := False;
Result := FAutoFd;
end;

{ ************************************************************************* }
procedure TParallelPort.SetAutoFd(Value: boolean);
begin
{ Condition the STROBE output }
if PortHandle <> INVALID_HANDLE_VALUE then
   SetBitState(FPortAddress + 2, not Value, AUTOFD);
end;

{ ************************************************************************* }
function TParallelPort.GetInit: boolean;
begin
{ Read the INITIALIZE output level }
if PortHandle <> INVALID_HANDLE_VALUE then
   FInit := GetBitState(FPortAddress + 2, INITL)
else
   FInit := False;
Result := FInit;
end;

{ ************************************************************************* }
procedure TParallelPort.SetInit(Value: boolean);
begin
{ Condition the INITIALIZE output }
if PortHandle <> INVALID_HANDLE_VALUE then
   SetBitState(FPortAddress + 2, Value, INITL);
end;

{ ************************************************************************* }
function TParallelPort.GetSlctIn: boolean;
begin
{ Read the SLCTIN output level }
if PortHandle <> INVALID_HANDLE_VALUE then
   FSlctIn := not GetBitState(FPortAddress + 2, SLCI)
else
   FSlctIn := False;
Result := FSlctIn;
end;

{ ************************************************************************* }
procedure TParallelPort.SetSlctIn(Value: boolean);
begin
{ Condition the SLCTIN output }
if PortHandle <> INVALID_HANDLE_VALUE then
   SetBitState(FPortAddress + 2, not Value, SLCI);
end;

{ ************************************************************************* }
function TParallelPort.GetError: boolean;
begin
{ Read the ERROR input level }
if PortHandle <> INVALID_HANDLE_VALUE then
   FError := GetBitState(FPortAddress + 1, ERR)
else
   FError := False;
Result := FError;
end;
{ ************************************************************************* }
function TParallelPort.GetSlct: boolean;
begin
{ Read the SLCT input level }
if PortHandle <> INVALID_HANDLE_VALUE then
   FSlct := GetBitState(FPortAddress + 1, SLCO)
else
   FSlct := False;
Result := FSlct;
end;
{ ************************************************************************* }
function TParallelPort.GetPaperEnd: boolean;
begin
{ Read the PE input level }
if PortHandle <> INVALID_HANDLE_VALUE then
   FPaperEnd := GetBitState(FPortAddress + 1, PE)
else
   FPaperEnd := False;
Result := FPaperEnd;
end;
{ ************************************************************************* }
function TParallelPort.GetAcknlg: boolean;
begin
{ Read the ACK input level }
if PortHandle <> INVALID_HANDLE_VALUE then
   FAcknlg := GetBitState(FPortAddress + 1, ACK)
else
   FAcknlg := False;
Result := FAcknlg;
end;
{ ************************************************************************* }
function TParallelPort.GetBusy: boolean;
begin
{ Read the inverted BUSY input level }
if PortHandle <> INVALID_HANDLE_VALUE then
   FBusy := not GetBitState(FPortAddress + 1, BSY)
else
   FBusy := False;
Result := FBusy;
end;

{ ************************************************************************* }
function TParallelPort.ClosePort: boolean;
begin
{ Close currently open LPT }
{ Returns True if successful }
if PortHandle <> INVALID_HANDLE_VALUE then
	begin
   { Output data = 0 }
   OutPort(FPortAddress, 0);
   { Control reg b0..3 = $0C }
   OutPort(FPortAddress + 2, (InPort(FPortAddress + 2) and $F0) or $0C);
   {$IFDEF VER90}
	Result := CloseHandle(PortHandle);
   {$ELSE}
   if CloseComm(PortHandle) = 0 then
       Result := True
   else
       Result := False;
   {$ENDIF}
   end
else
	Result := False;
PortHandle := INVALID_HANDLE_VALUE;
FPort := None;
FPortAddress := 0;
end;

{ ************************************************************************* }
procedure Register;
begin
  RegisterComponents('Samples', [TParallelPort]);
end;

end.
