unit Fpalex1;
{*************************************************************}
{          =Delphi Palatte Management Example #1=             }
{                                                             }
{              written by:   Brian A. Makuch                  }
{                    date:   8/31/95                          }
{                                                             }
{  This example is Freeware and it, and/or any of its parts,  }
{  may be used in any manner you see fit.                     }
{                                                             }
{  See the readme.txt file for more information.              }
{*************************************************************}
interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, StdCtrls, Buttons, Menus;

type
  TPalForm = class(TForm)
    MainMenu1: TMainMenu;
    File1: TMenuItem;
    Exit1: TMenuItem;
    N1: TMenuItem;
    SaveAs1: TMenuItem;
    Open1: TMenuItem;
    OpenDlg: TOpenDialog;
    SaveDlg: TSaveDialog;
    procedure FormPaint(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure Exit1Click(Sender: TObject);
    procedure Open1Click(Sender: TObject);
    procedure SaveAs1Click(Sender: TObject);
  private
    { Private declarations }
    procedure GetSystemPalette;
    procedure wmQueryNewPalette(var Msg : TWMQueryNewPalette); message WM_QueryNewPalette;
    procedure wmPaletteChanged(var Msg : TWMPaletteChanged); message WM_PaletteChanged;
  public
    { Public declarations }
    procedure PaletteToFile(Filename: string);
    procedure FileToPalette(Filename: string);
  end;

var
  PalForm: TPalForm;

  MyPal   : hPalette;
  OldPal  : hPalette;
  tmpPal  : pLogPalette;
  PalSize : Longint;

implementation

{$R *.DFM}

procedure TPalForm.wmQueryNewPalette(var Msg : TWMQueryNewPalette);
var
  rClrs  : Word;
  DC     : hDC;
  OldPal : hPalette;
begin
  DC     := PalForm.Canvas.Handle;
  OldPal := SelectPalette(DC, MyPal, False);
  rClrs  := RealizePalette(DC);
  if rClrs > 0 then InvalidateRect(Handle, nil, False);
  Msg.Result := rClrs;
end;

procedure TPalForm.wmPaletteChanged(var Msg : TWMPaletteChanged);
var
  rClrs : Word;
  DC    : hDC;
  OldP  : hPalette;
begin
  if (Msg.PalChg = Handle) then
    Msg.Result := 0
  else begin
    DC    := PalForm.Canvas.Handle;
    OldP  := SelectPalette(DC, MyPal, True);
    rClrs := RealizePalette(DC);
    UpdateColors(DC);
    Msg.Result := 0;
  end;
end;

procedure TPalForm.FormResize(Sender: TObject);
begin
  FormPaint(Sender);
end;

procedure TPalForm.FormCreate(Sender: TObject);
begin
  PalSize := sizeof(TLogPalette) + 256 * sizeof(TPaletteEntry);
  tmpPal  := MemAlloc(PalSize);
  GetSystemPalette;
  OldPal := SelectPalette(Canvas.Handle, MyPal, FALSE);
  RealizePalette(Canvas.Handle);
  FormPaint(sender);
end;

procedure TPalForm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  DeleteObject(MyPal);
  FreeMem(tmpPal, PalSize);
end;

procedure TPalForm.GetSystemPalette;
var
  x       : Integer;
  count   : Integer;
begin
  {This function retrieves a copy of the system palette.  If you wanted to use
   a custom palette, you could modify the commented section to load the array.}

  tmpPal^.palVersion    := $300;
  tmpPal^.palNumEntries := 256;
  {Get a copy of the system palette}
  GetSystemPaletteEntries(GetDC(GetDesktopWindow), 0, 256, tmpPal^.palPalEntry);

  (*
  {This code is used to read the palette entries from the CustomPal constant     }
  {file that this program outputs. You could save all 256 entries and replace    }
  {the entire palette, but your program will be more Windows friendly if you get }
  {the first and last 10 entries of the system palette and only replace the      }
  {middle 236 colors.                                                            }
  {$R-}   {Turn off range checking to prevent a runtime error when using dynamic arrays }
  count := 0;
  {replace all but the system colors}
  for x := 10 to 235 do
  begin
    tmpPal^.palPalEntry[x].peRed   := CustomPal[count];
    tmpPal^.palPalEntry[x].peGreen := CustomPal[count + 1];
    tmpPal^.palPalEntry[x].peBlue  := CustomPal[count + 2];
    tmpPal^.palPalEntry[x].peFlags := PC_NOCOLLAPSE;
    Inc(Count, 3);
  end;
  {$R+}   {Turn range checking back on}
  *)

  MyPal := CreatePalette(tmpPal^);
end;

procedure TPalForm.FormPaint(Sender: TObject);
var
  Row        : Integer;
  Column     : Integer;
  BoxHeight  : Integer;
  BoxWidth   : Integer;
  ColorIndex : Longint;
begin
  {Display the palette as a grid of boxes [16 rows * 16 columns = 256]}
  BoxWidth  := ClientWidth div 16;
  BoxHeight := ClientHeight div 16;

  {Place our palette into the device context of the form}
  OldPal    := SelectPalette(Canvas.Handle, MyPal, FALSE);
  {Realize our palette to map the colors}
  RealizePalette(Canvas.Handle);
  {Now we can paint!}

  {Step through the palette using the API macro PaletteIndex() }
  For ColorIndex := 0 To 255 do begin;
    Row    := ColorIndex DIV 16 + 1;
    Column := ColorIndex Mod 16 + 1;
    Canvas.Brush.Color :=  PaletteIndex(ColorIndex);
    Canvas.Rectangle((Column - 1) * BoxWidth, (Row - 1) * BoxHeight, Column * BoxWidth, Row * BoxHeight);
  end;

  {Paint around the edges - seems faster than filling the whole client area}
  Canvas.Brush.Color := clBlack;
  Canvas.Rectangle(0, 16 * BoxHeight, Width, Height);
  Canvas.Rectangle(16 * BoxWidth, 0, Width, Height);
end;

procedure TPalForm.Exit1Click(Sender: TObject);
begin
  Close;
end;

procedure TPalForm.Open1Click(Sender: TObject);
begin
  {Show the TOpenDialog, and if the user presses the Ok button, load the palette}
  If OpenDlg.Execute then begin
    Cursor := crHourglass;
    FileToPalette(OpenDlg.Filename);
    Cursor := crDefault;
  end;
end;

procedure TPalForm.SaveAs1Click(Sender: TObject);
begin
  {Write out palette to a file}
  If SaveDlg.Execute then begin
    Cursor := crHourglass;
    PaletteToFile(SaveDlg.Filename);
    Cursor := crDefault;
  end;
end;

procedure TPalForm.PaletteToFile(Filename: string);
var
  Outfile   : TextFile;
  LineCnt   : Integer;
  PalVer    : Integer;
  Buffer    : String;
  Extension : string;
  nColors   : Integer;
  R,G,B     : Byte;
  wResult   : word;
begin
  {This procedure writes the current palette to an include file which }
  {declares one constant, "CustomPal". I've also left in the code to  }
  {output PaintShop Pro format PAL files, should you want to expand   }
  {this into a palette editor                                         }

  If Filename = '' then exit;

  If FileExists(Filename) then begin
    wresult := MessageBox(Handle, 'The Specified file exists.'+#13#10+
       'Do you want to overwrite the file?', 'File Exists', mb_yesno+mb_iconquestion);
    If wresult <> idYes then exit;
  end;

  AssignFile(Outfile, Filename);
  Rewrite(Outfile);
  Writeln(Outfile, 'const');
  writeln(Outfile);
  writeln(Outfile, '  CustomPal : array[0..(235*3)-1] of byte = (');
  For LineCnt := 10 to 235 do begin
    {$R-}
    If LineCnt <> 235 then begin
      Buffer := '            ' +
        IntToStr(tmpPal^.palPalEntry[LineCnt].peRed)   + ', ' +
        IntToStr(tmpPal^.palPalEntry[LineCnt].peGreen) + ', ' +
        IntToStr(tmpPal^.palPalEntry[LineCnt].peBlue)  + ', ';
    end else begin
      Buffer := '            ' +
        IntToStr(tmpPal^.palPalEntry[LineCnt].peRed)   + ', ' +
        IntToStr(tmpPal^.palPalEntry[LineCnt].peGreen) + ', ' +
        IntToStr(tmpPal^.palPalEntry[LineCnt].peBlue)  + ');';
    end;
    {$R+}
    System.WriteLn(Outfile, Buffer);
  end;

  CloseFile(Outfile);

  {Let the user know we're done}
  MessageBox(Handle, 'File saved successfully.', 'Success', mb_ok+mb_IconInformation);

  (* =======================================================
    {//Output a PaintShot Pro palette file//}
    AssignFile(Outfile, Filename);
    Rewrite(Outfile);
    System.WriteLn(Outfile, 'JASC-PAL'); {Header          }
    System.WriteLn(Outfile, '0100');     {Version         }
    System.WriteLn(Outfile, '256');      {Number of colors}
    For LineCnt := 0 to 255 do begin
      {$R-}
      System.WriteLn(OutFile,
              tmpPal^.palPalEntry[LineCnt].peRed,   ' ',
              tmpPal^.palPalEntry[LineCnt].peGreen, ' ',
              tmpPal^.palPalEntry[LineCnt].peBlue);
      {$R+}
    end;
    CloseFile(Outfile);
  ========================================================*)
end;

procedure TPalForm.FileToPalette(Filename: string);
var
  Infile : TextFile;
  LineCnt: Integer;
  PalVer : string;
  Buffer : String;
  nColors: Integer;
  R,G,B  : Byte;
begin
  If Filename = '' then exit;
  AssignFile(Infile, Filename);
  Reset(Infile);

  System.Readln(Infile, Buffer); {Header}
  If (Buffer = 'JASC-PAL') then begin   {Is this a PaintShop Pro palette file?}
    {check version info}
    System.Readln(Infile, Buffer); {Version}
    If (Pos('0100', Buffer) = 0) then begin
      MessageBox(Handle, 'Unsupported palette file format.', 'Error', mb_ok+mb_iconstop);
      CloseFile(Infile);
      exit;
    end;
    System.Readln(Infile, nColors); {Number of colors}
    If nColors <> 256 then begin
      MessageBox(Handle, 'Only 256 color palettes are supported.', 'Error', mb_ok+mb_iconstop);
      CloseFile(Infile);
      exit;
    end;
    LineCnt := 0;
    while not Eof(Infile) do begin
      If LineCnt > 255 then break;
      System.ReadLn(Infile, r, g, b);
      {$R-}
      tmpPal^.palPalEntry[LineCnt].peRed   := R;
      tmpPal^.palPalEntry[LineCnt].peGreen := G;
      tmpPal^.palPalEntry[LineCnt].peBlue  := B;
      tmpPal^.palPalEntry[LineCnt].peFlags := PC_NOCOLLAPSE;
      {$R+}
      Inc(LineCnt);
    end;

  end else if Pos('NeoPaint', Buffer) <> 0 then begin  {Or is it a NeoPaint palette file.}
    System.Readln(Infile, Buffer);   {Copyright}
    If Pos('NeoSoft', Buffer) = 0 then begin
      MessageBox(Handle, 'Unsupported palette file format.', 'Error', mb_ok+mb_iconstop);
      CloseFile(Infile);
      exit;
    end;
    System.Readln(Infile, nColors);   {Number of colors}
    If nColors <> 256 then begin
      MessageBox(Handle, 'Only 256 color palettes are supported.', 'Error', mb_ok+mb_iconstop);
      CloseFile(Infile);
      exit;
    end;
    LineCnt := 0;
    while not Eof(Infile) do begin
      If LineCnt > 255 then break;
      System.ReadLn(Infile, r, g, b);
      {$R-}
      {NeoPaint RGB values range from 0 to 63}
      tmpPal^.palPalEntry[LineCnt].peRed   := (R*4)+1;
      tmpPal^.palPalEntry[LineCnt].peGreen := (G*4)+1;
      tmpPal^.palPalEntry[LineCnt].peBlue  := (B*4)+1;
      tmpPal^.palPalEntry[LineCnt].peFlags := PC_NOCOLLAPSE;
      {$R+}
      Inc(LineCnt);
    end;

  end else begin  {..We didn't recognize the first line of the file}
    MessageBox(Handle, 'Only PSP and NeoPaint palette files are supported.', 'Error', mb_ok+mb_iconstop);
    CloseFile(Infile);
    exit;
  end;

  CloseFile(Infile);
  MyPal := CreatePalette(tmpPal^);
  InvalidateRect(Handle, nil, false);  {Have the form repaint, by "dirtying" it}
end;

end.

