unit MySocket;
{(C) Alex Demchenko(alex@ritlabs.com)}
{$DEFINE USE_FORMS} //If you don't use forms unit remove this line

interface
uses
  Windows, Messages, WinSock, {$IFDEF USE_FORMS}Forms, {$ENDIF} Classes;

function InitMySocket(var WSA: TWSAData): LongWord;
procedure FinalMySocket;

type
  {$IFNDEF USE_FORMS}
  TWndMethod = procedure(var Message: TMessage) of object;
  {$ENDIF}

  TOnRecv = procedure(Sender: TObject; Socket: TSocket; Buffer: Pointer; BufLen: LongWord) of object;
  TOnPktParse = procedure(Sender: TObject; Buffer: Pointer; BufLen: LongWord) of object;

  TConnectTrd = class;

  TClSock = class(TObject)
  private
    FConnTrd: TConnectTrd;
    FIp: String;
    FDestPort: LongWord;
    FClSock: TSocket;
    FWndHandle: THandle;
    FOnRecv: TOnRecv;
    FOnDisconnect: TNotifyEvent;
    FOnConnect: TNotifyEvent;
    FOnConnectError: TNotifyEvent;
    FOnPktParse: TOnPktParse;
    FHostIp: array[0..MAXGETHOSTSTRUCT - 1] of Char;
    procedure OnSockMsg(var Msg: TMessage);
    function IsConnected: Boolean;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Connect;
    procedure Disconnect;
    procedure SendData(var Buf; BufLen: LongWord);
    property IP: String read FIp write FIp;
    property DestPort: LongWord read FDestPort write FDestPort;
    property Connected: Boolean read IsConnected;
  published
    property OnRecieve: TOnRecv read FOnRecv write FOnRecv;
    property OnDisconnect: TNotifyEvent read FOnDisconnect write FOnDisconnect;
    property OnConnect: TNotifyEvent read FOnConnect write FOnConnect;
    property OnConnectError: TNotifyEvent read FOnConnectError write FOnConnectError;
    property OnPktParse: TOnPktParse read FOnPktParse write FOnPktParse;
  end;

  TConnectTrd = class(TThread)
  public
    FSelf: TClSock;
    procedure Execute; override;
    procedure DoWork;
  end;

function GetLocalIP: LongInt;

implementation
const
  WSA_NETEVENT = WM_USER + $20;
  WSA_RESOLVE_COMPLETE = WM_USER + $30;

function InitMySocket(var WSA: TWSAData): LongWord;
begin
  Result := WSAStartup(MAKEWORD(1, 1), WSA);
end;

procedure FinalMySocket;
begin
  WSACleanUp;
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////
{$IFNDEF USE_FORMS}
type
  PObjectInstance = ^TObjectInstance;
  TObjectInstance = packed record
    Code: Byte;
    Offset: Integer;
    case Integer of
      0: (Next: PObjectInstance);
      1: (Method: TWndMethod);
  end;

  PInstanceBlock = ^TInstanceBlock;
  TInstanceBlock = packed record
    Next: PInstanceBlock;
    Code: array[1..2] of Byte;
    WndProcPtr: Pointer;
    Instances: array[0..100] of TObjectInstance;
  end;

var
  InstBlockList: PInstanceBlock;
  InstFreeList: PObjectInstance;

{ Standard window procedure }
{ In    ECX = Address of method pointer }
{ Out   EAX = Result }

function StdWndProc(Window: HWND; Message, WParam: Longint;
  LParam: Longint): Longint; stdcall; assembler;
asm
          XOR     EAX,EAX
          PUSH    EAX
          PUSH    LParam
          PUSH    WParam
          PUSH    Message
          MOV     EDX,ESP
          MOV     EAX,[ECX].Longint[4]
          CALL    [ECX].Pointer
          ADD     ESP,12
          POP     EAX
end;

{ Allocate an object instance }
function CalcJmpOffset(Src, Dest: Pointer): Longint;
begin
  Result := Longint(Dest) - (Longint(Src) + 5);
end;

function MakeObjectInstance(Method: TWndMethod): Pointer;
const
  BlockCode: array[1..2] of Byte = (
    $59,       { POP ECX }
    $E9);      { JMP StdWndProc }
  PageSize = 4096;
var
  Block: PInstanceBlock;
  Instance: PObjectInstance;
begin
  if InstFreeList = nil then
  begin
    Block := VirtualAlloc(nil, PageSize, MEM_COMMIT, PAGE_EXECUTE_READWRITE);
    Block^.Next := InstBlockList;
    Move(BlockCode, Block^.Code, SizeOf(BlockCode));
    Block^.WndProcPtr := Pointer(CalcJmpOffset(@Block^.Code[2], @StdWndProc));
    Instance := @Block^.Instances;
    repeat
      Instance^.Code := $E8;  { CALL NEAR PTR Offset }
      Instance^.Offset := CalcJmpOffset(Instance, @Block^.Code);
      Instance^.Next := InstFreeList;
      InstFreeList := Instance;
      Inc(Longint(Instance), SizeOf(TObjectInstance));
    until Longint(Instance) - Longint(Block) >= SizeOf(TInstanceBlock);
    InstBlockList := Block;
  end;
  Result := InstFreeList;
  Instance := InstFreeList;
  InstFreeList := Instance^.Next;
  Instance^.Method := Method;
end;

{ Free an object instance }
procedure FreeObjectInstance(ObjectInstance: Pointer);
begin
  if ObjectInstance <> nil then
  begin
    PObjectInstance(ObjectInstance)^.Next := InstFreeList;
    InstFreeList := ObjectInstance;
  end;
end;

var
  UtilWindowClass: TWndClass = (
  style: 0;
  lpfnWndProc: @DefWindowProc;
  cbClsExtra: 0;
  cbWndExtra: 0;
  hInstance: 0;
  hIcon: 0;
  hCursor: 0;
  hbrBackground: 0;
  lpszMenuName: nil;
  lpszClassName: 'TPUtilWindow');

function AllocateHWnd(Method: TWndMethod): THandle;
var
  TempClass: TWndClass;
  ClassRegistered: Boolean;
begin
  UtilWindowClass.hInstance := HInstance;
  ClassRegistered := GetClassInfo(HInstance, UtilWindowClass.lpszClassName,
    TempClass);
  if not ClassRegistered or (TempClass.lpfnWndProc <> @DefWindowProc) then
  begin
    if ClassRegistered then
      Windows.UnregisterClass(UtilWindowClass.lpszClassName, HInstance);
    Windows.RegisterClass(UtilWindowClass);
  end;
  Result := CreateWindow(UtilWindowClass.lpszClassName,
    '', WS_POPUP, 0, 0, 0, 0, 0, 0, HInstance, nil);
  if Assigned(Method) then
    SetWindowLong(Result, GWL_WNDPROC, Longint(MakeObjectInstance(Method)));
end;

procedure DeallocateHWnd(Wnd: THandle);
var
  Instance: Pointer;
begin
  Instance := Pointer(GetWindowLong(Wnd, GWL_WNDPROC));
  DestroyWindow(Wnd);
  if Instance <> @DefWindowProc then FreeObjectInstance(Instance);
end;
{$ENDIF}
/////////////////////////////////////////////////////////////////////////////////////////////////////////

constructor TClSock.Create;
begin
  inherited;
  FConnTrd := nil;
  FWndHandle := AllocateHwnd(OnSockMsg);
  FClSock := INVALID_SOCKET;
end;

destructor TClSock.Destroy;
begin
  Disconnect;
  DeallocateHwnd(FWndHandle);
  if FConnTrd <> nil then
  begin
    FConnTrd.Terminate;
    FConnTrd.Free;
  end;
  inherited;
end;


procedure TClSock.OnSockMsg(var Msg: TMessage);
var
  rc: Integer;
  buf: array[0..255] of Byte;
begin
  case Msg.Msg of
    WSA_RESOLVE_COMPLETE:
    begin
      if HIWORD(Msg.WParam) <> 0 then
      begin
        Disconnect;
        if Assigned(OnConnectError) then
          FOnConnectError(Self);
        Exit;
      end;
      if FConnTrd <> nil then
      begin
        FConnTrd.FreeOnTerminate := False;
        FConnTrd.Free;
      end;
      FConnTrd := TConnectTrd.Create(True);
      FConnTrd.FSelf := Self;
      FConnTrd.Resume;
    end;
    WSA_NETEVENT:
    begin
      if WSAGetSelectEvent(Msg.lParam) = FD_READ then
      begin
        rc := recv(Msg.wParam, buf, SizeOf(buf), 0);
        if rc <> SOCKET_ERROR then
        begin
          buf[rc] := 0;
          if Assigned(OnRecieve) then
            FOnRecv(Self, Msg.wParam, @buf, rc);
        end;
        Exit;
      end
      //Connection with server was lost
      else if WSAGetSelectEvent(Msg.lParam) = FD_CLOSE then
      begin
        if Assigned(OnDisconnect) then
          FOnDisconnect(Self);
      end;
    end;
  end;
end;

function TClSock.IsConnected: Boolean;
begin
  Result := FClSock <> INVALID_SOCKET;
end;

procedure TClSock.Connect;
begin
  if WSAAsyncGetHostByName(FWndHandle, WSA_RESOLVE_COMPLETE, PChar(FIp), @FHostIp, SizeOf(FHostIp)) = 0 then
  begin
    Disconnect;
    if Assigned(OnConnectError) then
      FOnConnectError(Self);
    Exit;
  end;
end;

procedure TClSock.Disconnect;
begin
  if FClSock <> INVALID_SOCKET then
  begin
    closesocket(FClSock);
    if Assigned(OnDisconnect) then
      FOnDisconnect(Self);
  end;
  FClSock := INVALID_SOCKET;
end;

procedure TClSock.SendData(var Buf; BufLen: LongWord);
begin
  if FClSock <> INVALID_SOCKET then
  begin
    if send(FClSock, Buf, BufLen, 0) = SOCKET_ERROR then
      Disconnect;
    if Assigned(OnPktParse) then
      FOnPktParse(Self, @Buf, BufLen);
  end;
end;

function GetLocalIP: LongInt;
type
  TaPInAddr = array[0..0] of PInAddr;
  PaPInAddr = ^TaPInAddr;
var
  phe: PHostEnt;
  pptr: PaPInAddr;
  Buffer: array[0..63] of Char;
  I: Integer;
begin
  Result := -1;
  GetHostName(Buffer, SizeOf(Buffer));
  phe := GetHostByName(buffer);
  if phe = nil then Exit;
  pptr := PaPInAddr(Phe^.h_addr_list);
  I := 0;
  while pptr^[I] <> nil do begin
    result := pptr^[I]^.S_addr;
    Inc(I);
  end;
end;

procedure TConnectTrd.Execute;
begin
  DoWork;
end;

procedure TConnectTrd.DoWork;
  function ResolveAddr(Value: String): LongInt;
  var
    addr: in_addr;
    hostent: PHostEnt;
  begin
    Result := -1;
    hostent := @FSelf.FHostIp;
    if hostent^.h_addr_list <> nil then
    begin
      addr.S_addr := PLongInt(hostent^.h_addr_list^)^;
      Result := addr.S_addr;
    end else
      Exit;
  end;
var
  dest_sin: sockaddr_in;
  rc: Integer;
begin
  //If were connected, - disconnect
  FSelf.Disconnect;

  //Creating socket
  FSelf.FClSock := socket(AF_INET, SOCK_STREAM, 0);
  if FSelf.FClSock = INVALID_SOCKET then
  begin
    MessageBox(0, 'socket  Error', 'Error', MB_OK);
    Exit;
  end;

  //Setting IP and Port
  dest_sin.sin_family := AF_INET;
  dest_sin.sin_addr.s_addr := ResolveAddr(FSelf.FIp);
  dest_sin.sin_port := htons(FSelf.FDestPort);

  if WinSock.connect(FSelf.FClSock, dest_sin, SizeOf(dest_sin)) < 0 then
  begin
    FSelf.Disconnect;
    if Assigned(FSelf.OnConnectError) then
      FSelf.FOnConnectError(Self);
    Exit;
  end;
  if Assigned(FSelf.OnConnect) then
    FSelf.FOnConnect(Self);

  rc := WSAAsyncSelect (FSelf.FClSock, FSelf.FWndHandle, WSA_NETEVENT,
    FD_READ or FD_CLOSE);
  if rc > 0 then
  begin
    FSelf.Disconnect;
    MessageBox(0, 'WSAAsyncSelect  Error', 'Error', MB_OK);
    Exit;
  end;
end;


end.
