unit ICQClient {v. 0.6};
{************************************************
    For updates checkout: http://soho.net.md
      (C) Alex Demchenko(alex@ritlabs.com)
*************************************************}

interface
uses
  Windows, Classes, ICQWorks, WinSock, MySocket;

type
  PUINEntry = ^TUINEntry;
  TUINEntry = record
    UIN: LongWord;
    Nick: ShortString;
    CType: Word;
    CTag: Word;
    CGroup: ShortString;
  end;

  TOnMsgProc = procedure(Sender: TObject; MsgType: LongWord; Msg, UIN: String) of object;
  TOnOffMsgProc = procedure(Sender: TObject; MsgType: Word; Msg, UIN: String) of object;
  TOnAdvPktParse = procedure(Sender: TObject; Buffer: Pointer; BufLen: LongWord; Incoming: Boolean) of object;
  TOnStatusChange = procedure(Sender: TObject; UIN: String; Status: LongWord) of object;
  TOnUserOffline = procedure(Sender: TObject; UIN: String) of object;
  TOnUserGeneralInfo = procedure(Sender: TObject; UIN, NickName, FirstName, LastName, Email, City, State, Phone, Fax, Street, Cellular,
    Zip, Country: String; TimeZone: Byte; PublishEmail: Boolean) of object;
  TOnLoginFailed = procedure(Sender: TObject) of object;

  TOnUserWorkInfo = procedure (Sender: TObject; UIN,
    WCity, WState, WPhone, WFax, FAddress, WZip, WCountry,
    WCompany, WDepartment, WPosition, WOccupation, WHomePage: String) of object;

  TOnUserInfoMore = procedure (Sender: TObject; UIN: String;
    Age: Word; Gender: Byte; HomePage: String; BirthYear: Word;
    BirthMonth: Word; BirthDay: Word; Lang1, Lang2, Lang3: String) of object;

  TOnUserInfoAbout = procedure(Sender: TObject; UIN, About: String) of object;

  TOnUserInfoInterests = procedure(Sender: TObject; UIN: String; Interests: TStringList) of object;
  TOnUserInfoMoreEmails = procedure(Sender: TObject; UIN: String; Emails: TStringList) of object;
  TOnUserInfoBackground = procedure(Sender: TObject; UIN: String; Pasts, Affiliations: TStringList) of object;

  TOnUserFound = procedure(Sender: TObject; UIN, Nick, FirstName, LastName, Email: String; Status: Word; Gender, Age: Byte; SearchComplete: Boolean) of object;
  TOnServerListRecv = procedure(Sender: TObject; SrvContactList: TList) of object;
  TOnAdvMsgAck = procedure(Sender: TObject; UIN: String; ID: Word; AcceptType: Byte; AcceptMsg: String) of object;

  TICQClient = class(TComponent)
  private
    FSock: TClSock;                                     //Client's socket
    FLUIN: LongWord;                                    //Client's UIN
    FLPass: String;                                     //Client's password
    FFirstConnect: Boolean;                             //Flag, used in login sequence
    FSrcBuf: array[0..MAX_DATA_LEN - 1] of Byte;        //.              .
    FSrcLen: Word;                                      //.PACKET READING.
    FNewFlap: TFlapHdr;                                 //.     DATA     .
    FFlapSet: Boolean;                                  //.              .
    FSeq: Word;                                         //Main Flap Seq
    FSeq2: Word;                                        //TO_ICQSRV Seq
    FCookie: String;                                    //Temporary cookie, used in login sequence
    FIp: String;                                        //Ip to connect to
    FPort: Word;                                        //Port to connect to
    //---
    FContactLst: TStrings;
    FVisibleLst: TStrings;
    FInvisibleLst: TStrings;
    FOnMsg: TOnMsgProc;
    FOnOffMsg: TOnOffMsgProc;
    FOnLogin: TNotifyEvent;
    FOnPktParse: TOnAdvPktParse;
    FOnConnectionFailed: TOnLoginFailed;
    FOnStatusChange: TOnStatusChange;
    FOnUserOffline: TOnUserOffline;
    FOnAddedYou: TOnUserOffline;
    FOnUserGeneralInfo: TOnUserGeneralInfo;
    FOnUserWorkInfo: TOnUserWorkInfo;
    FOnUserInfoMore: TOnUserInfoMore;
    FOnUserInfoAbout: TOnUserInfoAbout;
    FOnUserInfoInterests: TOnUserInfoInterests;
    FOnUserInfoMoreEmails: TOnUserInfoMoreEmails;
    FOnUserInfoBackground: TOnUserInfoBackground;
    FStatus: LongWord;
    FDoPlain: Boolean;
    FInfoChain: TStringList;
    FLastInfoUin: String;
    FConnected: Boolean;
    FOnUserFound: TOnUserFound;
    FOnUserNotFound: TNotifyEvent;
    FOnServerListRecv: TOnServerListRecv;
    FOnAdvMsgAck: TOnAdvMsgAck;
    procedure OnSockRecv(Sender: TObject; Socket: TSocket; Buffer: Pointer; BufLen: LongWord);
    procedure HandlePacket(Flap: TFlapHdr; Data: Pointer);
    procedure SetStatus(NewStatus: LongWord);
    procedure HSnac0407(Flap: TFlapHdr; Snac: TSnacHdr; Pkt: PRawPkt);
    procedure HSnac2103(Flap: TFlapHdr; Snac: TSnacHdr; Pkt: PRawPkt);
    procedure HSnac030B(Flap: TFlapHdr; Snac: TSnacHdr; Pkt: PRawPkt);
    procedure HSnac131C(Flap: TFlapHdr; Snac: TSnacHdr; Pkt: PRawPkt);
    procedure HSnac1319(Flap: TFlapHdr; Snac: TSnacHdr; Pkt: PRawPkt);
    procedure HSnac1306(Flap: TFlapHdr; Snac: TSnacHdr; Pkt: PRawPkt);
    procedure HSnac040b(Flap: TFlapHdr; Snac: TSnacHdr; Pkt: PRawPkt);
    procedure FTOnConnectError(Sender: TObject);
    procedure FTOnDisconnect(Sender: TObject);
    procedure FTOnPktParse(Sender: TObject; Buffer: Pointer; BufLen: LongWord);
    procedure SetContactList(Value: TStrings);
    procedure SetVisibleList(Value: TStrings);
    procedure SetInvisibleList(Value: TStrings);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Login(Status: LongWord = S_ONLINE);
    procedure Disconnect;
    procedure SendMessage(UIN: LongWord; const Msg: String);
    procedure SendURL(UIN: LongWord; const URL, Description: String);
    function AddContact(UIN: LongWord): Boolean;
    function AddContactVisible(UIN: LongWord): Boolean;
    function AddContactInvisible(UIN: LongWord): Boolean;
    procedure RemoveContact(UIN: LongWord);
    procedure RemoveContactVisible(UIN: LongWord);
    procedure RemoveContactInvisible(UIN: LongWord);
    procedure RequestInfo(UIN: LongWord);
    procedure SearchByMail(const Email: String);
    procedure SearchByUIN(UIN: LongWord);
    procedure SearchByName(const FirstName, LastName, NickName, Email: String);
    procedure SearchRandom(Group: Word);
    procedure SearchWhitePages(const First, Last, Nick, Email: String; MinAge, MaxAge: Word; Gender: Byte; const Language, City, State, Country, Company, Department, Position, Occupation, Past, PastDesc, Interests, InterDesc, Affiliation, AffiDesc, HomePage: String; Online: Boolean);
    procedure SetSelfInfoGeneral(NickName, FirstName, LastName, Email, City, State, Phone, Fax, Street, Cellular, Zip, Country: String; TimeZone: Byte; PublishEmail: Boolean);
    procedure SetSelfInfoMore(Age: Word; Gender: Byte; const HomePage: String; BirthYear: Word; BirthMonth, BirthDay: Byte; Language1, Language2, Language3: String);
    procedure SetSelfInfoAbout(const About: String);
    procedure RequestContactList;
    procedure DestroyUINList(var List: TList);
    procedure SendSMS(const Destination, Text: String);
    procedure SendMessageAdvanced(UIN: LongWord; const Msg: String; ID: Word; RTFFormat: Boolean);
    property Status: LongWord read FStatus write SetStatus;
    property LoggedIn: Boolean read FConnected;
  published
    property UIN: LongWord read FLUIN write FLUIN;
    property Pasword: String read FLPass write FLPass;
    property ICQServer: String read FIp write FIp;
    property ICQPort: Word read FPort write FPort;
    property ConvertToPlaintext: Boolean read FDoPlain write FDoPlain;
    property ContactList: TStrings read FContactLst write SetContactList;
    property VisibleList: TStrings read FVisibleLst write SetVisibleList;
    property InvisibleList: TStrings read FInvisibleLst write SetInvisibleList;
    property OnLogin: TNotifyEvent read FOnLogin write FOnLogin;
    property OnMessageRecv: TOnMsgProc read FOnMsg write FOnMsg;
    property OnOfflineMsgRecv: TOnOffMsgProc read FOnOffMsg write FOnOffMsg;
    property OnPktParse: TOnAdvPktParse read FOnPktParse write FOnPktParse;
    property OnConnectionFailed: TOnLoginFailed read FOnConnectionFailed write FOnConnectionFailed;
    property OnStatusChange: TOnStatusChange read FOnStatusChange write FOnStatusChange;
    property OnUserOffline: TOnUserOffline read FOnUserOffline write FOnUserOffline;
    property OnAddedYou: TOnUserOffline read FOnAddedYou write FOnAddedYou;
    property OnUserGeneralInfo: TOnUserGeneralInfo read FOnUserGeneralInfo write FOnUserGeneralInfo;
    property OnUserWorkInfo: TOnUserWorkInfo read FOnUserWorkInfo write FOnUserWorkInfo;
    property OnUserInfoMore: TOnUserInfoMore read FOnUserInfoMore write FOnUserInfoMore;
    property OnUserInfoAbout: TOnUserInfoAbout read FOnUserInfoAbout write FOnUserInfoAbout;
    property OnUserInfoInterests: TOnUserInfoInterests read FOnUserInfoInterests write FOnUserInfoInterests;
    property OnUserInfoMoreEmails: TOnUserInfoMoreEmails read FOnUserInfoMoreEmails write FOnUserInfoMoreEmails;
    property OnUserInfoBackground: TOnUserInfoBackground read FOnUserInfoBackground write FOnUserInfoBackground;
    property OnUserFound: TOnUserFound read FOnUserFound write FOnUserFound;
    property OnUserNotFound: TNotifyEvent read FOnUserNotFound write FOnUserNotFound;
    property OnServerListRecv: TOnServerListRecv read FOnServerListRecv write FOnServerListRecv;
    property OnAdvancedMsgAck: TOnAdvMsgAck read FOnAdvMsgAck write FOnAdvMsgAck;
  end;

procedure Register;

implementation
{*** CONSTRUCTOR ***}
constructor TICQClient.Create(AOwner: TComponent);
var
  WSA: TWSAData;
begin
  inherited;
  InitMySocket(WSA);  //Starting WSA

  FContactLst := TStringList.Create;  //Contact list
  FVisibleLst := TStringList.Create;  //Visible list
  FInvisibleLst := TStringList.Create; //Invisible list

  FInfoChain := TStringList.Create;   //Info request chaing

  //Socket for working with TCP
  FSock := TClSock.Create;
  FSock.OnRecieve := OnSockRecv;
  FSock.OnDisconnect := FTOnDisconnect;
  FSock.OnConnectError := FTOnConnectError;
  FSock.OnPktParse := FTOnPktParse;

  Randomize;
  FSeq := Random($AAAA);
end;

{*** DESTRUCTOR ***}
destructor TICQClient.Destroy;
begin
  FSock.OnRecieve := nil;          //.                                               .
  FSock.OnDisconnect := nil;       //.   DO NOT USE NOTIFICATIONS WHILE DESTROYING   .
  FSock.OnConnectError := nil;     //.      THE OBJECT, CAUSES ACCESS VIOLATIONS     .
  FSock.OnPktParse := nil;         //.                                               .
  FSock.Free;

  //Free TStringList objects
  FContactLst.Free;
  FVisibleLst.Free;
  FInvisibleLst.Free;
  FInfoChain.Free;

  FinalMySocket; //Cleanup WSA
  inherited;
end;

procedure TICQClient.Login(Status: LongWord = S_ONLINE);
begin
  FSeq2 := 2;
  FCookie := '';
  FSrcLen := 0;
  FFlapSet := False;
  FFirstConnect := True;
  //FSock.IP := '205.188.179.233';
  //FSock.IP := '127.0.0.1';
  FStatus := Status;
  FSock.IP := FIp;
  FSock.DestPort := FPort;
  FSock.Connect;
  FConnected := False;
  //FConnecting := True;
end;

procedure TICQClient.Disconnect;
begin
  FSock.Disconnect;
end;

{Send a message to UIN.}
procedure TICQClient.SendMessage(UIN: LongWord; const Msg: String);
var
  pkt: TRawPkt;
begin
  if not LoggedIn then Exit;
  CreateCLI_SENDMSG(@pkt, 0, Random($FFFFAA), UIN, Msg, FSeq);
  FSock.SendData(pkt, pkt.Len);
end;

{Send an URL message to UIN.}
procedure TICQClient.SendURL(UIN: LongWord; const URL, Description: String);
var
  pkt: TRawPkt;
begin
  if not LoggedIn then Exit;
  CreateCLI_SENDURL(@pkt, 0, Random($FFFFAA), FLUIN, UIN, URL, Description, FSeq);
  FSock.SendData(pkt, pkt.Len);
end;

{Adds UIN to contact list after logon(when you are online), UIN automaticly
added to ContactList TStrings. After adding the UIN you will receive status
notifications. Returns True when UIN is added to the list(it wasn't there before).}
function TICQClient.AddContact(UIN: LongWord): Boolean;
var
  pkt: TRawPkt;
  List: TStringList;
begin
  Result := False;
  if FContactLst.IndexOf(IntToStr(UIN)) < 0 then
  begin
    FContactLst.Add(IntToStr(UIN));
    Result := True;
  end else
    Exit;
  if not LoggedIn then Exit;
  List := TStringList.Create;                       //Create temporary UIN list
  List.Add(IntToStr(UIN));                          //Add a single UIN into it
  CreateCLI_ADDCONTACT(@pkt, List, FSeq);           {SNAC(x03/x04)}
  FSock.SendData(pkt, pkt.Len);
  List.Free;                                        //Free temporary list
end;

{Adds UIN to visible list after logon, UIN automaticly added to VisibleList
TStrings. After adding the UIN you will be visible to this UIN in invisible
status. Returns True when UIN is added to the list(it wasn't there before).}
function TICQClient.AddContactVisible(UIN: LongWord): Boolean;
var
  pkt: TRawPkt;
  List: TStringList;
begin
  Result := False;
  if FVisibleLst.IndexOf(IntToStr(UIN)) < 0 then
  begin
    FVisibleLst.Add(IntToStr(UIN));
    Result := True;
  end else
    Exit;
  if not LoggedIn then Exit;
  List := TStringList.Create;                       //Create temporary UIN list
  List.Add(IntToStr(UIN));                          //Add a single UIN into it
  CreateCLI_ADDVISIBLE(@pkt, List, FSeq);           {SNAC(x09/x05)}
  FSock.SendData(pkt, pkt.Len);
  List.Free;                                        //Free temporary list
end;

{Adds UIN to invisible list after logon, UIN automaticly added to InvisibleList
TStrings. After adding the UIN you will be invisible to this UIN in any status.
Returns True when UIN is added to the list(it wasn't there before).}
function TICQClient.AddContactInvisible(UIN: LongWord): Boolean;
var
  pkt: TRawPkt;
  List: TStringList;
begin
  Result := False;
  if FInvisibleLst.IndexOf(IntToStr(UIN)) < 0 then
  begin
    FInvisibleLst.Add(IntToStr(UIN));
    Result := True;
  end else
    Exit;
  if not LoggedIn then Exit;
  List := TStringList.Create;                       //Create temporary UIN list
  List.Add(IntToStr(UIN));                          //Add a single UIN into it
  CreateCLI_ADDINVISIBLE(@pkt, List, FSeq);         {SNAC(x09/x07)}
  FSock.SendData(pkt, pkt.Len);
  List.Free;                                        //Free temporary list
end;

{Removes UIN from contact list. Use while you are online.}
procedure TICQClient.RemoveContact(UIN: LongWord);
var
  idx: Integer;
  pkt: TRawPkt;
begin
  idx := FContactLst.IndexOf(IntToStr(UIN));
  if idx > -1 then
    FContactLst.Delete(idx);
  if not LoggedIn then Exit;
  CreateCLI_REMOVECONTACT(@pkt, UIN, FSeq);
  FSock.SendData(pkt, pkt.Len);
end;

{Removes UIN from the visible list. Use while you are online.}
procedure TICQClient.RemoveContactVisible(UIN: LongWord);
var
  idx: Integer;
  pkt: TRawPkt;
begin
  idx := FVisibleLst.IndexOf(IntToStr(UIN));
  if idx > -1 then
    FVisibleLst.Delete(idx);
  if not LoggedIn then Exit;
  CreateCLI_REMVISIBLE(@pkt, UIN, FSeq);
  FSock.SendData(pkt, pkt.Len);
end;

{Removes UIN from the invisible list. Use while you are online.}
procedure TICQClient.RemoveContactInvisible(UIN: LongWord);
var
  idx: Integer;
  pkt: TRawPkt;
begin
  idx := FInvisibleLst.IndexOf(IntToStr(UIN));
  if idx > -1 then
    FInvisibleLst.Delete(idx);
  if not LoggedIn then Exit;
  CreateCLI_REMINVISIBLE(@pkt, UIN, FSeq);
  FSock.SendData(pkt, pkt.Len);
end;

{Query info about UIN. As answer you will recieve theese events: OnUserWorkInfo,
OnUserInfoMore, OnUserInfoAbout, OnUserInfoInterests, OnUserInfoMoreEmails,
OnUserFound.}
procedure TICQClient.RequestInfo(UIN: LongWord);
var
  pkt: TRawPkt;
begin
  if not LoggedIn then Exit;
  FInfoChain.Add(IntToStr(UIN));
  CreateCLI_METAREQINFO(@pkt, FLUIN, UIN, FSeq, FSeq2);
  FSock.SendData(pkt, pkt.Len);
end;

{Searches user by Mail}
procedure TICQClient.SearchByMail(const Email: String);
var
  pkt: TRawPkt;
begin
  if not LoggedIn then Exit;
  CreateCLI_SEARCHBYMAIL(@pkt, FLUIN, Email, FSeq, FSeq2);
  FSock.SendData(pkt, pkt.Len);
end;

{Searches user by UIN}
procedure TICQClient.SearchByUIN(UIN: LongWord);
var
  pkt: TRawPkt;
begin
  if not LoggedIn then Exit;
  CreateCLI_SEARCHBYUIN(@pkt, FLUIN, UIN, FSeq, FSeq2);
  FSock.SendData(pkt, pkt.Len);
end;

{Searches user by Name and other data}
procedure TICQClient.SearchByName(const FirstName, LastName, NickName, Email: String);
var
  pkt: TRawPkt;
begin
  if not LoggedIn then Exit;
  CreateCLI_SEARCHBYNAME(@pkt, FLUIN, FirstName, LastName, NickName, Email, FSeq, FSeq2);
  FSock.SendData(pkt, pkt.Len);
end;

{Searches random user from Group, where Group id could be found in RandGroups:
array[1..11]...(ICQWorks.pas) constant. As answer you will receive OnUserFound
notification, only one user will be found.}
procedure TICQClient.SearchRandom(Group: Word);
var
  pkt: TRawPkt;
begin
  if not LoggedIn then Exit;
  CreateCLI_SEARCHRANDOM(@pkt, FLUIN, Group, FSeq, FSeq2);
  FSock.SendData(pkt, pkt.Len);
end;

{Searches user in 'White Pages'. As answer you will receive OnUserFound notification
when at least one user found or OnUserNotFound if such user does not exist.}
procedure TICQClient.SearchWhitePages(const First, Last, Nick, Email: String; MinAge, MaxAge: Word;
  Gender: Byte; const Language, City, State, Country, Company, Department, Position, Occupation,
  Past, PastDesc, Interests, InterDesc, Affiliation, AffiDesc, HomePage: String; Online: Boolean);
var
  pkt: TRawPkt;
begin
  if not LoggedIn then Exit;
  CreateCLI_SEARCHWP(@pkt, FLUIN, First, Last, Nick, Email, MinAge, MaxAge, Gender, StrToLanguageI(Language), City, State, StrToCountryI(Country),  Company,  Department, Position, StrToOccupationI(Occupation), StrToPastI(Past), PastDesc, StrToInterestI(Interests), InterDesc, StrToAffiliationI(Affiliation), AffiDesc, HomePage, Ord(Online), FSeq, FSeq2);
  FSock.SendData(pkt, pkt.Len);
end;

{Set general info about yourself. You can skip some parameters (eg. use '' -
empty strings) to unspecify some info. }
procedure TICQClient.SetSelfInfoGeneral(NickName, FirstName, LastName, Email, City, State, Phone, Fax, Street, Cellular, Zip, Country: String; TimeZone: Byte; PublishEmail: Boolean);
var
  pkt: TRawPkt;
begin
  if not LoggedIn then Exit;
  //Truncate state if more then 3 chars
  if Length(State) > 3 then
    State := Copy(State, 0, 3);
  CreateCLI_METASETGENERAL(@pkt, FLUIN, NickName, FirstName, LastName, Email, City, State, Phone, Fax, Street, Cellular, Zip, StrToCountryI(Country), TimeZone, PublishEmail, FSeq, FSeq2);
  FSock.SendData(pkt, pkt.Len);
end;

{Set more info about yourself.}
procedure TICQClient.SetSelfInfoMore(Age: Word; Gender: Byte; const HomePage: String; BirthYear: Word; BirthMonth, BirthDay: Byte; Language1, Language2, Language3: String);
var
  pkt: TRawPkt;
begin
  if not LoggedIn then Exit;
  CreateCLI_METASETMORE(@pkt, FLUIN, Age, Gender, HomePage, BirthYear, BirthMonth, BirthDay, StrToLanguageI(Language1), StrToLanguageI(Language2), StrToLanguageI(Language3), FSeq, FSeq2);
  FSock.SendData(pkt, pkt.Len);
end;

{Set info about yourself.}
procedure TICQClient.SetSelfInfoAbout(const About: String);
var
  pkt: TRawPkt;
begin
  if not LoggedIn then Exit;
  CreateCLI_METASETABOUT(@pkt, FLUIN, About, FSeq, FSeq2);
  FSock.SendData(pkt, pkt.Len);
end;

{Requests server side contact list. For more info look at OnServerListRecv event.}
procedure TICQClient.RequestContactList;
var
  pkt: TRawPkt;
begin
  if not LoggedIn then Exit;
  CreateCLI_REQROSTER(@pkt, FSeq);
  FSock.SendData(pkt, pkt.Len);
end;

{Releases memory used while parsing the server side contact list.}
procedure TICQClient.DestroyUINList(var List: TList);
var
  i: Word;
begin
  if List = nil then Exit;
  if List.Count > 0 then
    for i := 0 to List.Count - 1 do
      FreeMem(List.Items[i], SizeOf(TUINEntry)); //Free allocated memory for TUINEntry
  List.Free;
  List := nil;
end;

{Sends sms message to Destination with Text.}
procedure TICQClient.SendSMS(const Destination, Text: String);
var
  pkt: TRawPkt;
begin
  if (Length(Text) = 0) or (not LoggedIn) then Exit;
  CreateCLI_SENDSMS(@pkt, FLUIN, Destination, Text, GetACP, GetSMSTime, FSeq, FSeq2);
  FSock.SendData(pkt, pkt.Len);
end;

{Sends Msg to UIN with advanced options, after UIN has got your message you will
receive confirmation. ID - randomly generated value, may be used for packet acknowledgements
(see OnAdvancedMsgAck event). If your Msg is in the RTF(RichText Format), then RTFFormat
parameter should be True, otherwise - False. Beware of using the RTF Format, some clients
(old versions of ICQ, linux & windows clones) don't support it.}
procedure TICQClient.SendMessageAdvanced(UIN: LongWord; const Msg: String; ID: Word; RTFFormat: Boolean);
var
  pkt: TRawPkt;
begin
  if (Length(Msg) = 0) or (not LoggedIn) then Exit;
  CreateCLI_SENDMSG_ADVANCED(@pkt, 0, ID, UIN, Msg, RTFFormat, FSeq);
  FSock.SendData(pkt, pkt.Len);
end;

procedure TICQClient.OnSockRecv(Sender: TObject; Socket: TSocket; Buffer: Pointer; BufLen: LongWord);
var
  i: Word;
begin
  for i := 0 to BufLen - 1 do
  begin
    FSrcBuf[FSrcLen] := PByteArray(Buffer)^[i];
    Inc(FSrcLen);
    //Searching for the Flap header
    if (FSrcLen >= TFLAPSZ) and (not FFlapSet) then
    begin
      FFlapSet := True;
      FNewFlap := PFlapHdr(@FSrcBuf)^;
      FNewFlap.DataLen := Swap16(FNewFlap.DataLen);
      FNewFlap.Seq := Swap16(FNewFlap.Seq);
    end;
    //Whole packet was received
    if FSrcLen = FNewFlap.DataLen + TFLAPSZ then
    begin
      if Assigned(OnPktParse) then
        FOnPktParse(Self, @FSrcBuf, FSrcLen, True);
      //Handling packet
      HandlePacket(FNewFlap, Ptr(LongWord(@FSrcBuf) + TFLAPSZ));
      //Preparing structures for receiving the next packet
      FNewFlap.DataLen := 0;
      FSrcLen := 0;
      FFlapSet := False;
    end;
  end;
end;

{@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@}
{Handling of all incoming packet}
procedure TICQClient.HandlePacket(Flap: TFlapHdr; Data: Pointer);
var
  FUIN: String;
  FData: String;
  pkt: TRawPkt;
  T: Word;
  Snac: TSnacHdr;
begin
  case Flap.ChID of
    1: //Channel 1
    begin
      {SRV_HELLO}
      if Flap.DataLen = 4 then
      begin
        if FFirstConnect then
        begin
          //Send login packet
          CreateCLI_IDENT(@pkt, FLUIN, FLPass, FSeq);
          FSock.SendData(pkt, pkt.len);
        end
        else
        begin
          //Sending the cookie(second stage of login sequence)
          CreateCLI_COOKIE(@pkt, FCookie, FSeq);
          FSock.SendData(pkt, pkt.Len);
        end;
      end;
      FFirstConnect := False;
    end;
    2: //Channel 2
    begin
      Move(Data^, pkt.Data, Flap.DataLen); pkt.Len := 0;
      GetSnac(@pkt, Snac);
      case Snac.Family of
        $01: //Family x01
          case Snac.SubType of
            $03: {SRV_FAMILIES}
            begin
              CreateCLI_FAMILIES(@pkt, FSeq);           {SNAC(x01/x17)}
              FSock.SendData(pkt, pkt.Len);
            end;
            $07: {SRV_RATES}
            begin
              CreateCLI_ACKRATES(@pkt, FSeq);           {SNAC(x01/x08)}
              FSock.SendData(pkt, pkt.Len);
              CreateCLI_SETICBM(@pkt, FSeq);            {SNAC(x04/x02)}
              FSock.SendData(pkt, pkt.Len);
              CreateCLI_REQINFO(@pkt, FSeq);            {SNAC(x01/x0E)}
              FSock.SendData(pkt, pkt.Len);
              CreateCLI_REQLOCATION(@pkt, FSeq);        {SNAC(x02/x02)}
              FSock.SendData(pkt, pkt.Len);
              CreateCLI_REQBUDDY(@pkt, FSeq);           {SNAC(x03/x02)}
              FSock.SendData(pkt, pkt.Len);
              CreateCLI_REQICBM(@pkt, FSeq);            {SNAC(x04/x04)}
              FSock.SendData(pkt, pkt.Len);
              CreateCLI_REQBOS(@pkt, FSeq);             {SNAC(x09/x02)}
              FSock.SendData(pkt, pkt.Len);

              {OnLogin Event}
              FConnected := True;
              if Assigned(OnLogin) then
                FOnLogin(Self);
              //FConnecting := False;
            end;
            $13: {SRV_MOTD}
            begin
              CreateCLI_RATESREQUEST(@pkt, FSeq);       {SNAC(x01/x06)}
              FSock.SendData(pkt, pkt.Len);
            end;
          end;
        $03: //Family x03
        begin
          case Snac.SubType of
            $0B: {SRV_USERONLINE}
            begin
              HSnac030B(Flap, Snac, @pkt);
            end;
            $0C: {SRV_USEROFFLINE}
            begin
              FData := GetStr(@pkt, GetInt(@pkt, 1));
              if Assigned(OnUserOffline) then
                FOnUserOffline(Self, FData);
            end;
          end;
        end;
        $04: //Family x04
          if Snac.SubType = $07 then {SRV_MSG}
            HSnac0407(Flap, Snac, @pkt)
          else if Snac.SubType = $0b then {SRV_MSGACK}
            HSnac040b(Flap, Snac, @pkt);
        $09: //Family x09
        begin
          if Snac.SubType = $03 then
          begin
            CreateCLI_SETUSERINFO(@pkt, FSeq);          {SNAC(x02/x04)}
            FSock.SendData(pkt, pkt.Len);
            CreateCLI_ADDCONTACT(@pkt, FContactLst, FSeq);              {SNAC(x03/x04)}
            FSock.SendData(pkt, pkt.Len);
            if StatusToStr(FStatus) <> 'Invisible' then
            begin
              CreateCLI_ADDINVISIBLE(@pkt, FInvisibleLst, FSeq);        {SNAC(x09/x07)}
              FSock.SendData(pkt, pkt.Len);
            end else
            begin
              CreateCLI_ADDVISIBLE(@pkt, FVisibleLst, FSeq);            {SNAC(x09/x05)}
              FSock.SendData(pkt, pkt.Len);
            end;
            CreateCLI_SETSTATUS(@pkt, FStatus, GetLocalIP, 0, FSeq);    {SNAC(x01/x1E)}
            FSock.SendData(pkt, pkt.Len);
            CreateCLI_READY(@pkt, FSeq);                {SNAC(x01/x02)}
            FSock.SendData(pkt, pkt.Len);
            CreateCLI_TOICQSRV(@pkt, FLUIN, CMD_REQOFFMSG, nil, 0, FSeq, FSeq2);    {SNAC(x15/x02)}
            FSock.SendData(pkt, pkt.Len);
          end;
        end;
        $13: //Family x13
        begin
          if Snac.SubType = $1C then {SRV_ADDEDYOU}
          begin
            HSnac131C(Flap, Snac, @pkt);
          end else
          if Snac.SubType = $19 then {SRV_AUTH_REQ}
            HSnac1319(Flap, Snac, @pkt)
          else if Snac.SubType = $06 then {SRV_REPLYROSTER}
            HSnac1306(Flap, Snac, @pkt);
        end;
        $15: //Family x15
        begin
          if Snac.SubType = $03 then {SRV_FROMICQSRV}
            HSnac2103(Flap, Snac, @pkt);
        end;
      end;
    end;
    4: //Channel 4
    begin
      if FConnected then
      begin
        if Assigned(OnConnectionFailed) then
          FOnConnectionFailed(Self);
        FSock.Disconnect;
        Exit;
      end;
      Move(Data^, pkt.Data, Flap.DataLen); pkt.Len := 0;
      if Flap.DataLen <= 40 then
      begin
        if Assigned(OnConnectionFailed) then
          FOnConnectionFailed(Self);
        FSock.Disconnect;
        Exit;
      end
      else
      begin
        //SRV_COOKIE
        FUIN  := GetTLVStr(@pkt, T);                //Client's UIN in ASCII format
        if T <> 1 then
        begin
          if Assigned(OnConnectionFailed) then
            FOnConnectionFailed(Self);
          FSock.Disconnect;
          Exit;
        end;
        FData := GetTLVStr(@pkt, T);                //IP, Port to connect to
        if T <> 5 then
        begin
          if Assigned(OnConnectionFailed) then
            FOnConnectionFailed(Self);
          FSock.Disconnect;
          Exit;
        end;
        FCookie := GetTLVStr(@pkt, T);              //Cookie used in second stage of login
        if T <> 6 then
        begin
          if Assigned(OnConnectionFailed) then
            FOnConnectionFailed(Self);
          FSock.Disconnect;
          Exit;
        end;
        //Sending CLI_GOODBYE
        PktInit(@pkt, 4, FSeq);
        PktFinal(@pkt);
        FSock.SendData(pkt, pkt.Len);
        FSock.Disconnect;
        //Assigning new IP and Port to connect to in second attemp
        FSock.IP := Copy(FData, 0, Pos(':', FData) - 1);
        FSock.DestPort := StrToInt(Copy(FData, Pos(':', FData) + 1, Length(FData) - Pos(':', FData)));
        if (FSock.DestPort = 0) then
        begin
          if Assigned(OnConnectionFailed) then
            FOnConnectionFailed(Self);
          Exit;
        end;
        FSock.Connect;
      end
    end;
  end;
end;

{////////////////////////////////////////////////////////////////////////////////////////////////////}
procedure TICQClient.SetStatus(NewStatus: LongWord);
var
  pkt: TRawPkt;
begin
  if not LoggedIn then Exit;
  if (StatusToStr(FStatus) = 'Invisible') and (StatusToStr(NewStatus) <> 'Invisible') then
  begin
    CreateCLI_ADDINVISIBLE(@pkt, FInvisibleLst, FSeq);
    FSock.SendData(pkt, pkt.Len);
  end else
  if (StatusToStr(NewStatus) = 'Invisible') and (StatusToStr(FStatus) <> 'Invisible') then
  begin
    CreateCLI_ADDVISIBLE(@pkt, FVisibleLst, FSeq);
    FSock.SendData(pkt, pkt.Len);
  end;
  CreateCLI_SETSTATUS_SHORT(@pkt, NewStatus, FSeq);
  FSock.SendData(pkt, pkt.Len);
  FStatus := NewStatus;
end;

{Handling packet with messages}
procedure TICQClient.HSnac0407(Flap: TFlapHdr; Snac: TSnacHdr; Pkt: PRawPkt);
var
  ULen: Word;
  c, i: Word;
  ack_pkt: TRawPkt;
  chunks: array[0..49] of Byte;
  Msg, UIN: String;
  MsgType: Word;
begin
  Inc(Pkt^.Len, 8);                             //Time, RandomID
  Msg := '';
  {Subtypes}
  case GetInt(Pkt, 2) of
    1:                                          //Simply(old-type) message
    begin
      Uin := GetStr(Pkt, GetInt(Pkt, 1));
      Inc(Pkt^.Len, 2);
      c := GetInt(Pkt, 2);                      //A count of the number of following TLVs.
      for i := 0 to c - 1 do                    //Skip all TLVs
      begin
        Inc(Pkt^.Len, 2);
        Inc(Pkt^.Len, GetInt(Pkt, 2));
      end;
      if GetInt(Pkt, 2) = 2 then                //TLV with message remain
      begin
        Inc(Pkt^.Len, 4);                       //TLV length + Unknown const
        Inc(Pkt^.Len, GetInt(Pkt, 2));          //Counts of following bytes + following bytes
        Inc(Pkt^.Len, 2);                       //x0101, Unknown, constant
        ULen := GetInt(Pkt, 2) - 4;             //Length of the message + 4
        Inc(Pkt^.Len, 4);                       //Unknown seems to be constant
        Msg := GetStr(Pkt, ULen);               //The actual message text. There will be no ending NULL.
        if (Length(Msg) > 0) and Assigned(OnMessageRecv) then
          FOnMsg(Self, M_PLAIN, Msg, Uin);
      end;
    end;
    2:                                          //Complicate(new-type)
    begin
      Uin := GetStr(Pkt, GetInt(Pkt, 1));
      for c := 0 to 5 do
      begin
        if GetInt(Pkt, 2) = 5 then
        begin
          Inc(Pkt^.Len, 2);
          if GetInt(Pkt, 2) <> 0 then           //ACKTYPE: 0x0000 - This is a normal message
            Exit;
          Inc(Pkt^.Len, 16);                    //09 46 13 49 4C 7F 11 D1 82 22 44 45 53 54 00 00
          Inc(Pkt^.Len, 8);                     //TIME + RANDOM
          for i := 0 to 5 do
          begin
            if GetInt(Pkt, 2) = $2711 then      //Searching for TLV(2711) (with sources)
            begin
              Inc(Pkt^.Len, 2);                 //TLV Length
              Move(Ptr(LongWord(Pkt) + Pkt^.Len)^, chunks, 47);
              if GetInt(Pkt, 1) <> $1B then     //If this value is not present, this is not a message packet. Also, ICQ2001b does not send an ACK, SNAC(4,B), if this is not 0x1B.
                Exit;
              Inc(Pkt^.Len, 44);
              MsgType := GetInt(Pkt, 1);
              Inc(Pkt^.Len, 5);
              Msg := GetLNTS(Pkt);              //The actual message text. There will be ending NULL.

              {Sending ACK of the message}
              PktInit(@ack_pkt, 2, FSeq);
              PktSnac(@ack_pkt, $04, $0B, 0, 0);   //SNAC(x04/x0B)
              Move(Ptr(LongWord(Pkt) + TSNACSZ)^, Ptr(LongWord(@ack_pkt) + ack_pkt.Len)^, 10); //First 10 bytes of TLV(2711)
              Inc(ack_pkt.Len, 10);
              PktLStr(@ack_pkt, UIN);
              PktInt(@ack_pkt, $0003, 2);
              PktAddArrBuf(@ack_pkt, @chunks, 47); //First 47 bytes of source packet (with message)
              PktInt(@ack_pkt, 0, 4);              //00 00 00 00
              PktInt(@ack_pkt, 1, 1);              //01
              PktInt(@ack_pkt, 0, 4);              //00 00 00 00
              PktInt(@ack_pkt, 0, 1);              //00
              PktInt(@ack_pkt, $FFFFFF00, 4);      //FF FF FF 00
              PktFinal(@ack_pkt);
              FSock.SendData(ack_pkt, ack_pkt.Len);

              if (Length(Msg) > 0) and Assigned(OnMessageRecv) then
              begin
                if FDoPlain then Msg := Rtf2Txt(Msg); //Convert message from RTF to plaintext when needed
                FOnMsg(Self, MsgType, Msg, Uin);
              end;
              Exit;
            end else
              Inc(Pkt^.Len, GetInt(Pkt, 2));
          end;
        end else
          Inc(Pkt^.Len, GetInt(Pkt, 2));
      end;
    end;
    4:
    begin
      Uin := GetStr(Pkt, GetInt(Pkt, 1));
      for i := 0 to 4 do
      begin
        if GetInt(Pkt, 2) = 5 then                 //TLV(5) was found
        begin
          Inc(Pkt^.Len, 2);
          GetLInt(Pkt, 4);                         //UIN
          MsgType := GetLInt(Pkt, 2);
          Msg := GetLNTS(Pkt);
          if (Length(Msg) > 0) and Assigned(OnMessageRecv) then
          begin
            if FDoPlain then Msg := Rtf2Txt(Msg);  //Convert message from RTF to plaintext when needed
            FOnMsg(Self, MsgType, Msg, Uin);
          end;
          Exit;
        end else
          Inc(Pkt^.Len, GetInt(Pkt, 2));
      end;
    end;
  end;
end;

{Handling old type packets ICQ_FROMSRV}
procedure TICQClient.HSnac2103(Flap: TFlapHdr; Snac: TSnacHdr; Pkt: PRawPkt);
var
  FMsgType: Word;
  lpkt: TRawPkt;
  FNick, FFirst, FLast, FEmail, FCity,
  FState, FPhone, FFax, FStreet, FCellular,
  FZip, FCountry, FCompany, FDepartment,
  FPosition, FOccupation, FHomePage,
  FLang1, FLang2, FLang3, FAbout: String;
  FTimeZone: Byte;
  FPublishEmail: Boolean;
  FAge, FYear: Word;
  FGender, FMonth, FDay: Byte;
  Msg, UIN: String;
  List, List2: TStringList;
  C, i: Byte;
  WW: Word;
  FStatus: Word;
  cmd: Word;
begin
  if GetInt(Pkt, 2) = 1 then                      //TLV(1)
  begin
    Inc(Pkt^.Len, 8);
    case GetInt(Pkt, 2) of
      $4100:                                      //SRV_OFFLINEMSG
      begin
        if not Assigned(OnOfflineMsgRecv) then
          Exit;
        Inc(Pkt^.Len, 2);                         //The sequence number this packet is a response to.
        UIN := IntToStr(GetLInt(Pkt, 4));         //Source UIN
        Inc(Pkt^.Len, 6);                         //Date/time etc...
        FMsgType := GetLInt(Pkt, 2);              //The type of message sent, like URL message or the like.
        Msg := GetLNTS(Pkt);
        if FDoPlain then Msg := Rtf2Txt(Msg);     //Convert message from RTF to plaintext when needed
        FOnOffMsg(Self, FMsgType, Msg, UIN);
      end;
      $4200:                                      //All offline messages were sent, so we ACKING them
      begin
        FSeq2 := 2;
        CreateCLI_ACKOFFLINEMSGS(@lpkt, FLUIN, FSeq, FSeq2);
        FSock.SendData(lpkt, lpkt.Len);
      end;
      $da07: //SRV_META
      begin
        Inc(Pkt^.Len, 2);
        cmd := GetInt(Pkt, 2);
        case cmd of
          $c800: //SRV_METAGENERAL Channel: 2SNAC(0x15,0x3) 2010/200
          begin
            if GetInt(Pkt, 1) <> $0a then Exit;
            FNick := GetLNTS(Pkt);
            FFirst := GetLNTS(Pkt);
            FLast := GetLNTS(Pkt);
            FEmail := GetLNTS(Pkt);
            FCity := GetLNTS(Pkt);
            FState := GetLNTS(Pkt);
            FPhone := GetLNTS(Pkt);
            FFax := GetLNTS(Pkt);
            FStreet := GetLNTS(Pkt);
            FCellular := GetLNTS(Pkt);
            FZip := GetLNTS(Pkt);
            FCountry := CountryToStr(GetLInt(Pkt, 2));
            FTimeZone := GetInt(Pkt, 1);
            if GetInt(Pkt, 1) = 1 then
              FPublishEmail := True
            else
              FPublishEmail := False;
            if FInfoChain.Count > 0 then
            begin
              FLastInfoUin := FInfoChain.Strings[0];
              FInfoChain.Delete(0);
            end else
              FLastInfoUin := '0';
            if Assigned(OnUserGeneralInfo) then
              FOnUserGeneralInfo(Self, FLastInfoUin, FNick, FFirst,
                FLast, FEmail, FCity, FState, FPhone,
                FFax, FStreet, FCellular, FZip, FCountry,
                FTimeZone, FPublishEmail
              );
          end;
          $d200: //SRV_METAWORK Channel: 2SNAC(0x15,0x3) 2010/210
          begin
            if GetInt(Pkt, 1) <> $0a then Exit;
            FCity := GetLNTS(Pkt);
            FState := GetLNTS(Pkt);
            FPhone := GetLNTS(Pkt);
            FFax := GetLNTS(Pkt);
            FStreet := GetLNTS(Pkt);
            FZip := GetLNTS(Pkt);
            FCountry := CountryToStr(GetLInt(Pkt, 2));
            FCompany := GetLNTS(Pkt);
            FDepartment := GetLNTS(Pkt);
            FPosition := GetLNTS(Pkt);
            FOccupation := OccupationToStr(GetLInt(Pkt, 2));
            FHomePage := GetLNTS(Pkt);
            if Assigned(OnUserWorkInfo) then
              FOnUserWorkInfo(Self, FLastInfoUin, FCity, FState, FPhone,
                FFax, FStreet, FZip, FCountry, FCompany, FDepartment, FPosition,
                FOccupation, FHomePage
              );
          end;
          $dc00: //SRV_METAMORE Channel: 2SNAC(0x15,0x3) 2010/220
          begin
            if GetInt(Pkt, 1) <> $0a then Exit;
            FAge := GetLInt(Pkt, 2);
            if Integer(FAge) < 0 then
              FAge := 0;
            FGender := GetInt(Pkt, 1);
            FHomePage := GetLNTS(Pkt);
            FYear := GetLInt(Pkt, 2);
            FMonth := GetInt(Pkt, 1);
            FDay := GetInt(Pkt, 1);
            FLang1 := LanguageToStr(GetInt(Pkt, 1));
            FLang2 := LanguageToStr(GetInt(Pkt, 1));
            FLang3 := LanguageToStr(GetInt(Pkt, 1));
            if Assigned(OnUserInfoMore) then
              FOnUserInfoMore(Self, FLastInfoUin, FAge, FGender, FHomePage,
                FYear, FMonth, FDay, FLang1, FLang2, FLang3
              );
          end;
          $e600: //Channel: 2SNAC(0x15,0x3) 2010/230
          begin
            if GetInt(Pkt, 1) <> $0a then Exit;
            FAbout := GetLNTS(Pkt);
            if Assigned(OnUserInfoAbout) then
              FOnUserInfoAbout(Self, FLastInfoUin, FAbout);
          end;
          $eb00: //Channel: 2SNAC(21,3) 2010/235
          begin
            if GetInt(Pkt, 1) <> $0a then Exit;
            c := GetInt(Pkt, 1);        //The number of email addresses to follow. May be zero. Each consist of the following parameters:
            List := TStringList.Create;
            if c > 0 then
              for i := 0 to c - 1 do
              begin
                GetInt(Pkt, 1); //Publish email address? 1 = yes, 0 = no.
                List.Add(GetLNTS(Pkt)); //The email address.
              end;
            if Assigned(OnUserInfoMoreEmails) then
              FOnUserInfoMoreEmails(Self, FLastInfoUin, List)
            else
              List.Free;
          end;
          $f000: //Channel: 2, SNAC(21,3) 2010/240
          begin
            if GetInt(Pkt, 1) <> $0a then Exit;
            c := GetInt(Pkt, 1);
            List := TStringList.Create;
            if c > 0 then
              for i := 0 to c - 1 do
              begin
                WW := GetLInt(Pkt, 2);
                List.Add(InterestToStr(WW) + '=' + GetLNTS(Pkt))
              end;
            if Assigned(OnUserInfoInterests) then
              FOnUserInfoInterests(Self, FLastInfoUin, List)
            else
              List.Free;
          end;
          $a401, $ae01: //Channel: 2, SNAC(21,3) 2010/420 or Channel: 2, SNAC(21,3) 2010/430
          begin
            if GetInt(Pkt, 1) <> $0a then
            begin
              if Assigned(OnUserNotFound) then
                FOnUserNotFound(Self);
              Exit;
            end;
            Inc(Pkt^.Len, 2);                   //Length of the following data.
            UIN := IntToStr(GetLInt(Pkt, 4));   //The user's UIN.
            FNick := GetLNTS(Pkt);              //The user's nick name.
            FFirst := GetLNTS(Pkt);             //The user's first name.
            FLast := GetLNTS(Pkt);              //The user's last name.
            FEmail := GetLNTS(Pkt);             //The user's email address.
            Inc(Pkt^.Len, 1);                   //Publish email address? 1 = yes, 0 = no.
            FStatus := GetLInt(Pkt, 2);         //0 = Offline, 1 = Online, 2 = not Webaware.
            FGender := GetInt(Pkt, 1);          //The user's gender. 1 = female, 2 = male, 0 = not specified.
            FAge := GetInt(Pkt, 1);             //The user's age.
            if Assigned(OnUserFound) then
              FOnUserFound(Self, UIN, FNick, FFirst, FLast, FEmail, FStatus, FGender, FAge, cmd = $ae01);
          end;
          $6603:
          begin
            if GetInt(Pkt, 1) <> $0a then
            begin
              if Assigned(OnUserNotFound) then
                FOnUserNotFound(Self);
              Exit;
            end;
            UIN := IntToStr(GetLInt(Pkt, 4));   //The user's UIN.
            if Assigned(OnUserFound) then
              FOnUserFound(Self, UIN, '', '', '', '', 0, 0, 0, True);
          end;
          $fa00:
          begin
            if GetInt(Pkt, 1) <> $0a then Exit;
            List := TStringList.Create;
            List.Duplicates := dupIgnore;            
            c := GetInt(Pkt, 1);                             //The number of background items to follow. May be zero. Each background item consists of the following two parameters
            if c > 0 then
              for i := 0 to c - 1 do
              begin
                WW := GetLInt(Pkt, 2);                       //The group this background is in, according to a table.
                List.Add(PastToStr(WW) + '=' + GetLNTS(Pkt)) //A longer description of this background item.
              end;
            List2 := TStringList.Create;
            List2.Duplicates := dupIgnore;            
            c := GetInt(Pkt, 1);                             //The number of affiliations to follow. May be zero. Each affiliation consists of the following parameters:
            if c > 0 then
              for i := 0 to c - 1 do
              begin
                WW := GetLInt(Pkt, 2);                       //The group this affiliation is in, according to a table.
                List2.Add(AffiliationToStr(WW) + '=' + GetLNTS(Pkt)) //A longer description of the affiliation.
              end;
            if Assigned(OnUserInfoBackground) then
              FOnUserInfoBackground(Self, FLastInfoUin, List, List2)
            else
            begin
              List.Free;
              List2.Free;
            end;
          end;
        end;
      end;
    end;
  end;
end;

{Handling packet with status changes}
procedure TICQClient.HSnac030B(Flap: TFlapHdr; Snac: TSnacHdr; Pkt: PRawPkt);

var
  c, i: Word;
  UIN: String;
  Status: LongWord;
begin
  UIN := GetStr(Pkt, GetInt(Pkt, 1));
  Inc(Pkt^.Len, 2);
  c := GetInt(Pkt, 2);
  for i := 0 to c - 1 do
  begin
    if GetInt(Pkt, 2) = $06 then
    begin
      Inc(Pkt^.Len, 2);
      Status := GetInt(Pkt, 4);
      if Assigned(OnStatusChange) then
        FOnStatusChange(Self, UIN, Status);
      Exit;
    end else
      Inc(Pkt^.Len, GetInt(Pkt, 2));
  end;
end;

{Handling AddedYou packet}
procedure TICQClient.HSnac131C(Flap: TFlapHdr; Snac: TSnacHdr; Pkt: PRawPkt);
var
  T: Word;
  UIN: String;
begin
  Inc(Pkt^.Len, 2);
  GetTLVInt(Pkt, T);
  if T <> 1 then Exit;
  UIN := GetLStr(Pkt);
  if Assigned(OnAddedYou) then
    FOnAddedYou(Self, UIN);
end;

{Authorization request, we are automaticly authorizing the user}
procedure TICQClient.HSnac1319(Flap: TFlapHdr; Snac: TSnacHdr; Pkt: PRawPkt);
var
  FUin: String;
  FReason: String;
  opkt: TRawPkt;
begin
  Inc(Pkt^.Len, 8);
  FUin := GetLStr(Pkt);
  FReason := GetStr(Pkt, Swap16(GetInt(Pkt, 2)));
  CreateCLI_AUTHORIZE(@opkt, StrToInt(FUin), 1, '', FSeq);
  FSock.SendData(opkt, opkt.Len);
end;

{This packet contains your complete server side contact list.}
procedure TICQClient.HSnac1306(Flap: TFlapHdr; Snac: TSnacHdr; Pkt: PRawPkt);
var
  GroupIdents: TStringList;
  UINList: TList;
  procedure ReadChunk;
  var
    Len: Word;
    FGroup: ShortString;
    CTag, CId, CType: Word;
    TLen: Word;
    TType: Word;
    FNick: ShortString;
    lpEntry: PUINEntry;
  begin
    FGroup := GetWStr(Pkt);             //The name of the group.
    CTag := GetInt(Pkt, 2);             //This field seems to be a tag or marker associating different groups together into a larger group such as the Ignore List or 'General' contact list group, etc.
    CId := GetInt(Pkt, 2);              //This is a random number generated when the user is added to the contact list, or when the user is ignored.
    CType := GetInt(Pkt, 2);            //This field seems to indicate what type of group this is.
    Len := GetInt(Pkt, 2);              //The length in bytes of the following TLVs.
    FNick := '';
    while Integer(Len) > 0 do
    begin
      TType := GetInt(Pkt, 2);          //TLV Type
      TLen := GetInt(Pkt, 2);           //TLV Len
      if TType = $0131 then
      begin
        FNick := GetStr(Pkt, TLen);
      end else
        Inc(Pkt^.Len, TLen);            //Skip this TLV
      Dec(len, tlen + 4);               //TLV length + 2 bytes type + 2 bytes length
    end;

    //Group header
    if (FGroup <> '') and (CType = 1) and (CTag <> 0) and (CId = 0) then
      GroupIdents.Values[IntToStr(CTag)] := FGroup;

    //UIN entry
    if (CType = 0) or (CType = 2) or (CType = 3) or (CType = $e) then
    begin
      GetMem(lpEntry, SizeOf(lpEntry^));
      lpEntry^.UIN := StrToInt(FGroup);
      lpEntry^.Nick := FNick;
      lpEntry^.CType := Ctype;
      lpEntry^.CTag := CTag;
      UINList.Add(lpEntry);
    end;
  end;
var
  count, T: Word;
  i: Word;
begin
  GetTLVInt(Pkt, T); if T <> 6 then Exit;
  Inc(Pkt^.Len, 4);                     //02 00 02 00 - UNKNOWNs
  count := GetInt(Pkt, 2);              //Total count of following groups. This is the size of the server side contact list and should be saved and sent with CLI_CHECKROSTER.
  if count < 1 then Exit;
  GroupIdents := TStringList.Create;
  UINList := TList.Create;
  for i := 0 to count - 1 do
    ReadChunk;

  if UINList.Count > 0 then
    for i := 0 to UINList.Count - 1 do
      PUINEntry(UINList.Items[i])^.CGroup := GroupIdents.Values[IntToStr(PUINEntry(UINList.Items[i])^.CTag)];
  GroupIdents.Free;

  if Assigned(OnServerListRecv) then
    FOnServerListRecv(Self, UINList)
  else
    DestroyUINList(UINList);
end;

{This packet contains ack to message you've sent.}
procedure TICQClient.HSnac040b(Flap: TFlapHdr; Snac: TSnacHdr; Pkt: PRawPkt);
var
  RetCode: Word;
  RetAcc: Byte;
  RetMsg: String;
  FUIN: String;
begin
  Inc(Pkt^.Len, 4);                     //Time
  RetCode := GetInt(Pkt, 2);            //Random ID
  Inc(Pkt^.Len, 4);                     //Other data :)
  FUIN := GetLStr(Pkt);                 //User's UIN
  Inc(Pkt^.Len, 2);                     //00 03
  Inc(Pkt^.Len, 45);                    //Skip 50 bytes of packet
  Inc(Pkt^.Len, 2);                     //msg-type, msg-flags
  RetAcc := GetInt(Pkt, 1);             //Accept type
  Inc(Pkt^.Len, 3);
  if (RetAcc <> ACC_NORMAL) and (RetAcc <> ACC_NO_OCCUPIED) and
     (RetAcc <> ACC_NO_DND) and (RetAcc <> ACC_AWAY) and
     (RetAcc <> ACC_NA) and (RetAcc <> ACC_CONTACTLST) then Exit;
  if RetAcc <> ACC_NORMAL then
  begin
    RetMsg := GetLNTS(Pkt);
  end else
    RetMsg := '';
  if Assigned(OnAdvancedMsgAck) then
    FOnAdvMsgAck(Self, FUIN, RetCode, RetAcc, RetMsg);
end;

procedure TICQClient.FTOnConnectError(Sender: TObject);
begin
  if Assigned(OnConnectionFailed) then
    FOnConnectionFailed(Self);
  FConnected := False;
end;

procedure TICQClient.FTOnDisconnect(Sender: TObject);
begin
  if FConnected then
    if Assigned(OnConnectionFailed) then
      FOnConnectionFailed(Self);
  FConnected := False;
end;

procedure TICQClient.FTOnPktParse(Sender: TObject; Buffer: Pointer; BufLen: LongWord);
begin
  if Assigned(OnPktParse) then
    FOnPktParse(Self, Buffer, BufLen, False);
end;

procedure TICQCLient.SetContactList(Value: TStrings);
begin
  FContactLst.Assign(Value);
end;

procedure TICQCLient.SetVisibleList(Value: TStrings);
begin
  FVisibleLst.Assign(Value);
end;

procedure TICQCLient.SetInvisibleList(Value: TStrings);
begin
  FInvisibleLst.Assign(Value);
end;


procedure Register;
begin
  RegisterComponents('Standard', [TICQClient]);
end;



end.
