unit d3drmwnd;

{D3DRMWindow - a delphi component for interfacing to Direct3D functions}
{(c) February 1997 by Luc Cluitmans}
{This source is provided free as an example of using Direct3D in Delphi.
 No guarantees; I am not responsible for nasty things that may happen to you
 or your computer by using this code}
{Sorry for the lack of documentary comments :-)}

interface

uses
  OLE2, Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  D3DRMObj, D3DTypes, D3DRMDef, DDraw, D3D, D3Drm, dxtools, D3DrmWin, D3Dcaps;

type
  TD3DRMWindow = class(TCustomControl)
  private
    { Private declarations }
    FD3DRM:     IDirect3DRM;
    FClip:      IDirectDrawClipper;
    FDev:       IDirect3DRMDevice;
    FWDev:      IDirect3DRMWinDevice;

    FCamera:    IDirect3DRMFrame;
    FScene:     IDirect3DRMFrame;
    FView:      IDirect3DRMViewport;
    FSubScene:  IDirect3DRMFrame;

    FFailCount: Integer;
    FRenderFail: Integer;
    FInitialized: Boolean;
    FFrameCount: Longint;
    FBPP: Integer; {bits per pixel}
    FCurrentDriver: PGUID;

    FOnBuildScene: TNotifyEvent;

    FDriverIndex: Integer;
    FDriverGUID: PGUID;

    function GetDDBD: DWORD;
    function GetDriverNames(idx: Integer): string;
    function GetDriverCount: Integer;
    function GetDriverName: string;
    procedure SetDriverName(Value: string);
    procedure SetDriver(DriverIdx: Integer);
  protected
    { Protected declarations }
    procedure Paint; override;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure WMSize(var Msg: TWMSize); message WM_SIZE;
    procedure CMWantSpecialKey(var Msg: TCMWantSpecialKey); message CM_WANTSPECIALKEY;
    procedure Loaded; override;
    
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure IdleProc(Sender: TObject; var Done: Boolean);
              {assign this to your Application's OnIdle handler}
    procedure RenderLoop;
    function  Failed: Boolean; {when true, something went wrong seriously}
    property  Initialized: Boolean read FInitialized;
    procedure RebuildDevice(bForce: Boolean);
              {when bForce is true, the device is rebuild even when the 
               window's size has not changed}
    procedure ActivateCallback(Msg: TMessage);
              {couple this to the form's WM_ACTIVATE handler}
    procedure InitializeDevice;
    procedure CenterScene; {centers the Scene}
    procedure LoadFile(fnm: string); {loads mesh from *.x file}
    procedure LoadFrameFile(fnm: string);  {loads frame from *.x file}

    property D3DRM:    IDirect3DRM read FD3DRM;
    property Device:   IDirect3DRMDevice read FDev;
    property Camera:   IDirect3DRMFrame read FCamera;
    property Viewport: IDirect3DRMViewport read FView;
    property Scene:    IDirect3DRMFrame read FScene;
    property SubScene: IDirect3DRMFrame read FSubScene;
    property FrameCount: Longint read FFrameCount;
    property BPP:      Integer read FBPP;
    property DDBD:     DWORD read GetDDBD;
    property DriverCount: Integer read GetDriverCount;
    property DriverNames[idx: Integer]:string read GetDriverNames;
    property DriverName: string read GetDriverName write SetDriverName;
  published
    { Published declarations }
    property Height;
    property Width;
    property Left;
    property Top;
    property Align;
    property DragCursor;
    property DragMode;
    property Cursor;
    property ParentShowHint;
    property ShowHint;
    property Hint;
    property PopupMenu;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
    property Visible;
    property Enabled;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property TabStop;
    property TabOrder;
    property HelpContext;
    

    property OnBuildScene: TNotifyEvent read FOnBuildScene write FOnBuildScene;
             {this is the place to put the initial contents in the (Sub)Scene}
  end;

procedure Register;

implementation

{driver enumeration stuff}

type
  TD3DDriverInfo = record
    id:                TGUID;
    DeviceDescription: string;
    DeviceName:        string;
    HWDeviceDesc:      D3DDeviceDesc ;
    HELDeviceDesc:     D3DDeviceDesc ;
  end;

const
  MAXDRIVERS = 8;

var
  {a few global variables storing driver info}
  _D3Ddrivers: array[0.. MAXDRIVERS-1] of TD3DDriverInfo;
  _D3DdriverCount: Integer;
  _bDriversInitialized: Boolean;

function _EnumCallBack(lpGuid: PGUID ;
      lpDeviceDescription: LPSTR ; lpDeviceName: LPSTR ;
      const lpD3DHWDeviceDesc: D3DDeviceDesc ;
      const lpD3DHELDeviceDesc: D3DDeviceDesc ;
      lpUserArg: Pointer ): HRESULT ; stdcall ;
var
  dev: ^D3DDeviceDesc;
  DDBD: DWORD;
begin
  dev := @lpD3DHWDeviceDesc;
  DDBD := DWORD(lpUserArg);
  if Integer(lpD3DHWDeviceDesc.dcmColorModel) = 0 then dev := @lpD3DHELDeviceDesc;
  if (dev^.dwDeviceRenderBitDepth and DDBD) <> 0 then
  begin
    {current bit depth is supported by this driver}
    with _D3DDrivers[_D3DdriverCount] do
    begin
      Move(lpGUID^, id, SizeOf(TGUID));
      Move(lpD3DHWDeviceDesc, HWDeviceDesc, SizeOf(D3DDeviceDesc));
      Move(lpD3DHELDeviceDesc, HELDeviceDesc, SizeOf(D3DDeviceDesc));
      DeviceDescription := StrPas(lpDeviceDescription);
      DeviceName := StrPas(lpDeviceName);
    end;
    Inc(_D3DdriverCount)
  end;
  Result := D3DENUMRET_OK;
  if _D3DDriverCount >= MAXDRIVERS then Result := D3DENUMRET_CANCEL;
end;


procedure _InitializeDrivers(D3DRMWnd: TD3DRMWindow);
var
  d3d: IDirect3D;
  dd: IDirectDraw;
begin
  if not _bDriversInitialized then
  begin
    DXCheck(DirectDrawCreate(nil, dd, nil));
    try
      DXCheck(dd.QueryInterface(IID_IDirect3D, d3d));
      try
        _bDriversInitialized := True;
        DXCheck(d3d.EnumDevices(_EnumCallback, Pointer(D3DRMWnd.DDBD)));
      finally
        COMRelease(IUnknown(d3d));
      end;
    finally
      COMRelease(IUnknown(dd));
    end;
  end;
end;

{end of driver enumeration stuff}

procedure Register;
begin
  RegisterComponents('Samples', [TD3DRMWindow]);
end;

constructor TD3DRMWindow.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width := 65;
  Height := 65;
  Color := clBtnShadow;
  DXCheck(Direct3DRMCreate(FD3DRM));
  FInitialized := False; {indicates that no D3D device was created yet}
end;

procedure TD3DRMWindow.Loaded; {second half of constructor...}
var
  lightframe: IDirect3DRMFrame;
  light1, light2: IDirect3DRMLight;
  dc: HDC;
begin
  if not (csDesigning in ComponentState) then
  begin
    {retrieve bits per pixel}
    dc := GetDC(Handle);
    FBPP := GetDeviceCaps(dc, BITSPIXEL);
    ReleaseDC(Handle, dc);
    _InitializeDrivers(Self);

    {create scene}
    DXCheck(DirectDrawCreateClipper(0, FClip, nil));
    DXCheck(FClip.SetHWnd(0, Handle));
    DXCheck(FD3DRM.CreateFrame(nil, FScene));
    DXCheck(FD3DRM.CreateFrame(FScene, FCamera));
    DXCheck(FCamera.SetPosition(FScene, 0.0, 0.0, 0.0));
    DXCheck(FD3DRM.CreateFrame(FScene, lightframe));
    try
      {create lights}
      DXCheck(FD3DRM.CreateLightRGB(D3DRMLIGHT_DIRECTIONAL, 0.9, 0.9, 0.9, light1));
      try
        DXCheck(lightframe.AddLight(light1));
      finally
        COMRelease(IUnknown(light1));
      end;
      DXCheck(FD3DRM.CreateLightRGB(D3DRMLIGHT_AMBIENT, 0.1, 0.1, 0.1, light2));
      try
        DXCheck(lightframe.AddLight(light2));
      finally
        COMRelease(IUnknown(light2));
      end;
      DXCheck(lightframe.SetPosition(FScene, 2.0, 0.0, 22.0));
    finally
      COMRelease(IUnknown(lightframe));
    end;

    {create 'world' frame that will contain the user-defined objects}
    DXCheck(FD3DRM.CreateFrame(FScene, FSubScene));

    {set positions and orientations of frames}
    DXCheck(FCamera.SetPosition(FScene, 0.0, 0.0, -15.0));
    DXCheck(FCamera.SetOrientation(FScene, 0.0, 0.0, 1.0, 0.0, 1.0, 0.0));
      
    DXCheck(FSubScene.SetPosition(FScene, 0.0, 0.0, 0.0));
    DXCheck(FSubScene.SetOrientation(FScene, 0.0, 0.0, 1.0, 0.0, 1.0, 0.0));

    {create device and vieport}
    InitializeDevice;

    FInitialized := True;

    {user callback for getting content of scene}
    if Assigned(FOnBuildScene) then FOnBuildScene(Self);
  end;
end;


destructor TD3DRMWindow.Destroy;
begin
  if not (csDesigning in ComponentState) then
  begin
    if Assigned(FView) then COMRelease(IUnknown(FView));
    if Assigned(FWDev) then COMRelease(IUnknown(FWDev));
    if Assigned(FDev) then COMRelease(IUnknown(FDev));
    COMRelease(IUnknown(FSubScene));
    COMRelease(IUnknown(FCamera));
    COMRelease(IUnknown(FScene));
    COMRelease(IUnknown(FClip));
  end;
  COMRelease(IUnknown(FD3DRM));
  inherited Destroy;
end;

procedure TD3DRMWindow.Paint;
var
  x0, y0: Integer;
  s: string;
  r: HResult;
begin
  if (csDesigning in ComponentState) or (not Assigned(FWDev)) or Failed or
     not Initialized then
  with Canvas do
  begin
    Brush.Color := Color; 
    Font.Color := clWhite;
    Pen.Color := clRed;
    MoveTo(0, 0);
    LineTo(Width-1, Height-1);
    MoveTo(Width-1, 0);
    LineTo(0, Height-1);
    if csDesigning in ComponentState then
    begin
      s := '(designing)';
    end
    else if Failed then
    begin
      s := '(Direct3D failure)';
    end
    else
    begin
      s := '(no 3D device)';
    end;
    x0 := (Width  - TextWidth(s)) div 2;
    y0 := (Height - TextHeight(s)) div 2;
    TextOut(x0, y0, s);
  end
  else
  begin
    if COMFailed(r, FWDev.HandlePaint(Canvas.Handle)) then
    begin
      Inc(FFailCount);
      DXCheck(r);
    end;
  end;
end;

procedure TD3DRMWindow.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  Params.Style := Params.Style or WS_CLIPCHILDREN;
end;

procedure TD3DRMWindow.WMSize(var Msg: TWMSize);
begin
  if not (csDesigning in ComponentState) and Assigned(FDev) then
  begin
    RebuildDevice(False);
  end;
  inherited;
end;

procedure TD3DRMWindow.IdleProc(Sender: TObject; var Done: Boolean);
var
  ws: TWindowState;
begin
  {make sure we only get called when the app is truly idle by processing all
   pending messages}
  Application.ProcessMessages;
  ws := wsNormal;
  if Owner is TForm then ws := (Owner as TForm).WindowState;
  if (ws<>wsMinimized) and
     Assigned(FDev) and
     (FRenderFail<2) and
     Showing and
     Initialized and
     not Application.Terminated then
  begin
    Done := False;
    try
      RenderLoop;
    except
      on E: Exception do
      begin
        Inc(FRenderFail);
        raise;
      end;
    end;
    Inc(FFrameCount);
  end
  else
  begin
    Done := True;
  end;
end;

procedure TD3DRMWindow.RenderLoop;
begin
  if Assigned(FDev) and Assigned(FView) and Assigned(FScene) then
  begin
    DXCheck(FScene.Move(1.0));
    DXCheck(FView.Clear);
    DXCheck(FView.Render(scene));
    DXCheck(FDev.Update);
  end;
end;

function  TD3DRMWindow.Failed: Boolean; {when true, something went seriously wrong}
begin
  Result := (FRenderFail>2) or (FFailCount>2);
end;

procedure TD3DRMWindow.ActivateCallback(Msg: TMessage);
begin
  if Assigned(FWDev) then
  begin
    DXCheck(FWDev.HandleActivate(Msg.WParam));
  end;
end;

procedure TD3DRMWindow.InitializeDevice;
var
  w, h: Integer;
  r : HResult;
begin
  DXCheck(FD3DRM.CreateDeviceFromClipper(FClip, FDriverGUID, Width, Height, FDev));
  DXCheck(FDev.QueryInterface(IID_IDirect3DRMWinDevice, FWDev));
  w := FDev.GetWidth;
  h := FDev.GetHeight;
  if COMFailed(r, FD3DRM.CreateViewport(FDev, FCamera, 0, 0, w, h, FView)) then
  begin
    COMRelease(IUnknown(FWDev));
    COMRelease(IUnknown(FDev));
    DXCheck(r);
  end;
  if COMFailed(r, FView.SetBack(5000.0)) then
  begin
    COMRelease(IUnknown(FView));
    COMRelease(IUnknown(FWDev));
    COMRelease(IUnknown(FDev));
    DXCheck(r);
  end;

  {set render quality}
  DXCheck(FDev.SetQuality(D3DRMRENDER_FLAT));
  {render quality part 2 - standard incantation}
  if FBPP = 1 then
  begin
    DXCheck(FDev.SetShades(4));
    DXCheck(FD3DRM.SetDefaultTextureShades(4));
  end
  else if FBPP = 16 then
  begin
    DXCheck(FDev.SetShades(32));
    DXCheck(FD3DRM.SetDefaultTextureColors(64));
    DXCheck(FD3DRM.SetDefaultTextureShades(32));
  end
  else
  if (FBPP = 24) or (FBPP = 32) then
  begin
    DXCheck(FDev.SetShades(256));
    DXCheck(FD3DRM.SetDefaultTextureColors(64));
    DXCheck(FD3DRM.SetDefaultTextureShades(256));
  end;

end;

procedure TD3DRMWindow.RebuildDevice(bForce: Boolean);
var
  oldDither: BOOL;
  oldQuality: D3DRMRENDERQUALITY;
  oldShades: DWORD;
  w, h, w0, h0: Integer;
  r: HRESULT;
begin
  w0 := Width;
  h0 := Height;
  if Assigned(FDev) and ((FDev.GetWidth<>w0) or (FDev.GetHeight<>h0) or bForce) then
  begin
    oldDither := FDev.GetDither;
    oldQuality := FDev.GetQuality;
    oldShades := FDev.GetShades;

    COMRelease(IUnknown(FView));
    COMRelease(IUnknown(FWDev));
    COMRelease(IUnknown(FDev));
    
    DXCheck(FD3DRM.CreateDeviceFromClipper(FClip, FDriverGUID, w0, h0, FDev));
    DXCheck(FDev.QueryInterface(IID_IDirect3DRMWinDevice, FWDev));
    w := FDev.GetWidth;
    h := FDev.GetHeight;
    if COMFailed(r, FD3DRM.CreateViewport(FDev, FCamera, 0, 0, w, h, FView)) then
    begin
      COMRelease(IUnknown(FWDev));
      COMRelease(IUnknown(FDev));
      DXCheck(r);
    end;
    if COMFailed(r, FView.SetBack(5000.0)) then
    begin
      COMRelease(IUnknown(FView));
      COMRelease(IUnknown(FWDev));
      COMRelease(IUnknown(FDev));
      DXCheck(r);
    end;
    DXCheck(FDev.SetQuality(oldQuality));
    DXCheck(FDev.SetDither(oldDither));
    DXCheck(FDev.SetShades(oldShades));
  end;
end;

procedure TD3DRMWindow.CMWantSpecialKey(var Msg: TCMWantSpecialKey);
begin
  if Msg.CharCode in [VK_LEFT, VK_RIGHT, VK_UP, VK_DOWN] then
  begin
    Msg.Result := 1;
  end;
end;

procedure TD3DRMWindow.CenterScene; {centers the Scene and positions the camera}
var
  mesh: IDirect3DRMMeshBuilder;  
  box:  D3DRMBox;
  x, y, z, s: Single;
begin
  if Initialized and (not Failed) and Assigned(FDev) then
  begin
    DXCheck(FD3DRM.CreateMeshBuilder(mesh));
    try
      DXCheck(mesh.AddFrame(SubScene));
      DXCheck(mesh.GetBox(box));
    finally
      COMRelease(IUnknown(mesh));
    end;
    x := (box.max.x + box.min.x)/2.0;
    y := (box.max.y + box.min.y)/2.0;
    z := (box.max.z + box.min.z)/2.0;
    s := box.max.x - box.min.x;
    if (box.max.y - box.min.y) > s then s := box.max.y - box.min.y;
    if (box.max.z - box.min.z) > s then s := box.max.z - box.min.z;
    s := (s + 1.0) * 2.0;
    DXCheck(SubScene.SetOrientation(FScene, 0.0, 0.0, 1.0, 0.0, 1.0, 0.0));
    DXCheck(SubScene.SetPosition(Scene, -x, -y, -z));
    DXCheck(FCamera.SetPosition(FScene, -0.3*s, 0.3*s, -s));
    DXCheck(FCamera.SetOrientation(FScene, 0.3, -0.3, 1.0, 0.0, 1.0, 0.0));
  end;
end;

procedure TD3DRMWindow.LoadFile(fnm: string);
var
  spherebuilder: IDirect3DRMMeshBuilder;
  orgcur: TCursor;
begin
  if Initialized and (not Failed) and Assigned(FDev) then
  begin
    DXCheck(FScene.DeleteChild(FSubScene));
    COMRelease(IUnknown(FSubScene));
    
    DXCheck(FD3DRM.CreateFrame(FScene, FSubScene));
    DXCheck(FSubScene.SetOrientation(FScene, 0.0, 0.0, 1.0, 0.0, 1.0, 0.0));
    DXCheck(FD3DRM.CreateMeshBuilder(spherebuilder));

    orgcur := Screen.Cursor;
    Screen.Cursor := crHourGlass;
    try
      DXCheck(spherebuilder.Load(Pointer(fnm), nil, D3DRMLOAD_FROMFILE, nil, nil));
      DXCheck(spherebuilder.Scale(2.0, 2.0, 2.0));
      DXCheck(FSubScene.AddVisual(spherebuilder));
    finally
      Screen.Cursor := orgcur;
    end;

    COMRelease(IUnknown(spherebuilder));
    CenterScene;
  end;
end;

procedure TD3DRMWindow.LoadFrameFile(fnm: string);
var
  orgcur: TCursor;
begin
  if Initialized and (not Failed) and Assigned(FDev) then
  begin
    DXCheck(FScene.DeleteChild(FSubScene));
    COMRelease(IUnknown(FSubScene));
    
    DXCheck(FD3DRM.CreateFrame(FScene, FSubScene));
    DXCheck(FSubScene.SetOrientation(FScene, 0.0, 0.0, 1.0, 0.0, 1.0, 0.0));

    orgcur := Screen.Cursor;
    Screen.Cursor := crHourGlass;
    try
      DXCheck(FSubScene.Load(Pointer(fnm), nil, D3DRMLOAD_FROMFILE, nil, nil));
    finally
      Screen.Cursor := orgcur;
    end;

    CenterScene;
  end;
end;

function TD3DRMWindow.GetDDBD: DWORD;
begin
  case FBPP of
    1: Result := DDBD_1;
    2: Result := DDBD_2;
    4: Result := DDBD_4;
    8: Result := DDBD_8;
    16: Result := DDBD_16;
    24: Result := DDBD_24;
    32: Result := DDBD_32;
  else
    Result := 0;
  end;
end;

function TD3DRMWindow.GetDriverNames(idx: Integer): string;
begin
  if (idx<0) or (idx>=_D3DdriverCount) then
    raise Exception.Create('Invalid driver name index');
  Result := _D3Ddrivers[idx].DeviceName;
end;

function TD3DRMWindow.GetDriverCount: Integer;
begin
  Result := _D3DdriverCount;
end;

procedure TD3DRMWindow.SetDriverName(Value: string);
var
  i, idx: Integer;
begin
  if CompareText(DriverName, Value) <> 0 then
  begin
    idx := -1;
    for i := 0 to _D3DdriverCount-1 do
    begin
      if CompareText(_D3Ddrivers[i].DeviceName, Value) = 0 then
      begin
        idx := i;
      end;
    end;
    SetDriver(idx);
  end;
end;

procedure TD3DRMWindow.SetDriver(DriverIdx: Integer);
var
  pid: PGUID;
begin
  if (DriverIdx>=0) and (DriverIdx<_D3DdriverCount) then
  begin
    pid := @_D3Ddrivers[DriverIdx].id;
    FDriverIndex := DriverIdx;
  end
  else
  begin
    {default driver}
    pid := nil;
    FDriverIndex := -1;
  end;
  FDriverGUID := pid;
  RebuildDevice(True);
end;

function TD3DRMWindow.GetDriverName: string;
begin
  if (FDriverIndex>=0) and (FDriverIndex<_D3DdriverCount) then
  begin
    Result := _D3Ddrivers[FDriverIndex].DeviceName;
  end
  else
  begin
    Result := '';
  end;
end;

end.
 
