unit Calend;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, Buttons, Grids, StdCtrls, DB, DBTables;

(***************************************************************************
Calendrio
Autor: Reinaldo M.R. Alves
Data: 14/11/96
ltima alterao: 02/01/97

TCalendario
-----------

Propriedades:


    DestacarDia: TDiaDestaque => leitura/escrita do dia da semana que
                                 deve ser destacado. Valor -1 no destaca
                                 nenhum.
    InicioDaSemana: TDiasDaSemana => leitura/escrita do dia que comea a
                                     semana
    VisibleBarra: Boolean     => Define a visibilidade da barra que
                                 apresenta mes e ano, e os botes.
    dtDia: Byte           => leitura/escrita do dia
    dtMes: Byte           => leitura/escrita do mes
    dtAno: Byte           => leitura/escrita do Ano
    Data: TDateTime       => leitura/escrita da data. Apenas run time.
    ShowFeriados: Boolean => Se True apresenta a descrio do feriado
                             no vdeo. Semelhante ao ShowHint;
    FeriadosFont: TFont   => permite destacar feriados e domingo
    Feriados: TStrings    => Lista de feriados.
                           Exemplo:
                           with Calendario1.Feriados do
                                begin
                        {1}     Add('25/12/00Natal');
                        {2}     Add('19/03/00Meu aniversrio'); -> Deveria ser um feriado internacional, no?
                        {3}     Add('01/01/00Primeiro dia do ano');
                        {4}     Add('12/12/96Feriado Frustrado de 100 anos de BH');
                                end;
                           Formato da lista:
                           dd/mm/00+Nome Feriado -> O feriado  destacado em todos os anos. Ex.: 1,2,3
                           dd/mm/yy+Nome Feriado -> O feriado  destacado somente no ano especificado. Ex.: 4

Mtodos:

    Alteram o dia, semana, mes e ano atual.
    AnteriorMes
    ProximoMes
    AnteriorAno
    ProximoAno

    Atualiza        => Redesenha o calendrio
Eventos:

        OnDblClick  => Duplo clique
        OnChange    => Executado na alterao de data.

        OnHintDia: TNotifyFeriado; => Semelhante ao OnHint.  executado
                    a medida que o mouse  movido sobre o calendrio,
                    informando se o dia  um feriado ou no.

        TNotifyFeriado = procedure (Sender: TObject; Data: TDateTime; Descricao: string; Feriado: Boolean) of object;
        Data      => Data que o mouse est apontando. Se no for uma data vlida retorna 1/1/1.
        Descricao => Descrio do feriado, '' caso contrrio.
        Feriado   => True se for feriado.


TDBCalendario
-------------

Possibilita realizar anotaes dirias.

Propriedades:

           CampoMemo: string  => Nome do campo memo para anotao
           CampoData: string  => Nome do campo data para marcao
           DataSource: TDataSource

Mtodos:

        EditaDia( Data: TDateTime) => abre a janela de edio para
                                      a data informada.

        Obs.: Clique com o boto da direita tambm edita a data.
Eventos:

        OnDrawDia: TNotifyDrawDia => Executado antes de desenhar
                                     cada dia. Permite controlar
                                     o fonte que ser usado;
                                     principalmente para datas
                                     que tm anotaes.

        TNotifyDrawDia = procedure (Sender: TObject; Data: TDateTime; Anotacao: Boolean; Fonte: TFont) of object;
        Data: Data que ser desenhada;
        Anotao: Se True esta data tem anotaes;
        Fonte: Permite alterar a fonte.

Usando com um TTable: A tabela dever ter um campo data com ndice e um
campo memo. A propriedade IndexName dever conter o nome do ndice
da data.

Usando com TQuery: Faa um SQL como no exemplo:
SELECT * FROM Agenda
WHERE Data= :DataFil
Crie no params um paramtro DataFil do tipo Date.
***************************************************************************}

{***************************************************************************
Este componente foi desenvolvido apartir do calendrio que acompanha o
Delphi 1.0.

Agradecimentos:
A Bruno Sonnino que identificou um erro na primeira verso do componente,
indicou o caminho para a construo do segundo e por todas a suas dicas.
A Arnaldo Braun por suas dicas.
A Eric Paschoalick por suas dicas.
A Walter Campelo que colocou seu componente dbgrid3d a disposicao, onde
aprendi tcnicas fundamentais p/ a criao deste.

Este componente  de domnio pblico. Sendo proibido a venda do mesmo
sem a autorizao de seu autor.
O autor no se responsabiliza por qualquer dano causado pelo uso do
mesmo.
Qualquer bug apresentado, crtica ou necessidade de melhorias, por favor
fale-me.

Reinaldo
alfa@stinet.com.br
Belo Horizonte
Brasil

O caminho de DEUS  perfeito. A palavra do SENHOR  pura. Ele  um escudo
para todos os que nele se refugiam. Sl 18:30
****************************************************************************)

{***************************************************************************
DICAS:
        O melhor mtodo de traduzir os nomes dos meses que conheco consiste
em alterar os vetores abaixo. Alterando-os, todas as funes de data so
afetadas.

LongMonthNames[1]  := 'Janeiro';
LongMonthNames[2]  := 'Fevereiro';
LongMonthNames[3]  := 'Maro';
LongMonthNames[4]  := 'Abril';
LongMonthNames[5]  := 'Maio';
LongMonthNames[6]  := 'Junho';
LongMonthNames[7]  := 'Julho';
LongMonthNames[8]  := 'Agosto';
LongMonthNames[9]  := 'Setembro';
LongMonthNames[10] := 'Outubro';
LongMonthNames[11] := 'Novembro';
LongMonthNames[12] := 'Dezembro';

ShortDayNames[1] := 'Dom';
ShortDayNames[2] := 'Seg';
ShortDayNames[3] := 'Ter';
ShortDayNames[4] := 'Qua';
ShortDayNames[5] := 'Qui';
ShortDayNames[6] := 'Sex';
ShortDayNames[7] := 'Sab';

        Eu coloco este cdigo na seo initialization no meu arquivo de
biblioteca, de forma q ele  executado automaticamente na entrada do
sistema.

        Fica assim:

initialization

LongMonthNames[1]  := 'Janeiro';
LongMonthNames[2]  := 'Fevereiro';
.
.
.
end. <=== todo PAS termina com ele, OK?

Para preencher a lista de feriados:

     Feriados.Add( FormatDateTime( 'dd/mm/00', Data1)+'Natal');
     Feriados.Add( FormatDateTime( 'dd/mm/yy', Data2)+'Luto pela morte de ...');
***************************************************************************}

type
  TDiasDaSemana = 0..6;
  TDiaDestaque = -1..6;
  TNotifyFeriado = procedure (Sender: TObject; Data: TDateTime; Descricao: string; Feriado: Boolean) of object;
  TNotifyDrawDia = procedure (Sender: TObject; Data: TDateTime; Anotacao: Boolean; Fonte: TFont) of object;

  TCalendario = class(TCustomPanel)
  private
    { Private declarations }
    FReadOnly, FVisibleBarra: Boolean;
    FDispAnoMes: TPanel;
    FData: TDateTime;
    FFeriadosFont: TFont;
    FMensa: TImage;
    FFeriados: TStrings;
    FUpdating, FDesenhar: Boolean;
    FDestacarDia: TDiaDestaque;
    FMonthOffset: Integer;
    FInicioDaSemana: TDiasDaSemana;
    FHint: string;
    FShowFeriado: Boolean;
    FOnChange: TNotifyEvent;
    FOnHintDia: TNotifyFeriado;
    procedure ChangeMonth(Delta: Integer);
    procedure SelecCel(Sender: TObject; Col, Row: Longint; var CanSelect: Boolean);
    function GetCellText(ACol, ARow: Integer): string;
    function IsLeapYear(AYear: Integer): Boolean; virtual;
    procedure MontaCalendario;
    procedure SetVisibleBarra(Value: Boolean);
    procedure SetFeriadosFont(Value: TFont);
    procedure SetData(Value: TDateTime);
    procedure SetFeriados(Value: TStrings);
    procedure SetDestacarDia( Value: TDiaDestaque);
    procedure SetDataEle(Index: Integer; Value: Integer);
    procedure SetInicioDaSemana(Value: TDiasDaSemana);
    procedure SetHint( Value: String);
    function GetDataEle(Index: Integer): Integer;
    procedure BotaoClick(Sender: TObject);
    procedure DuploClick( Sender: TObject);
    procedure GridMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure Redimensiona;
    function IndiceFeriado( const lDia: Byte): Integer;
    procedure DesenhaCel(Sender: TObject; ACol, ARow: Longint; ARect: TRect; AState:
              TGridDrawState);
    procedure AlteracaoData; dynamic;
    procedure AprFeriado( Data: TDateTime; Descricao: string; Feriado: Boolean);
  protected
    { Protected declarations }
    FGrid: TDrawGrid;
    procedure GridClick(Sender: TObject); virtual;
    procedure DrawDia( Fonte: TFont; Dia: Integer); virtual;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    function DaysThisMonth: Integer; virtual;
    function DaysPerMonth(AYear, AMonth: Integer): Integer; virtual;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ProximoMes;
    procedure ProximoAno;
    procedure AnteriorMes;
    procedure AnteriorAno;
    procedure Atualiza; virtual;
    property CellText[ACol, ARow: Integer]: string read GetCellText;
    property Data: TDateTime  read FData write SetData;
  published
    { Published declarations }
    property VisibleBarra: Boolean read FVisibleBarra write SetVisibleBarra;
    property ReadOnly: Boolean read FReadOnly write FReadOnly default False;
    property InicioDaSemana: TDiasDaSemana read FInicioDaSemana write SetInicioDaSemana;
    property dtDia: Integer index 3  read GetDataEle write SetDataEle stored False;
    property dtMes: Integer index 2  read GetDataEle write SetDataEle stored False;
    property dtAno: Integer index 1  read GetDataEle write SetDataEle stored False;
    property DestacarDia: TDiaDestaque read FDestacarDia write SetDestacarDia;
    property Feriados: TStrings read FFeriados write SetFeriados;
    property FeriadosFont: TFont read FFeriadosFont write SetFeriadosFont;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnHintDia: TNotifyFeriado read FOnHintDia write FOnHintDia;
    property Hint: string read FHint write SetHint;
    property ShowFeriado: Boolean read FShowFeriado write FShowFeriado;
    property Align;
    property Font;
    property Visible;
    property TabOrder;
    property TabStop;
    property ShowHint;
    property OnDblClick;
  end;

  TDBCalendario = class(TCalendario)
  private
         FCampoMemo, FCampoData: string;
         FDataSource: TDataSource;
         FTab: TDataSet;
         FOnDrawDia: TNotifyDrawDia;
         procedure DefineTab;
         function FindAnotacao( lData: TDateTime): Boolean;
  public
        constructor Create(AOwner: TComponent); override;
        procedure EditaDia( Data: TDateTime);
        procedure Atualiza; override;
  protected
           procedure CreateParams(var Params: TCreateParams); override;
           procedure DrawDia( Fonte: TFont; Dia: Integer); override;
           procedure GridMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
  published
           property CampoMemo: string read FCampoMemo write FCampoMemo;
           property CampoData: string read FCampoData write FCampoData;
           property DataSource: TDataSource read FDataSource write FDataSource;
           property OnDrawDia: TNotifyDrawDia read FOnDrawDia write FOnDrawDia;
  end;

procedure Register;

implementation

{$R CALEND.RES}

procedure Register;
begin
RegisterComponents('DB', [TDBCalendario]);
RegisterComponents('XT', [TCalendario]);
end;

Function Trim( const Var1: String): String;
var IniStr, FimStr, TamStr: Byte;
begin
     TamStr := Length(Var1);
     for IniStr:=1 to TamStr do if Var1[IniStr]<>' ' then break;
     for FimStr:=TamStr downto 1 do if Var1[FimStr]<>' ' then break;
     if FimStr<IniStr then Trim := ''
     else Trim := Copy( Var1, IniStr, FimStr-IniStr+1);
end;

{**************************************************************************
Criao do componente
**************************************************************************}
constructor TCalendario.Create(AOwner: TComponent);
begin
inherited Create(AOwner);
FShowFeriado  := True;
FDesenhar     := True;
FVisibleBarra := True;
FFeriadosFont := TFont.Create;
FFeriadosFont.Color := clRed;
MontaCalendario;
FMensa := TImage.Create(Self);
with FMensa, FMensa.Canvas do
     begin
     Parent      := FGrid;
     Visible     := False;
     Brush.Color := clYellow;
     Font.Name   := 'MS Sans Serif';
     Font.Size   := 8;
     Height      := TextHeight('Z')+4;
     Width       := 0;
     end;
FFeriados := TStringList.Create;
FData     := Date;
Height    := 150;
Width     := 150;
AlteracaoData;
end;

procedure TCalendario.MontaCalendario;
      Procedure CriaBotao( const lNum: Byte);
      var lBot: TSpeedButton;
           procedure LeFig( lNome: PChar);
           begin
           lBot.Glyph.Handle := LoadBitmap( hInstance, lNome);
           end;
      begin
      lBot := TSpeedButton.Create( Self);
      with lBot do
           begin
           Parent := FDispAnoMes;
           Tag    := lNum;
           Name   := 'Botao'+IntToStr(lNum);
           case lNum of
                0: LeFig( 'CALANT2');
                1: LeFig( 'CALANT1');
                2: LeFig( 'CALPRO1');
                3: LeFig( 'CALPRO2');
                end;
           OnClick   := BotaoClick;
           end;
      end;

      Procedure CriaCab;
      var i: Byte;
      begin
      FDispAnoMes := TPanel.Create( Self);
      with FDispAnoMes do
           begin
           Parent     := Self;
           Align      := alTop;
           BevelOuter := bvNone;
           Visible    := FVisibleBarra;
           end;
      for i:=0 to 3 do CriaBotao(i);
      end;

begin
{Garante execucao unica}
if ComponentCount>0 then Exit;

{Define o painel de baixo, que e' o painel herdado}
BevelOuter := bvNone;
CriaCab;
FGrid := TDrawGrid.Create(Self);
with FGrid do
     begin
     Parent     := Self;
     Ctl3D     := False;
     FixedCols  := 0;
     FixedRows  := 1;
     ColCount   := 7;
     RowCount   := 7;
     ScrollBars := ssNone;
     Options    := Options - [goRangeSelect];
     DefaultDrawing := False;
     GridLineWidth  := 0;
     BorderStyle    := bsNone;
     Color          := clBtnFace;
     Align          := alClient;
     OnClick        := GridClick;
     OnDblClick     := DuploClick;
     OnMouseMove    := GridMouseMove;
     OnDrawCell     := DesenhaCel;
     OnSelectCell   := SelecCel;
     end;
end;

destructor TCalendario.Destroy;
begin
FFeriados.Free;
FFeriadosFont.Free;
inherited Destroy;
end;

procedure TCalendario.DesenhaCel(Sender: TObject; ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
var
  lTexCel: string;
  lRect: TRect;
  lDia: Integer;
begin
with FGrid do
     begin
     lTexCel := CellText[ACol, ARow];
     lDia    := StrToIntDef(lTexCel,0);
     if FDesenhar then
        begin
        with ARect, Canvas do
             begin
             if (IndiceFeriado( lDia)<0) And (ACol<>FDestacarDia) then FGrid.Canvas.Font.Assign(Self.Font)
             else FGrid.Canvas.Font.Assign(FFeriadosFont);

             Brush.Color := clBtnFace;
             FillRect(ARect);
             DrawDia( Font, lDia);
             TextRect(ARect, Left + (Right - Left - TextWidth(lTexCel)) div 2,
                       Top + (Bottom - Top - TextHeight(lTexCel)) div 2, lTexCel);

             Pen.Color := clBtnHighLight;
             PolyLine([Point(Left,Top), Point(Right,Top)]);
             if ACol=0 then PolyLine([Point(Left,Bottom-2),Point(Left,Top)]);
             Pen.Color := clBtnShadow;
             PolyLine([Point(Left,Bottom-1),Point(Right,Bottom-1)]);
             if ACol=6 then PolyLine([Point(Right-1,Bottom-1),Point(Right-1,Top)]);

             if ARow=0 then
                begin
                Pen.Color := clBtnHighLight;
                PolyLine([Point(Left,Top+1), Point(Right,Top+1)]);
                Pen.Color := clBtnShadow;
                PolyLine([Point(Left,Bottom-2),Point(Right,Bottom-2)]);
                if ACol=6 then
                   begin
                   PolyLine([Point(Right-1,Bottom-1),Point(Right-1,Top)]);
                   PolyLine([Point(Right-2,Bottom-1),Point(Right-2,Top+1)]);
                   end;
                Pen.Color := clBtnHighLight;
                if ACol=0 then
                   begin
                   PolyLine([Point(Left,Bottom-2),Point(Left,Top)]);
                   PolyLine([Point(Left+1,Bottom-3),Point(Left+1,Top)]);
                   end;
                end;
             end;
        end
     else
         begin
         lRect := Rect( ARect.Left+2, ARect.Top+2, ARect.Right-2, ARect.Bottom-2);
         Frame3d(Canvas,lRect,clBtnFace,clBtnFace,1);
         end;
     if dtDia=lDia then
        begin
        lRect := Rect( ARect.Left+2, ARect.Top+2, ARect.Right-2, ARect.Bottom-2);
        Frame3d(Canvas,lRect,clBtnShadow,clBtnHighLight,1)
        end;
     end;
end;

procedure TCalendario.SelecCel(Sender: TObject; Col, Row: Longint; var CanSelect: Boolean);
begin
CanSelect := Not (((not FUpdating) and FReadOnly) or (CellText[Col, Row] = ''));
end;

procedure TCalendario.DrawDia( Fonte: TFont; Dia: Integer);
begin

end;
{**************************************************************************
Manuteno do componente
**************************************************************************}
procedure TCalendario.WMSize(var Message: TWMSize);
begin
Redimensiona;
FGrid.DefaultColWidth  := FGrid.Width div 7;
FGrid.DefaultRowHeight := FGrid.Height div 7;
end;

procedure TCalendario.Redimensiona;
          procedure DimensoesBotao( const lNum: Byte);
          var
             uBot: TSpeedButton;
             lWid: Integer;
          begin
          lWid := (FGrid.Width div 7)*7;
          uBot := TSpeedButton(Self.FindComponent('Botao'+IntToStr(lNum)));
          with  uBot do
               begin
               Top    := 1;
               Width  := FDispAnoMes.Height-2;
               Height := Width;
               case lNum of
                    0: Left := 1;
                    1: Left := Width+1;
                    2: Left := lWid-Width shl 1-1;
                    3: Left := lWid-Width-1;
                    end;
               end;
          end;

var
   i: Word;
begin
FDispAnoMes.Height  := (Height div 8);
FDispAnoMes.Visible := FVisibleBarra;
for i:=0 to 3 do DimensoesBotao(i);
Atualiza;
end;

procedure TCalendario.Atualiza;
var
  AYear, AMonth, ADay: Word;
  FirstData: TDateTime;
begin
  FUpdating := True;
  try
    DecodeDate(FData, AYear, AMonth, ADay);
    FirstData := EncodeDate(AYear, AMonth, 1);
    FMonthOffset := 2 - ((DayOfWeek(FirstData) - FInicioDaSemana + 7) mod 7); { dtDia of week for 1st of month }
    if FMonthOffset = 2 then FMonthOffset := -5;
{    FGrid.Col := (ADay - FMonthOffset) mod 7;
    FGrid.Row := (ADay - FMonthOffset) div 7 + 1;}
    FGrid.Invalidate;
    Application.ProcessMessages;
  finally
         FDesenhar := True;
         FUpdating := False;
  end;
end;

{**************************************************************************
Propriedades e eventos
**************************************************************************}
function TCalendario.GetCellText(ACol, ARow: Integer): string;
var
  DayNum: Integer;
begin
  if ARow = 0 then Result := ShortDayNames[(InicioDaSemana + ACol) mod 7 + 1]
  else
      begin
      DayNum := FMonthOffset + ACol + (ARow - 1) * 7;
      if (DayNum < 1) or (DayNum > DaysThisMonth) then Result := ''
      else Result := Copy( IntToStr(DayNum+100), 2, 2);
      end;
end;

procedure TCalendario.SetInicioDaSemana(Value: TDiasDaSemana);
begin
if Value <> FInicioDaSemana then
   begin
   FInicioDaSemana := Value;
   Atualiza;
   end;
end;

function TCalendario.GetDataEle(Index: Integer): Integer;
var
  AYear, AMonth, ADay: Word;
begin
DecodeDate(FData, AYear, AMonth, ADay);
case Index of
    1: Result := AYear;
    2: Result := AMonth;
    3: Result := ADay;
    else Result := -1;
    end;
end;

procedure TCalendario.SetDataEle(Index: Integer; Value: Integer);
var
  AYear, AMonth, ADay: Word;
begin
if Value > 0 then
   begin
   DecodeDate(FData, AYear, AMonth, ADay);
   case Index of
        1: if AYear <> Value then AYear := Value else Exit;
        2: if (Value <= 12) and (Value <> AMonth) then AMonth := Value else Exit;
        3: if (Value <= DaysThisMonth) and (Value <> ADay) then ADay := Value else Exit;
        else Exit;
        end;
   FData := EncodeDate(AYear, AMonth, ADay);
   Atualiza;
   AlteracaoData;
   end;
end;

procedure TCalendario.SetDestacarDia( Value: TDiaDestaque);
begin
if Value <> FDestacarDia then
   begin
   FDestacarDia := Value;
   Atualiza;
   end;
end;

procedure TCalendario.SetFeriados(Value: TStrings);
begin
FFeriados.Assign(Value);
end;

procedure TCalendario.SetData(Value: TDateTime);
begin
FData := Value;
Atualiza;
AlteracaoData;
end;

procedure TCalendario.SetVisibleBarra(Value: Boolean);
begin
FVisibleBarra := Value;
FDispAnoMes.Visible := Value;
end;

procedure TCalendario.DuploClick( Sender: TObject);
var F: TNotifyEvent;
begin
F := OnDblClick;
if Assigned(F) then F(Sender);
end;

procedure TCalendario.AlteracaoData;
begin
FDispAnoMes.Caption := LongMonthNames[dtMes]+'  '+IntToStr( dtAno);
if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TCalendario.SetFeriadosFont(Value: TFont);
begin
FFeriadosFont.Assign( Value);
Atualiza;
end;

procedure TCalendario.SetHint( Value: String);
begin
FHint := Value;
FDispAnoMes.Hint := Value;
FGrid.Hint := Value;
FDispAnoMes.ShowHint := ShowHint;
FGrid.ShowHint := ShowHint;
end;

procedure TCalendario.AprFeriado( Data: TDateTime; Descricao: string; Feriado: Boolean);
begin
if Assigned( FOnHintDia) then
   FOnHintDia( Self, Data, Descricao, Feriado);
end;

{**************************************************************************
Metodos
**************************************************************************}
procedure TCalendario.ProximoMes;
begin
ChangeMonth(1);
end;

procedure TCalendario.ProximoAno;
begin
if IsLeapYear(dtAno) and (dtMes = 2) and (dtDia = 29) then dtDia := 28;
dtAno := dtAno + 1;
end;

procedure TCalendario.AnteriorAno;
begin
if IsLeapYear(dtAno) and (dtMes = 2) and (dtDia = 29) then dtDia := 28;
dtAno := dtAno - 1;
end;

procedure TCalendario.AnteriorMes;
begin
ChangeMonth(-1);
end;
{**************************************************************************
Diversos
**************************************************************************}
procedure TCalendario.BotaoClick(Sender: TObject);
begin
case TSpeedButton(Sender).Tag of
     0: AnteriorAno;
     1: AnteriorMes;
     2: ProximoMes;
     3: ProximoAno;
     end;
end;

function TCalendario.IndiceFeriado( const lDia: Byte): Integer;
   Function Procura( const lDiaS: string): Integer;
   var i: Word;
   begin
   Result := -1;
   if FFeriados.Count=0 then Exit;
   for i:=0 to FFeriados.Count-1 do
       begin
       if Copy( FFeriados[i], 1, 8)=lDiaS then
          begin
          Result := i;
          Break;
          end;
       end;
   end;

var
   lData: TDateTime;
begin
Result := -1;
if lDia>0 then
   begin
   lData := EncodeDate( dtAno, dtMes, lDia);
   Result := Procura( FormatDateTime( 'dd/mm/yy', lData));
   if Result=-1 then Result := Procura( Format( '%.2d/%.2d/00', [lDia,dtMes]));
   end;
end;

{**************************************************************************
Tratamento de datas
**************************************************************************}
function TCalendario.DaysThisMonth: Integer;
begin
Result := DaysPerMonth(dtAno, dtMes);
end;

function TCalendario.DaysPerMonth(AYear, AMonth: Integer): Integer;
const
  DaysInMonth: array[1..12] of Integer = (31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);
begin
  Result := DaysInMonth[AMonth];
  if (AMonth = 2) and IsLeapYear(AYear) then Inc(Result); { leap-year Feb is special }
end;

function TCalendario.IsLeapYear(AYear: Integer): Boolean;
begin
Result := (AYear mod 4 = 0) and ((AYear mod 100 <> 0) or (AYear mod 400 = 0));
end;

procedure TCalendario.ChangeMonth(Delta: Integer);
var
  AYear, AMonth, ADay: Word;
  NewData: TDateTime;
  CurDay: Integer;
begin
  DecodeDate(FData, AYear, AMonth, ADay);
  CurDay := ADay;
  if Delta > 0 then ADay := DaysPerMonth(AYear, AMonth)
  else ADay := 1;
  NewData := EncodeDate(AYear, AMonth, ADay);
  NewData := NewData + Delta;
  DecodeDate(NewData, AYear, AMonth, ADay);
  if DaysPerMonth(AYear, AMonth) > CurDay then ADay := CurDay
  else ADay := DaysPerMonth(AYear, AMonth);
  Data := EncodeDate(AYear, AMonth, ADay);
end;

procedure TCalendario.GridClick(Sender: TObject);
var TheCellText: string;
begin
FDesenhar   := False;
TheCellText := CellText[FGrid.Col, FGrid.Row];
if TheCellText <> '' then dtDia := StrToInt(TheCellText);
end;

procedure TCalendario.GridMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
const
     cPonFer: Integer = -1; {Evita chamadas excessivas ao evento OnHintDia e
                          reduz operaes no FMensa}
var
   lData: TDateTime;
   lPonFer, lDia: Integer;
   lMens: string;
   lFlag: Boolean;
   lLin, lCol: Byte;
begin
lFlag := False;
lMens := '';
with FGrid do
     begin
     try
        lCol    := X div DefaultColWidth;
        lLin    := Y div DefaultRowHeight;
        lDia    := StrToIntDef( CellText[ lCol, lLin], 0);
        lPonFer := IndiceFeriado( lDia);
        if lDia>0 then lData := EncodeDate( dtAno, dtMes, lDia)
        else lData := EncodeDate( 1, 1, 1);

        if (X<15) or (Y<15) or (X>Width-15) or (Y>Height-15) Or (lPonFer<0) then Exit;
        lMens := Copy(FFeriados[lPonFer],9,255);

        if lPonFer<>cPonFer then
           begin
           AprFeriado( lData, lMens, True);
           if FShowFeriado And (lMens<>'') then
              with FMensa do
                   begin
                   Top     := (lLin+1)*DefaultRowHeight+5;
                   Left    := lCol*DefaultColWidth;
                   Width   := Canvas.TextWidth(lMens)+6;
                   if Left+Width+6>Self.Width then Left := Self.Width-Width-6;
                   if Top+Height+10>Self.Height then Top := (lLin-1)*DefaultRowHeight-5;
                   Picture.Graphic.Height := Height;
                   Picture.Graphic.Width  := Width;
                   Canvas.FillRect( Canvas.ClipRect);
                   Canvas.Rectangle(0,0,Width,Height);
                   Canvas.TextOut( 2, 2, lMens);
                   Visible := True;
                   end;
           end;
        lFlag := True;
        cPonFer  := lPonFer;
     finally
            if Not lFlag then
               begin
               cPonFer        := -1;
               FMensa.Visible := False;
               AprFeriado( lData, '', False);
               end;
     end;
end;
end;

{**************************************************************************
***************************************************************************
TDBCalendrio
***************************************************************************
**************************************************************************}

{**************************************************************************
Rotinas de criao
**************************************************************************}
constructor TDBCalendario.Create(AOwner: TComponent);
begin
inherited Create(AOwner);
FDataSource := nil;
FGrid.OnMouseDown := GridMouseDown;
end;

procedure TDBCalendario.CreateParams(var Params: TCreateParams);
begin
inherited CreateParams( Params);
DefineTab;
end;

procedure TDBCalendario.DrawDia( Fonte: TFont; Dia: Integer);
var
   lData: TDateTime;
   lAnotacao: Boolean;
begin
if (FTab=nil) Or (Dia<1) then Exit;
lData := EncodeDate( dtAno, dtMes, Dia);
lAnotacao := FindAnotacao(lData);
if Assigned(FOnDrawDia) then FOnDrawDia( Self, lData, lAnotacao, Fonte)
else
    if lAnotacao then
       Fonte.Style := Fonte.Style+[fsItalic];
end;

{**************************************************************************
Diversos
**************************************************************************}
procedure TDBCalendario.DefineTab;
begin
FTab := nil;
if (FDataSource=nil) Or (FCampoData='') Or (FCampoMemo='') then Exit;
if FDataSource.DataSet=nil then Exit;
if FDataSource.Dataset.Active then FTab := FDataSource.Dataset;
end;

{Procura por uma anotao na data lData}
function TDBCalendario.FindAnotacao( lData: TDateTime): Boolean;
begin
Result := False;
if FTab=nil then Exit;
if FTab is TTable then Result := TTable(FTab).FindKey([lData])
else
    with TQuery(FTab) do
         begin
         Close;
         Params[0].AsDateTime := lData;
         Open;
         Result := FieldByName(FCampoData).AsDateTime=lData;
         end;
end;

procedure TDBCalendario.Atualiza;
begin
DefineTab;
inherited Atualiza;
end;

procedure TDBCalendario.GridMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
   lDia: Word;
begin
if Button=mbRight then
   begin
   with FGrid do
        lDia    := StrToIntDef( CellText[ X div DefaultColWidth, Y div DefaultRowHeight], 0);
   EditaDia( EncodeDate( dtAno, dtMes, lDia));
   end;
end;

{**************************************************************************
Mtodos
**************************************************************************}
procedure TDBCalendario.EditaDia( Data: TDateTime);
var
   lForm: TForm;
   lMemo: TMemo;
          procedure CriaForm;
          var lPan: TPanel;
          begin
          lForm := TForm.Create(Self);
          with lForm do
               begin
               Caption     := 'Edio '+FormatDateTime('dd/mm/yyyy',Data);
               BorderIcons := [biSystemMenu];
               BorderStyle := bsDialog;
               Height      := 234;
               Width       := 283;
               Position    := poScreenCenter;
               Font.Assign( TForm(Owner).Font);
               end;
          lPan := TPanel.Create(lForm);
          with lPan do
               begin
               Parent := lForm;
               Align  := alBottom;
               Caption:= '';
               Height := 44;
               end;
          with TBitBtn.Create(lForm) do
               begin
               Parent := lPan;
               Width  := 81;
               Height := 25;
               Left   := 96;
               Top    := 8;
               Kind   := bkOK;
               Caption:= '&OK';
               end;
          with TBitBtn.Create(lForm) do
               begin
               Parent := lPan;
               Width  := 81;
               Height := 25;
               Left   := 184;
               Top    := 8;
               Kind   := bkCancel;
               Caption:= '&Cancelar';
               end;
          lMemo := TMemo.Create(lForm);
          with lMemo do
               begin
               Align    := alClient;
               Parent   := lForm;
               TabOrder := 0;
               end;
          end;

          function MemoVazio: Boolean;
          var i: Integer;
          begin
          Result := True;
          if lMemo.Lines.Count<1 then Exit;
          for i:=0 to lMemo.Lines.Count-1 do
              if Trim(lMemo.Lines[i])<>'' then Result := False;
          end;
begin
if FTab=nil then Exit;
CriaForm;

if FindAnotacao(Data) then lMemo.Lines.Assign(FTab.FieldByName(FCampoMemo))
else lMemo.Lines.Clear;

try
   if lForm.ShowModal=mrOK then
      begin
      if FindAnotacao(Data) then
         begin
         if MemoVazio then
            begin
            FTab.Delete;
            Exit;
            end;
         FTab.Edit;
         end
      else
          begin
          if MemoVazio then Exit;
          FTab.Append;
          end;
      FTab.FieldByName(FCampoData).AsDateTime := Data;
      FTab.FieldByName(FCampoMemo).Assign(lMemo.Lines);
      FTab.Post;
      end;
finally
       Atualiza;
       lForm.Release;
end;
end;

end.
