//  Filename: Main.pas
//  Version : 01.00
//  Date    : 29/03/97
//  Author  : Glenn Davies
//
//  Description:
//    Alias Editor example application main form.
//
//  History:
//    29/03/97 - V01.00
//      Initial version.

unit Main;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, AliasEd;

type

  TAppMode = (amBrowse,amNew,amEdit);

  TFormMain = class(TForm)
    AliasEditor: TAliasEditor;
    GroupBoxAliases: TGroupBox;
    ListBoxAliases: TListBox;
    GroupBoxSelectedAlias: TGroupBox;
    MemoParams: TMemo;
    EditName: TEdit;
    LabelName: TLabel;
    ComboBoxDrivers: TComboBox;
    LabelDriver: TLabel;
    ButtonNew: TButton;
    ButtonEdit: TButton;
    ButtonDelete: TButton;
    LabelParameters: TLabel;
    ButtonOK: TButton;
    ButtonCancel: TButton;
    procedure FormCreate(Sender: TObject);
    procedure ListBoxAliasesClick(Sender: TObject);
    procedure ButtonNewClick(Sender: TObject);
    procedure ButtonCancelClick(Sender: TObject);
    procedure EditNameChange(Sender: TObject);
    procedure ButtonEditClick(Sender: TObject);
    procedure ButtonOKClick(Sender: TObject);
    procedure ComboBoxDriversChange(Sender: TObject);
    procedure ButtonDeleteClick(Sender: TObject);
  private
    fAppMode: TAppMode; //  Current application mode - browsing, editing or creating a new alias
    procedure DoDelete;
    procedure DoNew;
    procedure DoEdit;
    procedure GetAliasList;
    procedure GetDriverList;
    procedure GetSelectedAliasInfo;
    procedure GetSelectedDefaultParams;
    procedure InitialiseBrowseMode;
    procedure InitialiseEditMode;
    procedure InitialiseNewMode;
    procedure Startup;
  public
  end;

var
  FormMain: TFormMain;

//------------------------------------------------------------------------------

implementation

{$R *.DFM}

//------------------------------------------------------------------------------

//  Private methods

//------------------------------------------------------------------------------

//  Delete the selected alias

procedure TFormMain.DoDelete;
begin
  AliasEditor.Delete(ListBoxAliases.Items[ListBoxAliases.ItemIndex]);
  InitialiseBrowseMode;
end;

//------------------------------------------------------------------------------

//  Create a new alias

procedure TFormMain.DoNew;
var
  selected_alias: String;
  selected_index: Integer;
begin
  selected_alias := EditName.Text;
  AliasEditor.Add(EditName.Text,ComboBoxDrivers.Text,MemoParams.Lines);
  InitialiseBrowseMode;
  //  Select this alias in the listbox
  selected_index := ListBoxAliases.Items.IndexOf(selected_alias);
  if selected_index <> -1 then
  begin
    ListBoxAliases.ItemIndex := selected_index;
    GetSelectedAliasInfo;
  end;
end;

//------------------------------------------------------------------------------

//  Edit an existing alias

procedure TFormMain.DoEdit;
var
  selected_alias: String;
  selected_index: Integer;
begin
  selected_alias := EditName.Text;
  AliasEditor.Modify(EditName.Text,MemoParams.Lines);
  InitialiseBrowseMode;
  //  Select this alias in the listbox
  selected_index := ListBoxAliases.Items.IndexOf(selected_alias);
  if selected_index <> -1 then
  begin
    ListBoxAliases.ItemIndex := selected_index;
    GetSelectedAliasInfo;
  end;
end;

//------------------------------------------------------------------------------

//  Populate the listbox with the current list of aliases

procedure TFormMain.GetAliasList;
begin
  AliasEditor.GetAliasList(ListboxAliases.Items);
end;

//------------------------------------------------------------------------------

//  Populate the combo box with the list of installed drivers

procedure TFormMain.GetDriverList;
begin
  AliasEditor.GetDriverList(ComboBoxDrivers.Items);
end;

//------------------------------------------------------------------------------

//  Get alias info for selected alias in listbox

procedure TFormMain.GetSelectedAliasInfo;
begin
  if ListBoxAliases.ItemIndex <> -1 then
  begin
    ButtonEdit.Enabled := True;
    ButtonEdit.Enabled := True;
    EditName.Text := ListBoxAliases.Items[ListBoxAliases.ItemIndex];
    AliasEditor.GetParams(EditName.Text,MemoParams.Lines);
    ComboBoxDrivers.ItemIndex :=
      ComboBoxDrivers.Items.IndexOf(AliasEditor.GetDriver(EditName.Text));
  end
  else
  begin
    ButtonEdit.Enabled := False;
    ButtonEdit.Enabled := False;
    ComboBoxDrivers.ItemIndex := -1;
    EditName.Text := '';
    MemoParams.Text := '';
  end;
end;

//------------------------------------------------------------------------------

//  Get the default driver params for selected driver in combobox

procedure TFormMain.GetSelectedDefaultParams;
begin
  AliasEditor.GetDriverParams(ComboBoxDrivers.Text,MemoParams.Lines);
end;

//------------------------------------------------------------------------------

//  Initialise the application for browsing aliases

procedure TFormMain.InitialiseBrowseMode;
begin
  ListBoxAliases.Enabled := True;
  ButtonNew.Enabled := True;
  ButtonEdit.Enabled := True;
  ButtonDelete.Enabled := True;
  EditName.Enabled := False;
  ComboBoxDrivers.Enabled := False;
  MemoParams.Enabled := False;
  ButtonOK.Enabled := False;
  ButtonCancel.Enabled := False;
  GetAliasList;
  if ListboxAliases.Items.Count > 0 then
    ListboxAliases.ItemIndex := 0;
  GetSelectedAliasInfo;
  fAppMode := amBrowse;
end;

//------------------------------------------------------------------------------

//  Initialise the application for creating a new alias

procedure TFormMain.InitialiseEditMode;
begin
  ListBoxAliases.ItemIndex := -1;
  ListBoxAliases.Enabled := False;
  ButtonNew.Enabled := False;
  ButtonEdit.Enabled := False;
  ButtonDelete.Enabled := False;
  EditName.Enabled := False;        //  These parameters cannot be changed once the alias
  ComboBoxDrivers.Enabled := False; //  has been created
  MemoParams.Enabled := True;
  ButtonOK.Enabled := True;
  ButtonCancel.Enabled := True;
  fAppMode := amEdit;
end;

//------------------------------------------------------------------------------

//  Initialise the application for creating a new alias

procedure TFormMain.InitialiseNewMode;
begin
  ListBoxAliases.ItemIndex := -1;
  ListBoxAliases.Enabled := False;
  ButtonNew.Enabled := False;
  ButtonEdit.Enabled := False;
  ButtonDelete.Enabled := False;
  EditName.Text := '';
  ComboBoxDrivers.ItemIndex := 0;
  GetSelectedDefaultParams;
  EditName.Enabled := True;
  ComboBoxDrivers.Enabled := True;
  MemoParams.Enabled := True;
  ButtonOK.Enabled := False;
  ButtonCancel.Enabled := True;
  fAppMode := amNew;
end;

//------------------------------------------------------------------------------

//  Initialise the application

procedure TFormMain.Startup;
begin
  try
    GetDriverList;
    if ComboBoxDrivers.Items.Count = 0 then
      Abort;
  except
    on Exception do
    begin
      //  The BDE is incorrectly initialised - terminate the application
      MessageDlg(Application.Title+' encountered a critical error while initialising the '+
                 'Borland Database Engine.',mtError,[mbOK],0);
      Application.Terminate;
    end;
  end;
  InitialiseBrowseMode;
end;

//------------------------------------------------------------------------------

//  Delphi-generated event handlers

//------------------------------------------------------------------------------

procedure TFormMain.FormCreate(Sender: TObject);
begin
  Startup;
end;

//------------------------------------------------------------------------------

procedure TFormMain.ListBoxAliasesClick(Sender: TObject);
begin
  GetSelectedAliasInfo;
end;

//------------------------------------------------------------------------------

procedure TFormMain.ButtonNewClick(Sender: TObject);
begin
  InitialiseNewMode;
end;

//------------------------------------------------------------------------------

procedure TFormMain.ButtonCancelClick(Sender: TObject);
begin
  InitialiseBrowseMode;
end;

//------------------------------------------------------------------------------

procedure TFormMain.EditNameChange(Sender: TObject);
begin
  //  Only enable the OK button when there is a valid alias name
  if (fAppMode = amNew) then
    ButtonOK.Enabled := EditName.Text <> '';
end;

//------------------------------------------------------------------------------

procedure TFormMain.ButtonEditClick(Sender: TObject);
begin
  InitialiseEditMode;
end;

//------------------------------------------------------------------------------

procedure TFormMain.ButtonOKClick(Sender: TObject);
begin
  case fAppMode of
    amEdit: DoEdit;
    amNew: DoNew;
  end;
end;

//------------------------------------------------------------------------------

procedure TFormMain.ComboBoxDriversChange(Sender: TObject);
begin
  GetSelectedDefaultParams;
end;

//------------------------------------------------------------------------------

procedure TFormMain.ButtonDeleteClick(Sender: TObject);
begin
  DoDelete;
end;

//------------------------------------------------------------------------------

end.
