unit Win97izer;

{The TWin97izer control turns any TWinControl into the Office-97 style!
 To use, simply drop it on the form, assign the Win97Control property to the
 TWinControl you wish to Win97-ize, and that's it!
 Version 1.0
 Written by Christopher Sansone: ChrisSansone@Rocketmail.com

 Special thanks to Jordan Russell for his assistance in debugging and for
 the code to his TEdit97, of which this component is based.

 Known bugs:
   --It works best with Window controls (TEdit, TListBox, TTreeView, etc.).
     It also works well with Panels, TCheckBoxes, and the like.  It does not
     work very well with TComboBox, TScrollBar, and other controls like those.
   --On large controls, the border will flicker a bit as the user moves another
     window over the client area.  This occurs because the original border is
     first drawn (on the WM_NCPAINT message) and then is re-drawn.  However, the
     control needs to handle the WM_NCPAINT message, or the BorderWidth of the
     control will not be painted.  To fix this, the WMNCPAINT procedure needs to
     first clip the border before painting, then re-assign the border.  Anyone
     want to give it a shot?  :)

 Use this component at your own risk.  I am hereby not held responsible for any
 damages caused through the use of this component.

 This component is distributed as freeware.  You may use the compiled version in
 freeware, shareware, and commercialware.  Modify it as you please, but there
 must never be a charge for it.  If you make any fixes or enhancements, please
 send me a copy!  I do not plan on enhancing it myself, so feel free to take it
 over!

 Modified: 1-1-98  }


interface

uses StdCtrls, Windows, Messages, Controls, Classes, Forms, Graphics;

type
  TWin97Control = class(TWinControl)
  public
    property Color;
    property BorderWidth;
  end;

  TEdgeType = (etSunkenOuter, etRaisedInner, etSunkenInner, etRaisedOuter, 
               etBump, etBigBump, etEtched, etRaised, etSunken);

  TWin97izer = class(TComponent)
  private
    F97Control: TWinControl;
    FEdgeType: TEdgeType;
    FControlColor: TColor;
    MouseInControl: Boolean;
    OldWin97WindowProc: TWndMethod;
    procedure SetF97Control(Value: TWinControl);
    procedure Win97WindowProc(var Message: TMessage);
    procedure CMEnabledChanged;
    procedure CMMouseEnter;
    procedure CMMouseLeave;
    function GetEdgeType: Integer;
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    constructor Create (AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Win97Control: TWinControl read F97Control write SetF97Control;
    property EdgeType: TEdgeType read FEdgeType write FEdgeType;
    procedure RedrawBorder;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents ('Win32', [TWin97izer]);
end;

constructor TWin97izer.Create (AOwner: TComponent);
{inherits the Create method and sets up the Win97Control}
begin
  inherited;
  SetF97Control(F97Control);
end;

destructor TWin97izer.Destroy;
{resets the Window-handling procedure of the Win97Control to
 its default}
begin
  If Assigned(F97Control) then begin
    F97Control.WindowProc := OldWin97WindowProc;
    F97Control := nil;
  end;
  inherited;
end;

procedure TWin97izer.Notification(AComponent: TComponent; Operation:TOperation);
{checks to see if F97Control has been destroyed}
begin
  inherited;
  if (Operation = opRemove) and (AComponent = F97Control) then //destroyed
    SetF97Control(nil);  //assign nil to F97Control
end;

procedure TWin97izer.SetF97Control(Value: TWinControl);
{sets Value as the TWinControl to Win97-ize}
begin
  //reset the Window-handling procedure of the previous Win97Control to
  //its default
  if Assigned(F97Control) then
    With TWin97Control(F97Control) do
        WindowProc := OldWin97WindowProc;

  F97Control := Value;   //set the Win97Control
  If Assigned(F97Control) then begin
    F97Control.FreeNotification(Self); //so it knows when F97Control is destroyed
    With TWin97Control(F97Control) do begin
      OldWin97WindowProc := WindowProc; //store its default Window method
      WindowProc := Win97WindowProc; //change its Window method to this one
      if not(csDesigning in ComponentState) then begin
        If (Align <> alNone) and (BorderWidth < 2) then  //set the BorderWidth
          BorderWidth := 2;
        FControlColor := Color;  //store its color
        ControlStyle := ControlStyle - [csFramed];  //remove all borders
        CMEnabledChanged; //paint depending on the Enabled state
      end;
    end;
  end;
end;

procedure TWin97izer.Win97WindowProc(var Message: TMessage);
{the new WindowProc of the Win97Control--
 first performs all default functionality and then handles each
 message for Win97 functionality}
begin
  OldWin97WindowProc(Message);   //perform default method
  if (F97Control <> nil) and not (csDesigning in ComponentState) then
    try
      Case Message.Msg of
        CM_EnabledChanged: CMEnabledChanged;
        CM_MouseEnter: CMMouseEnter;
        CM_MouseLeave: CMMouseLeave;
        WM_SetFocus, WM_KillFocus, WM_NCPaint: RedrawBorder;
      end;
    except
    end;
end;

procedure TWin97izer.CMMouseEnter;
{the mouse entered the control--store that, and redraw the control}
begin
  MouseInControl := True;
  RedrawBorder;
end;

procedure TWin97izer.CMMouseLeave;
{the mouse left the control--store that, and redraw the control}
begin
  MouseInControl := False;
  RedrawBorder;
end;

procedure TWin97izer.CMEnabledChanged;
{the control was enabled or disabled--set the color and redraw the border}
begin
  If not (csDesigning in ComponentState) then
    With TWin97Control(F97Control) do
      try
        If Enabled then
          Color := FControlColor
        else
          Color := clBtnFace;
        RedrawBorder;
      except
      end;
end;

function TWin97izer.GetEdgeType: Integer;
{returns the Windows value corresponding to the current EdgeType}
begin
  Case FEdgeType of
    etRaisedOuter: Result := BDR_RAISEDOUTER;
    etRaisedInner: Result := BDR_RAISEDINNER;
    etSunkenOuter: Result := BDR_SUNKENOUTER;
    etSunkenInner: Result := BDR_SUNKENINNER;
    etBump, etBigBump: Result := EDGE_BUMP;
    etEtched: Result := EDGE_ETCHED;
    etRaised: Result := EDGE_RAISED
    else Result := EDGE_SUNKEN;
  end;

end;

procedure TWin97izer.RedrawBorder;
{redraws the border as needed}
var
  DC: HDC;
  R: TRect;
  BtnFaceBrush, WindowBrush, ShadowBrush: HBRUSH;
begin
  try
    With F97Control do begin
      DC := GetWindowDC(Handle);  //get the device context of the control
      try
        GetWindowRect(Handle, R);  //get the TRect of the control
        OffsetRect(R, -R.Left, -R.Top);  //offset the control as needed
        BtnFaceBrush := GetSysColorBrush(COLOR_BTNFACE); //get the Windows brush
        WindowBrush := GetSysColorBrush(COLOR_WINDOW); //get the Windows brush
        ShadowBrush := GetSysColorBrush(COLOR_BTNSHADOW);
        if not (csDesigning in ComponentState) and
            (Focused or (MouseInControl)) then begin  //draw the new border
          DrawEdge(DC, R, GetEdgeType, BF_RECT or BF_ADJUST);  //draw border

          Case FEdgeType of
            etSunkenOuter: with R do begin
              FillRect (DC, Rect(Left, Top, Left+1, Bottom-1), BtnFaceBrush);
              FillRect (DC, Rect(Left, Top, Right-1, Top+1), BtnFaceBrush);
            end;
           etBump: with R do begin
              FillRect (DC, Rect(Left-1, Top-1, Left, Bottom), ShadowBrush);
              FillRect (DC, Rect(Left, Top-1, Right, Top), ShadowBrush);
              FillRect (DC, Rect(Right+2, Top-2, Right+1, Bottom+2), ShadowBrush);
              FillRect (DC, Rect(Left-2, Bottom+1, Right+2, Bottom+2), ShadowBrush);
            end;
            etBigBump: with R do begin
              FillRect (DC, Rect(Left-1, Top-1, Left, Bottom), ShadowBrush);
              FillRect (DC, Rect(Left, Top-1, Right, Top), ShadowBrush);
              FillRect (DC, Rect(Right+2, Top-2, Right+1, Bottom+2), ShadowBrush);
              FillRect (DC, Rect(Left-2, Bottom+1, Right+2, Bottom+2), ShadowBrush);
              FillRect (DC, Rect(Left-2, Top-2, Left-1, Bottom+1), WindowBrush);
              FillRect (DC, Rect(Left-2, Top-2, Right+1, Top-1), WindowBrush);
              FillRect (DC, Rect(Right+1, Top-2, Right, Bottom+1), WindowBrush);
              FillRect (DC, Rect(Left-2, Bottom, Right+1, Bottom+1), WindowBrush);
            end;
            etEtched: If TWin97Control(F97Control).Color = clWindow then
              with R do begin
                FillRect (DC, Rect(Left-1, Top, Left, Bottom), BtnFaceBrush);
                FillRect (DC, Rect(Left-1, Top-1, Right, Top), BtnFaceBrush);
              end;
            etRaised: If TWin97Control(F97Control).Color = clWindow then
              with R do begin
              FillRect (DC, Rect(Left-1, Top, Left, Bottom), BtnFaceBrush);
              FillRect (DC, Rect(Left-2, Top-1, Right+1, Top), BtnFaceBrush);
              FillRect (DC, Rect(Right+1, Top-2, Right+2, Bottom+2), ShadowBrush);
              FillRect (DC, Rect(Left-2, Bottom+1, Right+2, Bottom+2), ShadowBrush);
            end;
            etRaisedInner: with R do begin
              FillRect (DC, Rect(Left, Top, Left+1, Bottom), BtnFaceBrush);
              FillRect (DC, Rect(Left, Top, Right, Top+1), BtnFaceBrush);
            end;
            etRaisedOuter: with R do begin
              FillRect (DC, Rect(Left, Top, Left+1, Bottom), BtnFaceBrush);
              FillRect (DC, Rect(Left, Top, Right, Top+1), BtnFaceBrush);
              FillRect (DC, Rect(Right, Top-1, Right+1, Bottom+1), ShadowBrush);
              FillRect (DC, Rect(Left-1, Bottom, Right+1, Bottom+1), ShadowBrush);
            end;
            etSunkenInner: with R do begin
              FillRect (DC, Rect(Left, Top, Left+1, Bottom), BtnFaceBrush);
              FillRect (DC, Rect(Left, Top, Right, Top+1), BtnFaceBrush);
            end;
          end;
        end
        else begin //draw the regular border
          FrameRect (DC, R, BtnFaceBrush); //outer edge--clBtnFace
          InflateRect (R, -1, -1);
          With TWin97Control(F97Control) do
            If (not Enabled) then
              FrameRect(DC, R, WindowBrush)
            else
              FrameRect (DC, R, BtnFaceBrush); //1 pixel in--clBtnFace
        end;
      finally
        ReleaseDC (Handle, DC);
      end;
    end;
  except
  end;
end;

end.
