
//======================== FORM MIN & MAX SIZE CONTROL =========================
//
// Component to limit the size that a form can be resize to.  Used by simply
// placing the component on the form, setting the required size limits, and
// setting to active.
//
// Works by adding a new WndProc which looks for the WM_GETMINMAXINFO message.
// This has the lParam pointing to a TMinMaxInfo record which contains size
// information as follows:
//   ptMaxSize     - Specifies the maximized width (point.x) and the maximized
//                   height (point.y) of the window.
//   ptMaxPosition - Specifies the position of the left side of the maximized
//                   window (point.x) and the position of the top of the
//                   maximized window (point.y).
//   ptMinTrackSize- Specifies the minimum tracking width (point.x) and the
//                   minimum tracking height (point.y) of the window.
//   ptMaxTrackSize- Specifies the maximum tracking width (point.x) and the
//                   maximum tracking height (point.y) of the window.
//
// Therefore there are eight coordinates which can be modified.  When the new
// WndProc is called, these values can be modified by values preset as
// properties within the component.  The size limiting behaviour can be turned
// on and off by an Active property.  An event may be called, after the component
// has modified the TMinMaxInfo record, but before it is used by windows in
// resizing the window, thus the user can make further adjustments to the
// form size.  After resizing, the form's OnResize event is fired.
//
// Use the SettingMin and SettingMax properties at design time to drag the window
// to the required minimum and then required maximum size.
//
// This component controls the form size, not the client size.
//
// NB This component will "fight" with to TAspect component.  The second of this
// and TAspect placed on a form will win.
//
// Version 1.00  Grahame Marsh 19 January 1997:
//    Initial component
//
// Version 1.10  Colin Dawson  15 September 1997
//    Bug fixed - previously allowed track resize when zoomed
//    Enhanced  - design time min and max size setting allowed
//
// Freeware - you get it for free, I take nothing, I make no promises!
//
// Please feel free to contact me: grahame.s.marsh@courtaulds.com

unit
  Minmax;

interface

uses
  SysUtils, Messages, Classes, Forms, Windows;

type

// This event is called after this component has altered the MinMaxInfo but
// before the window is resized.
  TBeforeResizeEvent = procedure (Sender : TObject; var MinMaxInfo : TMinMaxInfo) of object;

// enables selection of which of the eight possible values in the MinMaxInfo
// should be be modified.
  TMinMaxChange = (mmMaximizedWidth, mmMaximizedHeight, mmMaximizedLeft, mmMaximizedTop,
                   mmMinTrackWidth,  mmMinTrackHeight,  mmMaxTrackWidth, mmMaxTrackHeight);
  TMinMaxChanges = set of TMinMaxChange;

  TMinMaxTrackSize = class (TPersistent)
  private
    FMinTrackWidth,
    FMinTrackHeight,
    FMaxTrackWidth,
    FMaxTrackHeight : integer;
  protected
    constructor Create; virtual;
  published
    property MinTrackWidth : integer read FMinTrackWidth write FMinTrackWidth;
    property MinTrackHeight : integer read FMinTrackHeight write FMinTrackHeight;
    property MaxTrackWidth : integer read FMaxTrackWidth write FMaxTrackWidth;
    property MaxTrackHeight : integer read FMaxTrackHeight write FMaxTrackHeight;
  end;

  TMaximizedInfo = class (TPersistent)
  private
    FMaximizedWidth,
    FMaximizedHeight,
    FMaximizedLeft,
    FMaximizedTop  : integer;
  protected
    constructor Create; virtual;
  published
    property MaximizedWidth : integer read FMaximizedWidth write FMaximizedWidth;
    property MaximizedHeight : integer read FMaximizedHeight write FMaximizedHeight;
    property MaximizedLeft : integer read FMaximizedLeft write FMaximizedLeft;
    property MaximizedTop : integer read FMaximizedTop write FMaximizedTop;
  end;

  TMinMax = class(TComponent)
  private
    FParent : THandle;                   // used to sub-class the form
    FOldDefWndProc,
    FNewDefWndProc : pointer;
    FActive : boolean;                   // is the component active
    fSettingMin : boolean;               // Design time only Sets the MinTrack Size
    fSettingMax : boolean;               // Design time only Sets the MaxTrack Size
    fMinMaxTrackSize : TMinMaxTrackSize;
    fMaximizedInfo : TMaximizedInfo;
    FChanges : TMinMaxChanges;           // which value to be modified
    FBeforeResize : TBeforeResizeEvent;  // before resize event
    procedure NewDefWndProc (var Msg : TMessage);
  protected
    constructor Create (AOwner : TComponent); override;
    destructor Destroy; override;
  public
  published
    property MinMaxTrackSize : TMinMaxTrackSize read fMinMaxTrackSize write fMinMaxTrackSize;
    property MaximizedInfo : TMaximizedInfo read fMaximizedInfo write fMaximizedInfo;
    property Active : boolean read FActive write FActive default true;
    property Changes : TMinMaxChanges read FChanges write FChanges default [];
    property OnBeforeResize : TBeforeResizeEvent read FBeforeResize write FBeforeResize;
    Property SetMinTrackSize : Boolean read fSettingMin write fSettingMin;
    Property SetMaxTrackSize : Boolean read fSettingMax write fSettingMax;
  end;

implementation

constructor TMinMaxTrackSize.Create;
begin
  Inherited Create;
  FMaxTrackWidth   := GetSystemMetrics (sm_CXScreen) - 2 * (-GetSystemMetrics (sm_CXFrame));
  FMaxTrackHeight  := GetSystemMetrics (sm_CYScreen) - 2 * (-GetSystemMetrics (sm_CXFrame));
  FMinTrackWidth   := GetSystemMetrics (sm_CXMin);
  FMinTrackHeight  := GetSystemMetrics (sm_CYMin)
end;

constructor TMaximizedInfo.Create;
begin
  inherited Create;
  FMaximizedLeft   := - GetSystemMetrics (sm_CXFrame);
  FMaximizedTop    := - GetSystemMetrics (sm_CXFrame);
  FMaximizedWidth  := GetSystemMetrics (sm_CXScreen) - 2 * FMaximizedLeft;
  FMaximizedHeight := GetSystemMetrics (sm_CYScreen) - 2 * FMaximizedTop
end;

constructor TMinMax.Create (AOwner : TComponent);
begin
  inherited Create (AOwner);
//Create Component Properties
  fMinMaxTrackSize := TMinMaxTrackSize.Create;
  fMaximizedInfo := TMaximizedInfo.Create;

// the first part of subclassing a form
  FParent := (AOwner as TForm).Handle;
  FNewDefWndProc := MakeObjectInstance (NewDefWndProc);

// assume an active component
  FActive := true;

// assume tracking changes
  FChanges := [mmMinTrackWidth,  mmMinTrackHeight,  mmMaxTrackWidth, mmMaxTrackHeight];

// put our WndProc in to use
  FOldDefWndProc := pointer(SetWindowLong (FParent, GWL_WndProc, longint(FNewDefWndProc)))
end;

destructor TMinMax.Destroy;
begin
// chuck property placeholder objects
  fMinMaxTrackSize.Free;
  fMaximizedInfo.Free;

// disconnect event
  FBeforeResize := nil;

// restore old WndProc
  SetWindowLong (FParent, GWL_WndProc, longint(FOldDefWndProc));

// chuck the object instance
  FreeObjectInstance (FNewDefWndProc);
  inherited Destroy
end;

// This is our WndProc
//             -------
procedure TMinMax.NewDefWndProc (var Msg : TMessage);
var
  Rect : TRect;
begin
  with Msg do
  begin
    if (Msg = WM_Size) and (WParam = SIZE_RESTORED) and
// Stop Min and Max track size when maximized
       (not IsZoomed (fParent)) and
// Only use this code the the form designer
       (csDesigning in ComponentState) and
// Get the Current Form Size
        GetWindowRect(fParent, Rect) then
    begin
      if fSettingMin then // If Setting the Form Minimum Size
      begin
        fMinMaxTrackSize.MinTrackWidth := Rect.Right - Rect.Left;
        fMinMaxTrackSize.MinTrackHeight := Rect.Bottom - Rect.Top
      end;
      if fSettingMax Then  // If Setting the Form Maximum Size
      begin
        fMinMaxTrackSize.MaxTrackWidth  := Rect.Right - Rect.Left;
        fMinMaxTrackSize.MaxTrackHeight := Rect.Bottom - Rect.Top
      end
    end;

// test for WM_GETMINMAXINFO and Active
    if (Msg = WM_GetMinMaxInfo) and FActive then
    begin
      with PMinMaxInfo (lParam)^ do
      begin

// ring the changes
        if mmMaximizedWidth in FChanges then
          ptMaxSize.X := fMaximizedInfo.MaximizedWidth;
        if mmMaximizedHeight in FChanges then
          ptMaxSize.Y := fMaximizedInfo.MaximizedHeight;
        if mmMaximizedLeft in FChanges then
          ptMaxPosition.X := fMaximizedInfo.MaximizedLeft;
        if mmMaximizedTop in FChanges then
          ptMaxPosition.Y := fMaximizedInfo.MaximizedTop;
        if not IsZoomed(fParent) then   // Stop Min and Max track size when maximized
        begin
          if not ((csDesigning in ComponentState) and fSettingMin) then
          begin
            if mmMinTrackWidth in FChanges then
              ptMinTrackSize.X := fMinMaxTrackSize.MinTrackWidth;
            if mmMinTrackHeight in FChanges then
              ptMinTrackSize.Y := fMinMaxTrackSize.MinTrackHeight
          end;
          if not ((csDesigning in ComponentState) And fSettingMax) then
          begin
            if mmMaxTrackWidth in FChanges then
              ptMaxTrackSize.X := fMinMaxTrackSize.MaxTrackWidth;
            if mmMaxTrackHeight in FChanges then
              ptMaxTrackSize.Y := fMinMaxTrackSize.MaxTrackHeight
          end
        end
      end;

// call the BeforeResize Event
      if Assigned (FBeforeResize) then
        FBeforeResize (Self, PMinMaxInfo (lParam)^);

// Done
      Result := 0
    end else

// otherwise call the old WndProc
      Result := CallWindowProc (FOldDefWndProc, FParent, Msg, wParam, lParam)
  end
end;

end.



