///////////////////////////////////////////////////////////
//     TCLabel Delphi Visual Component Library
//     Copyright (c) 2000 by Chipmunk
//     ******************************
//     Description : Varied-pattern Label Delphi's VCL
//     Version     : 1.0.0
//     Ancestor    : TCustomControl
//     Programmer  : Chipmunk
//     Date        : 27 Sept. 2000
//     ******************************
//     Illegal distribution is prohibited
//////////////////////////////////////////////////////////

unit CLabel;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

type
  TColorType = array[0..255] of TColor;
  TStyle = (sNormal,sBitmap,sVGradient,sHGradient);

  TCLabel = class(TCustomControl)
  private
    FPicture       : TPicture;
    FFont          : TFont;
    FStyle         : TStyle;
    FCaption       : TCaption;
    FOutlined      : Bool;
    FOutlineColor  : TColor;
    FOutlineWidth  : integer;
    FFromColor     : TColor;
    FToColor       : TColor;

    procedure SetCaption(Value : TCaption);
    procedure SetOutlined(Value : Bool);
    procedure SetPicture(Value : TPicture);
    procedure SetStyle(Value : TStyle);
    procedure SetOutlineWidth(Value : integer);
    procedure CreateRegion(ABitmap : TBitmap;var RGN : HRGN);
    procedure PaintAsNormal(ABitmap : TBitmap);
    procedure PaintAsBitmap(ABitmap : TBitmap);
    procedure PaintAsVGradient(ABitmap : TBitmap);
    procedure PaintAsHGradient(ABitmap : TBitmap);
    procedure SetFrameColor(Value : TColor);
    procedure SetFromColor(Value : TColor);
    procedure SetToColor(Value : TColor);

  protected
    procedure Paint;override;
    procedure WMSize(var Message:TMessage);message WM_SIZE;
    procedure CMFontChanged(var Message:TMessage);message CM_FONTCHANGED;
  public
    constructor Create(AOwner : TComponent);override;
    destructor Destroy;override;
  published
    property Font;
    property OutlineWidth:integer read FOutlineWidth write SetOutlineWidth;
    property Picture : TPicture read FPicture write SetPicture;
    property Style : TStyle read FStyle write SetStyle;
    property Enabled;
    property FromColor : TColor read FFromColor write SetFromColor;
    property ToColor : TColor read FToColor write SetToColor;
    property OutlineColor : TColor read FOutlineColor write SetFrameColor;
    property Caption:TCaption read FCaption write SetCaption;
    property Outlined:Bool read FOutlined write SetOutlined;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property ShowHint;
    property OnClick;
    property OnMouseDown;
    property OnMouseUp;
  end;

procedure Register;


implementation



constructor TCLabel.Create(AOwner : TComponent);
var
 i,CLabelCount : integer;
begin
   inherited Create(AOwner);
   ControlStyle := ControlStyle + [csReplicatable];
   FPicture       := TPicture.Create;
   FOutlineColor  := clWhite;
   FCaption       := 'Chipmunk';
   FFromColor     := clBlack;
   FToColor       := clYellow;
   FFont          := Font;
   FFont.Size     := 48;
   FFont.Name     := 'Arial';
   FFont.Style    := [fsBold];
   FOutlineWidth  := 1;
   Width          := 320;
   Height         := 100;
   FStyle         := sVGradient;
   CLabelCount := 0;
   for i:=0 to AOwner.ComponentCount-1 do
    begin
     if (AOwner.Components[i] is TCLabel) then
      begin
        CLabelCount := CLabelCount +1;
      end;
    end;
   FCaption := 'CLabel'+IntToStr(CLabelCount);

end;

destructor TCLabel.Destroy;
begin
   FPicture.Free;
   inherited Destroy;
end;

procedure CalColor(FFromColor,FToColor:TColor;FNumberOfColors:integer;var FColors:TColorType);
var
  BeginRGB  : array[0..2] of Byte;
  RGBDifference  : array[0..2] of integer;
  R         : Byte;
  G         : Byte;
  B         : Byte;
  i: Byte;
begin
  BeginRGB[0] := GetRValue (ColorToRGB (FFromColor));
  BeginRGB[1] := GetGValue (ColorToRGB (FFromColor));
  BeginRGB[2] := GetBValue (ColorToRGB (FFromColor));

  RGBDifference[0] := GetRValue (ColorToRGB (FToColor)) - BeginRGB[0];
  RGBDifference[1] := GetGValue (ColorToRGB (FToColor)) - BeginRGB[1];
  RGBDifference[2] := GetBValue (ColorToRGB (FToColor)) - BeginRGB[2];
 for i := 0 to 255 do
 begin
    R := BeginRGB[0] + MulDiv (I, RGBDifference[0], FNumberOfColors - 1);
    G := BeginRGB[1] + MulDiv (I, RGBDifference[1], FNumberOfColors - 1);
    B := BeginRGB[2] + MulDiv (I, RGBDifference[2], FNumberOfColors - 1);
    FColors[i] := RGB (R, G, B);
 end;
end;

procedure TCLabel.CreateRegion(ABitmap : TBitmap;var RGN : HRGN);
var
 R : TRect;
begin
  R := ClientRect;
  with ABitmap.Canvas do
   begin
    Brush.Style:=bsClear;
    Font := FFont;
    R.Right := TextWidth(FCaption);
    R.Bottom := TextHeight(FCaption);
    SelectObject(Handle,Font.Handle);
    SelectObject(Handle,Brush.Handle);
    BeginPath(Handle);
     DrawText(Handle,PChar(FCaption),Length(FCaption),R,DT_CENTER or
              DT_VCENTER or DT_SINGLELINE);
    Endpath(Handle);
    SetBkMode(Handle,OPAQUE);
    RGN := PathToRegion(Handle);
    SelectClipRgn(Handle,RGN);
   end;
end;

procedure TCLabel.PaintAsNormal(ABitmap : TBitmap);
var
 H,W : integer;
 FRGN : HRGN;
begin
  CreateRegion(ABitmap,FRGN);
  with ABitmap.Canvas do
   begin
    W := TextWidth(FCaption);
    H := TextHeight(FCaption);
    Brush.Color := FFont.Color;
    Rectangle(0,0,W,H);
    if FOutlined then
     begin
      Brush.Color := FOutlineColor;
      FrameRgn(Handle,FRGN,Brush.Handle,FOutLineWidth,FOutLineWidth);
     end;
   end;
  DeleteObject(FRGN);
end;

procedure TCLabel.PaintAsBitmap(ABitmap : TBitmap);
var
 H,W : integer;
 FRGN : HRGN;
begin
  CreateRegion(ABitmap,FRGN);
  with ABitmap.Canvas do
   begin
    W := TextWidth(FCaption);
    H := TextHeight(FCaption);
    StretchBlt(Canvas.Handle,0,0,W,H,FPicture.Bitmap.Canvas.Handle,
               0,0,FPicture.Width,FPicture.Height,SRCCOPY);
    if FOutlined then
     begin
      Brush.Color := FOutlineColor;
      FrameRgn(Handle,FRGN,Brush.Handle,FOutLineWidth,FOutLineWidth);
     end;
  end;
  DeleteObject(FRGN);
end;

procedure TCLabel.PaintAsVGradient(ABitmap : TBitmap);
var
 H,W,i,Index    : integer;
 FRGN           : HRGN;
 FNumCol        : TColorType;
begin
  CreateRegion(ABitmap,FRGN);
  with ABitmap.Canvas do
   begin
    W := TextWidth(FCaption);
    H := TextHeight(FCaption);
    CalColor(FFromColor,FToColor,255,FNumCol);
    Index := 0;
    for i:=0 to H do
     begin
      MoveTo(0,i);
      Index := Round(i*255/H);
      Pen.Color := FNumCol[Index];
      LineTo(W,i);
     end;

    if FOutlined then
     begin
      Brush.Color := FOutlineColor;
      FrameRgn(Handle,FRGN,Brush.Handle,FOutLineWidth,FOutLineWidth);
     end;
  end;
  DeleteObject(FRGN);
end;

procedure TCLabel.PaintAsHGradient(ABitmap : TBitmap);
var
 H,W,i,Index    : integer;
 FRGN           : HRGN;
 FNumCol        : TColorType;
begin
  CreateRegion(ABitmap,FRGN);
  with ABitmap.Canvas do
   begin
    W := TextWidth(FCaption);
    H := TextHeight(FCaption);
    CalColor(FFromColor,FToColor,255,FNumCol);
    Index := 0;
    for i:=0 to W do
     begin
      MoveTo(i,0);
      Index := Round(i*255/W);
      Pen.Color := FNumCol[Index];
      LineTo(i,H);
     end;

    if FOutlined then
     begin
      Brush.Color := FOutlineColor;
      FrameRgn(Handle,FRGN,Brush.Handle,FOutLineWidth,FOutLineWidth);
     end;
  end;
  DeleteObject(FRGN);
end;

procedure TCLabel.Paint;
var
 OffScreen : TBitmap;
begin
  inherited Paint;
// Create OffScreen buffer to enhance performance
  try
   OffScreen:=TBitmap.Create;
   OffScreen.Width:=Width;
   OffScreen.Height:=Height;
   OffScreen.Canvas.Handle:=Canvas.Handle;
   case FStyle of
   sNormal    : PaintAsNormal(OffScreen);
   sVGradient : PaintAsVGradient(OffScreen);
   sHGradient : PaintAsHGradient(OffScreen);
   sBitmap    : PaintAsBitmap(OffScreen);
   end;
// Copy back to screen >><< BitBlt is faster than canvas Draw method
   BitBlt(Canvas.Handle,0,0,Width,Height,OffScreen.Canvas.Handle,0,0,SRCCOPY);
   finally
    OffScreen.Free;
  end;
end;

procedure TCLabel.WMSize(var Message:TMessage);
var
 H : integer;
begin
  Canvas.Font := FFont;
  H := Canvas.TextHeight(FCaption);
  Height := H;
end;

procedure TCLabel.CMFontChanged(var Message:TMessage);
begin
  FFont := Font;
  Refresh;
end;

procedure TCLabel.SetOutlineWidth(Value : integer);
begin
  if Value <> FOutlineWidth then
   begin
    FOutLineWidth := Value;
    Refresh;
   end;
end;

procedure TCLabel.SetCaption(Value : TCaption);
begin
  if Value <> FCaption then
   begin
    FCaption := Value;
    Refresh;
   end;
end;

procedure TCLabel.SetOutlined(Value : Bool);
begin
  if Value <> FOutlined then
   begin
    FOutlined := Value;
    Refresh;
   end;
end;

procedure TCLabel.SetFrameColor(Value : TColor);
begin
  if Value <> FOutlineColor then
   begin
    FOutlineColor := Value;
    Refresh;
   end;
end;

procedure TCLabel.SetPicture(Value : TPicture);
begin
  FPicture.Assign(Value);
  Refresh;
end;

procedure TCLabel.SetFromColor(Value : TColor);
begin
  if Value <> FFromColor then
   begin
    FFromColor := Value;
    Refresh;
   end;
end;

procedure TCLabel.SetToColor(Value : TColor);
begin
  if Value <> FToColor then
   begin
    FToColor := Value;
    Refresh;
   end;
end;

procedure TCLabel.SetStyle(Value : TStyle);
begin
  if Value <> FStyle then
   begin
    FStyle := Value;
    Refresh;
   end;
end;


procedure Register;
begin
  RegisterComponents('Chip2000', [TCLabel]);
end;

end.

// Phew.. that's all the stuff..