{ Animated Bitmap Component version 1.0 Copyright 1997 by A. Meeder

  This is a simple component to use bitmaps (as a single file) for animation, you can use it with the
  following versions op Delphi: 1.x, 2.x and 3.x (this a is the version with which I developed it)

  To use this component you must do the following (after installing the component)
  Place this component on a form, choose a bitmap, set cols, rows and active properties

  ATTENTION: You don't have to install ThdTimer.pas but it must be available when you use
             Delphi 2.x or 3.x (put it in the same directory). Don't try to use ThdTimer.pas with
             Delphi 1.x, this version of Delphi (Windows 3.x) don't allow threads

  PROPERTIES: (besides standard properties like name)
    Active (default false)
    AutoSize (default true)
    Bitmap (default empty)
    Cols (default 1)
    Direction (default drForward)
    Interval (default 500)
    Rows (default 1)

  EVENTS:
    OnTimer (every time when a new frame showed)

  Questions, suggestions etc. mail me: ameeder@dds.nl
}

unit AnimBmp;

interface

uses
  {$IFDEF WIN32}
    Windows, ThdTimer,
  {$ELSE}
    WinTypes, WinProcs,
  {$ENDIF}
    Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, ExtCtrls;

type
  TColRow = 1..MAXINT;
  TDirection = (drForward, drBackwards);

  TAnimateBmp = class(TCustomControl)
  private
    FActive: boolean;
    FAutoSize: boolean;
    FBitmap: TBitmap;
    FCols: TColRow;
    FDirection: TDirection;
    FInterval: integer;
    {$IFDEF WIN32}
      FPriority: TThreadPriority;
    {$ENDIF}
    FRows: TColRow;

    {$IFDEF WIN32}
      FTimer: TThreadedTimer;
    {$ELSE}
      FTimer: TTimer;
    {$ENDIF}
    FOnTimer: TNotifyEvent;

    FFrameWidth, FFrameHeight,
    X, Y: integer;

    procedure SetActive(value: boolean);
    procedure SetAutoSize(value: boolean);
    procedure SetBitmap(value: TBitmap);
    procedure SetDimensions(index: integer; value: TColRow);
    procedure SetDirection(value: TDirection);
    procedure SetInterval(value: integer);
    {$IFDEF WIN32}
      procedure SetPriority(value: TThreadPriority);
    {$ENDIF}

    procedure CountDimensions;
    procedure GetCurrentBitmap;
  protected
    procedure TimerTimer(Sender: TObject);
    procedure Loaded; override;
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Active: boolean read FActive write SetActive default false;
    property AutoSize: boolean read FAutoSize write SetAutoSize default true;
    property Bitmap: TBitmap read FBitmap write SetBitmap;
    property Cols: TColRow index 1 read FCols write SetDimensions default 1;
    property Direction: TDirection read FDirection write SetDirection default drForward;
    property Interval: integer read FInterval write SetInterval default 500;
    {$IFDEF WIN32}
      property Priority: TThreadPriority read FPriority write SetPriority default tpNormal;
    {$ENDIF}
    property Rows: TColRow index 2 read FRows write SetDimensions default 1;
    property OnTimer: TNotifyEvent read FOnTimer write FOnTimer;
  end;

procedure Register;

implementation

{$IFDEF WIN32}
procedure TAnimateBmp.SetPriority(value: TThreadPriority);
begin
  if value <> FPriority then
  begin
    FPriority := value;
    FTimer.Priority := value;
  end;
end;
{$ENDIF}


procedure TAnimateBmp.GetCurrentBitmap;
var
  DestRect, SrcRect: TRect;
begin
  DestRect := Rect(0, 0, FFrameWidth, FFrameHeight);
  SrcRect := Rect((X-1) * FFrameWidth, (Y-1) * FFrameHeight, X*FFrameWidth, Y*FFrameHeight);
  Canvas.CopyRect(DestRect, FBitmap.Canvas, SrcRect);
end;

procedure TAnimateBmp.CountDimensions;
begin
  if not FBitmap.Empty then
  begin
    FFrameWidth := FBitmap.Width div FCols;
    FFrameHeight := FBitmap.Height div FRows;

    if FAutoSize then
    begin
      Width := FFrameWidth;
      Height := FFrameHeight;
    end;

    case FDirection of
      drForward:   begin
                     X := 0;
                     Y := 1;
                   end;
      drBackwards: begin
                     X := FCols;
                     Y := FRows;
                   end;
    end;
  end;
end;

procedure TAnimateBmp.TimerTimer(Sender: TObject);
begin
  case FDirection of
    drForward:   begin
                   inc(X);
                   if X > FCols then
                   begin
                     X := 1;
                     inc(Y);
                   end;
                   if Y > FRows then
                   begin
                     Y := 1;
                     X := 1;
                   end;
                 end;
    drBackwards: begin
                   dec(x);
                   if X = 0 then
                   begin
                     X := FCols;
                     dec(Y);
                   end;
                   if Y = 0 then
                   begin
                     Y := FRows;
                     X := FCols;
                   end;
                 end;
  end;
  Repaint;
  if Assigned(FOnTimer) then FOnTimer(Sender);
end;

procedure TAnimateBmp.Loaded;
begin
  inherited Loaded;
  CountDimensions;
end;

procedure TAnimateBmp.Paint;
begin
  inherited Paint;
  canvas.brush.style := bsClear;
  if (csDesigning in ComponentState) then
  begin
    with canvas do
    begin
      pen.style := psDot;
      pen.width := 1;
      pen.color := clBlack;
      Rectangle(0, 0, Width, Height);
    end;
  end;
  GetCurrentBitmap;
end;

procedure TAnimateBmp.SetActive(value: boolean);
begin
  if value <> FActive then
  begin
    FActive := value;
    FTimer.Enabled := value;
    CountDimensions;
  end;
end;

procedure TAnimateBmp.SetAutoSize(value: boolean);
begin
  if value <> FAutoSize then
  begin
    FAutoSize := value;
    CountDimensions;
  end;
end;

procedure TAnimateBmp.SetBitmap(value: TBitmap);
begin
  if value <> FBitmap then
  begin
    FBitmap.Assign(value);
    CountDimensions;
    Repaint;
  end;
end;

procedure TAnimateBmp.SetDimensions(index: integer; value: TColRow);
begin
  case index of
    1: if value <> FCols then FCols := value;
    2: if value <> FRows then FRows := value;
  end;
  CountDimensions;
end;

procedure TAnimateBmp.SetDirection(value: TDirection);
begin
  if value <> FDirection then
  begin
    FDirection := value;
    CountDimensions;
  end;
end;

procedure TAnimateBmp.SetInterval(value: integer);
begin
  if value <> FInterval then
  begin
    FInterval := value;
    FTimer.Interval := value;
    if value = 0 then Active := false;
  end;
end;

constructor TAnimateBmp.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  SetBounds(0, 0, 50, 50);
  ControlStyle := ControlStyle + [csOpaque];
  FBitmap := TBitmap.Create;
  {$IFDEF WIN32}
    FTimer := TThreadedTimer.Create(Self);
    FPriority := tpNormal;
    FTimer.Priority := FPriority;
  {$ELSE}
    FTimer := TTimer.Create(self);
  {$ENDIF}
  Active := false;
  AutoSize := true;
  Cols := 1;
  Direction := drForward;
  Interval := 500;
  Rows := 1;
  FTimer.Enabled := FActive;
  FTimer.Interval := FInterval;
  FTimer.OnTimer := TimerTimer;
  X := 0;
  Y := 1;
end;

destructor TAnimateBmp.Destroy;
begin
  FBitmap.Free;
  FTimer.Free;
  inherited Destroy;
end;

procedure Register;
begin
  RegisterComponents('AM Controls', [TAnimateBmp]);
end;

end.
