# ======================================================================
# Credit Card Validation Solution, Perl Edition
#
# ======================================================================
# Ensures credit card information is keyed in correctly.
#
# Checks that the length is correct, the first four digits
# are within accepted ranges, the number passes the Mod 10 / Luhn
# checksum algorithm and that you accept the given type of card.  It
# also determines the card's type via the number's first four digits.
#
# The procedure has the option to check the card's expiration date.
#
# The text shown in this function's error messages come from variables
# defined by files in the ./language subdirectory.  These files are
# named after their ISO 639-1 two letter language code.  The language
# file used depends on which code is put in the $Language parameter.
#
# Just to be clear, this process does not check with banks or
# credit card companies to see if the card number given is actually
# associated with a good account. It just checks to see if the
# number matches the expected format.
#
# Warning: this function uses exact number ranges as part of the 
# validation process.  These ranges are current as of 30 July 2002.
# If presently undefined ranges come into use in the future,
# this program will improperly deject card numbers in such ranges,
# rendering an error saying "First four digits indicate unknown card
# type."  If this happens while entering a card and type you KNOW are
# valid, please contact us so we can update the ranges.
#
# Please consider making a donation to support our open source
# development:  http://www.AnalysisAndSolutions.com/donate/
# Update notifications are sent to people who make donations that exceed
# the small registration threshold.
#
# Credit Card Validation Solution is a trademark of The Analysis and
# Solutions Company.
#
# Several people deserve praise for the Credit Card Validation
# Solution. I learned of the Mod 10 Algorithm in some Perl code,
# entitled "The Validator," available on Matt's Script Archive,
# http://www.scriptarchive.com/ccver.html.  That code was written by
# David Paris, who based it on material Melvyn Myers reposted from an
# unknown author.  Paris credits Aries Solis for tracking down the data
# underlying the algorithm.  I pruned down the algorithm to it's core
# components, making things smaller, cleaner and more flexible.  Plus,
# I added the expiration date checking routine.  My first attemts at
# this were in Visual Basic, on which Allen Browne and Rico Zschau
# assisted. Neil Fraser helped a bit on the Perl version.  Steve
# Horsley, Roedy Green and Jon Skeet provided tips on the Java Edition.
#
# @param   string   $Number      the number of the credit card to
#                                  validate
# @param   string   $Language    the ISO 639-1 two letter code of
#                                  the language for error messages.
# @param   array    $Accepted    credit card types you accept.  If
#                                  not used in function call, all
#                                  known cards are accepted.  Set
#                                  it before calling the function:
#                                    my @A = array('Visa', 'JCB');
#                                  Known types:
#                                    American Express 
#                                    Australian BankCard 
#                                    Carte Blanche 
#                                    Diners Club 
#                                    Discover/Novus 
#                                    JCB 
#                                    MasterCard 
#                                    Visa
# @param   string   $RequireExp  should the expiration date be
#                                  checked?  Y or N.
# @param   integer  $Month       the card's expiration month
#                                  in M, 0M or MM foramt.
# @param   integer  $Year        the card's expiration year in YYYY format.
# @return  ingeger  1 if everything is fine.  0 if problems.
#
# @link    http://www.loc.gov/standards/iso639-2/langcodes.html
#
# ======================================================================
# Author     Daniel Convissor <danielc@AnalysisAndSolutions.com>
# Copyright  The Analysis and Solutions Company, 2002-2003
# Version    $Name: rel-5-12 $ $Id: ccvs.pm,v 1.18 2003/08/05 16:59:53 danielc Exp $
# Link       http://www.ccvs.info/
#
# ======================================================================
# SIMPLE PUBLIC LICENSE                        VERSION 1.1   2003-01-21
#
# Copyright (c) The Analysis and Solutions Company
# http://www.analysisandsolutions.com/
#
# 1.  Permission to use, copy, modify, and distribute this software and
# its documentation, with or without modification, for any purpose and
# without fee or royalty is hereby granted, provided that you include
# the following on ALL copies of the software and documentation or
# portions thereof, including modifications, that you make:
#
#     a.  The full text of this license in a location viewable to users
#     of the redistributed or derivative work.
#
#     b.  Notice of any changes or modifications to the files,
#     including the date changes were made.
#
# 2.  The name, servicemarks and trademarks of the copyright holders
# may NOT be used in advertising or publicity pertaining to the
# software without specific, written prior permission.
#
# 3.  Title to copyright in this software and any associated
# documentation will at all times remain with copyright holders.
#
# 4.  THIS SOFTWARE AND DOCUMENTATION IS PROVIDED "AS IS," AND
# COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO, WARRANTIES OF MERCHANTABILITY
# OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE OF THE SOFTWARE
# OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD PARTY PATENTS,
# COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
#
# 5.  COPYRIGHT HOLDERS WILL NOT BE LIABLE FOR ANY DAMAGES, INCLUDING
# BUT NOT LIMITED TO, DIRECT, INDIRECT, SPECIAL OR CONSEQUENTIAL,
# ARISING OUT OF ANY USE OF THE SOFTWARE OR DOCUMENTATION.
# ======================================================================



package CreditCardValidationSolution;

use File::Spec::Functions;
use File::Basename;
use File::stat;


sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto;
    my $self = {};
    $self->{CCVSNumber} = '';
    bless ($self, $class);
    return $self;
}


sub validateCreditCard(\$;\$\@\$\$\$) {

    my $self         = shift;
    my $NumberLength = 0;
    my $ShouldLength = 0;
    my $DoChecksum   = 'Y';

    #  The credit card number with all non-numeric characters removed.
    $self->{CCVSNumber} = '';

    #  The first four digits of the card.
    $self->{CCVSNumberLeft} = '';

    #  The card's last four digits.
    $self->{CCVSNumberRight} = '';

    #  The name of the type of card presented.  Automatically determined
    #  from the first four digits of the card number.
    $self->{CCVSType} = '';

    #  If the $RequireExp parameter is Y, the expiration date is valid, 
    #  and there are no other problems with the card number, this variable 
    #  contains the expiration date in MMYY format.
    $self->{CCVSExpiration} = '';

    #  String explaining the first problem detected, if any.
    $self->{CCVSError} = '';

    my($Number, $Language, $Accepted, $RequireExp, $Month, $Year) = @_;

    # Check/import language preferences.

    if ($Language eq 'de') {
        eval('use language::ccvs_de');
    } elsif ($Language eq 'es') {
        eval('use language::ccvs_es');
    } elsif ($Language eq 'fr') {
        eval('use language::ccvs_fr');
    } else {
        eval('use language::ccvs_en');
    }

    #  Catch malformed input.
    if ( !defined($Number) ) {
        $self->{CCVSError} = $CCVSErrNumberString;
        return 0;
    }

    #  Ensure number doesn't overrun.
    $Number = substr($Number, 0, 30);

    #  Remove non-numeric characters.
    $Number =~ tr/0-9//cd;

    #  Set up variables.

    $self->{CCVSNumber}      = $Number;
    $self->{CCVSNumberLeft}  = substr($self->{CCVSNumber}, 0, 4);
    $self->{CCVSNumberRight} = substr($self->{CCVSNumber}, -4);
    $NumberLength            = length($Number);

    #  Determine the card type and appropriate length.

    RANGE: {
        if ( ($self->{CCVSNumberLeft} >= 3000) &&
             ($self->{CCVSNumberLeft} <= 3059) ) {
            $self->{CCVSType} = 'Diners Club';
            $ShouldLength = 14;
            last RANGE;
        }
        if ( ($self->{CCVSNumberLeft} >= 3600) &&
             ($self->{CCVSNumberLeft} <= 3699) ) {
            $self->{CCVSType} = 'Diners Club';
            $ShouldLength = 14;
            last RANGE;
        }
        if ( ($self->{CCVSNumberLeft} >= 3800) &&
             ($self->{CCVSNumberLeft} <= 3889) ) {
            $self->{CCVSType} = 'Diners Club';
            $ShouldLength = 14;
            last RANGE;
        }

        if ( ($self->{CCVSNumberLeft} >= 3400) &&
             ($self->{CCVSNumberLeft} <= 3499) ) {
            $self->{CCVSType} = 'American Express';
            $ShouldLength = 15;
            last RANGE;
        }
        if ( ($self->{CCVSNumberLeft} >= 3700) &&
             ($self->{CCVSNumberLeft} <= 3799) ) {
            $self->{CCVSType} = 'American Express';
            $ShouldLength = 15;
            last RANGE;
        }

        if ( ($self->{CCVSNumberLeft} >= 3088) &&
             ($self->{CCVSNumberLeft} <= 3094) ) {
            $self->{CCVSType} = 'JCB';
            $ShouldLength = 16;
            last RANGE;
        }
        if ( ($self->{CCVSNumberLeft} >= 3096) &&
             ($self->{CCVSNumberLeft} <= 3102) ) {
            $self->{CCVSType} = 'JCB';
            $ShouldLength = 16;
            last RANGE;
        }
        if ( ($self->{CCVSNumberLeft} >= 3112) &&
             ($self->{CCVSNumberLeft} <= 3120) ) {
            $self->{CCVSType} = 'JCB';
            $ShouldLength = 16;
            last RANGE;
        }
        if ( ($self->{CCVSNumberLeft} >= 3158) &&
             ($self->{CCVSNumberLeft} <= 3159) ) {
            $self->{CCVSType} = 'JCB';
            $ShouldLength = 16;
            last RANGE;
        }
        if ( ($self->{CCVSNumberLeft} >= 3337) &&
             ($self->{CCVSNumberLeft} <= 3349) ) {
            $self->{CCVSType} = 'JCB';
            $ShouldLength = 16;
            last RANGE;
        }
        if ( ($self->{CCVSNumberLeft} >= 3528) &&
             ($self->{CCVSNumberLeft} <= 3589) ) {
            $self->{CCVSType} = 'JCB';
            $ShouldLength = 16;
            last RANGE;
        }

        if ( ($self->{CCVSNumberLeft} >= 3890) &&
             ($self->{CCVSNumberLeft} <= 3899) ) {
            $self->{CCVSType} = 'Carte Blanche';
            $ShouldLength = 14;
            last RANGE;
        }

        if ( ($self->{CCVSNumberLeft} >= 4000) &&
             ($self->{CCVSNumberLeft} <= 4999) ) {
            $self->{CCVSType} = 'Visa';
            VISALENGTH: {
                if ($NumberLength > 14) {
                    $ShouldLength = 16;
                    last VISALENGTH;
                }
                if ($NumberLength < 14) {
                    $ShouldLength = 13;
                    last VISALENGTH;
                }
                $self->{CCVSError} = $CCVSErrVisa14;
                return 0;
            }
            last RANGE;
        }

        if ( ($self->{CCVSNumberLeft} >= 5100) &&
             ($self->{CCVSNumberLeft} <= 5599) ) {
            $self->{CCVSType} = 'MasterCard';
            $ShouldLength = 16;
            last RANGE;
        }

        if ($self->{CCVSNumberLeft} == 5610) {
            $self->{CCVSType} = 'Australian BankCard';
            $ShouldLength = 16;
            last RANGE;
        }

        if ($self->{CCVSNumberLeft} == 6011) {
            $self->{CCVSType} = 'Discover/Novus';
            $ShouldLength = 16;
            last RANGE;
        }

        $self->{CCVSError} = sprintf($CCVSErrUnknown, $self->{CCVSNumberLeft});
        return 0;
    }


    # Check acceptance.

    if ( defined($Accepted) ) {
        if ( ref($Accepted) ne 'ARRAY' ) {
            $self->{CCVSError} = $CCVSErrAccepted;
            return 0;
        }
        my $Accept = 0;
        for (@$Accepted) {
            if ($_ eq $self->{CCVSType}) {
                $Accept = 1;
            }
        }
        if (!$Accept) {
            $self->{CCVSError} = sprintf($CCVSErrNoAccept, $self->{CCVSType});
            return 0;
        }
    }

    # Check length.

    if ($NumberLength != $ShouldLength) {
        my $Missing = $NumberLength - $ShouldLength;
        if ($Missing < 0) {
            $self->{CCVSError} = sprintf($CCVSErrShort, abs($Missing));
        } else {
            $self->{CCVSError} = sprintf($CCVSErrLong, $Missing);
        }
        return 0;
    }


    # Mod10 checksum process...

    if ($DoChecksum eq 'Y') {

        my $Location = 0;
        my $Checksum = 0;
        my $Digit    = '';

        # Add even digits in even length strings
        # or odd digits in odd length strings.
        for ($Location = 1 - ($NumberLength % 2);
                 $Location < $NumberLength; $Location += 2) {
            $Checksum += substr($self->{CCVSNumber}, $Location, 1);
        }

        # Analyze odd digits in even length strings
        # or even digits in odd length strings.
        for ($Location = ($NumberLength % 2);
                 $Location < $NumberLength; $Location += 2) {
            $Digit = substr($self->{CCVSNumber}, $Location, 1) * 2;
            if ($Digit < 10) {
                $Checksum += $Digit;
            } else {
                $Checksum += $Digit - 9;
            }
        }

        # Checksums not divisible by 10 are bad.
        if ($Checksum % 10 != 0) {
            $self->{CCVSError} = $CCVSErrChecksum;
            return 0;
        }

    }

    # Expiration date process...

    if ( (defined($RequireExp)) && ($RequireExp eq 'Y') ) {

        if ($Month !~ /^(0?[1-9]|1[0-2])$/) {
            $self->{CCVSError} = $CCVSErrMonthFormat;
            return 0;
        }

        if ($Year !~ /^[0-9]{4}$/) {
            $self->{CCVSError} = $CCVSErrYearFormat;
            return 0;
        }

        my ($sec, $min, $hour, $mday, $CurrentMonth, $CurrentYear, $wday,
                $yday, $isdst) = localtime;
        $CurrentYear  += 1900;
        $CurrentMonth += 1;

        if ($Year < $CurrentYear) {
            $self->{CCVSError} = $CCVSErrExpired;
            return 0;
        }

        if ($Year == $CurrentYear) {
            if ($Month < $CurrentMonth) {
                $self->{CCVSError} = $CCVSErrExpired;
                return 0;
            }
        }

        $self->{CCVSExpiration} = sprintf('%02d', $Month) . substr($Year, -2);

    }

    return 1;

}

1;
