#   Binary Concepts Bulletin Board System
#   Author: Brian Kirkland

#   --- layout.pm --- 
#   Layout Module

# Layout Format
#  - Primary Routines are called by most of of the layouts. These functions are
#    the basic headers, footers, and other commonly included components.
#  - Specific Routines are the specific layouts for each section of the site


# --- default global variables ---

$_bgcolor       = "#ffffff";    # <body bgcolor>
$_altbgcolor    = "#5E84A8";    # bgcolor to use on user query, error messages etc...
$_linkcolor     = "#005697";    # <body link>
$_vlinkcolor    = "#003862";    # <body vlink>
$_alinkcolor    = "#FF0000";    # <body alink>
$_sidebgcolor   = "#E4E4E4";    # <body bgcolor> for the left/side navigation frame
$_hovercolor    = "#FF0000";    # CSS hover color
$_titlecolor    = "#005697";    # Title text color
$_boldcategory  = "#000000";    # Left frame category links
$_tableborder   = "#d3d3d3";    # Generic table borders
$_tablebg       = "#f8f8f8";    # Generic table backgrounds
$_alttablebg    = "#ededed";    # Alternate table background
$_previewcolor  = "#444444";    # "Show previews" text color
$_linkbarcolor  = "#e6e6e6";    # Post, refresh, etc. Background color
$_topbarcolor   = "#E6E6E6";    # Top title bar bgcolor
$_errorcolor    = "#660000";    # Color to display error text with
$_idcolor       = "#666666";    # Color to display board/article/reply IDs with

$_fontsmall     = "1";          # Size of small text, font tag
$_fontregular   = "2";          # Size of normal text, font tag
$_fontheadersize= "4";          # Size of post/reply header text, font tag (used also with $_titlecolor)
$_fonttitlesize = "5";          # Size of title text, font tag (used also with $_titlecolor)
$_fontmenusize  = "2";          # Size of the Font Menu

# Use comma delimited list to specify backup fonts
$_fontmenu      = "Verdana, Arial, Helvetica, Geneva";    # Font used in the size Menu
$_font          = "Verdana, Arial, Helvetica, Geneva";    # Font used throughout the boards

$_frameborder   = "no";         # <frame frameborder>
$_framebdrsize  = "0";          # <frame border>
$_framespacing  = "0";          # <frame framespacing>
$_frameleft     = "176";        # Pixel size of left frame
$_frametopshow  = "91";         # Pixel size of top frame when header image is showing
$_frametophide  = "25";         # Pixel size of top frame when header image is hidden

$_linkbarmargin = "23";         # Distance from left side of the browser to the first item in the nav menu 
$_linkbarseparator = " | ";     # Character(s) (may also be HTML) that seperate navigation items

$_textareacols = "60";          # Post/Reply textarea width (columns)
$_textarearows = "10";          # Post/Reply textarea height (rows)


# Date & Time Preferences
$_timezone_DL   = "PDT";        # Timezone name when time is in daylight savings
$_timezone      = "PST";        # Timezone name when in not in daylight savings
$_am            = "am";         # AM symbol
$_pm            = "pm";         # PM symbol
$_date_zeros    = 0;            # Pads month and day with zeros (eg. 01/02/2002 vs. 1/2/2002)
$_hour_zero     = 0;            # Pads hour with extra zero (eg. 1:03 vs. 01:03) 
$_24hour        = 0;            # 24-hour clock, 0 for no, 1 for yes
$_2digityear    = 1;            # Make the year two digits instead of the normal 4 (eg "02" instead of "2002")            
$_month_name    = 0;            # Use "Jan, Feb, ..." instead of "1,2,..." for months
$_fuzzydate     = 1;
$_zoneoffset    = 0;            # If the server you have is displaying the wrong time, use this 
                                #   to offset it by a given number of minutes (eg. if you want Eastern and the 
                                #   server is Pacific, make _zoneoffset 180), value can be negative

# --- global string values ---

# Month names, only if $_month_name = 1
@_month_names = ("Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec");

# Names and Titles used throughout the Board Layout
%_str_names = (
    'login'                     => "Board Login",
    'userquery'                 => "User Query",
    'preferences'               => "User Preferences",
    'statistics'                => "Database Statistics",
    'search'                    => "Search",
    'post_to'                   => "Post to",
    'reply_to'                  => "Reply to",
    'posts'                     => "Posts",
    'hits'                      => "Page Views",
    'access_list'               => "Access List",
    'account_create'            => "Account Creation",
    'account_created'           => "Account Created",
    'username'                  => "Username",
    'branch'                    => "Branch",
    'board'                     => "Board",
    'user'                      => "User",
    'guest'                     => "GUEST",

    # Links on Top NavBar
    'top_home'                  => "Home",
    'top_admin'                 => "Admin",
    'top_search'                => "Search",
    'top_prefs'                 => "Preferences",
    'top_stats'                 => "Statistics",
    'top_query'                 => "User Query",
    'top_logoff'                => "Logoff",
    'top_shead'                 => "Show Header",
    'top_hhead'                 => "Hide Header",
    'top_sside'                 => "Show Sidebar",
    'top_hside'                 => "Hide Sidebar",
    'top_sboard'                => "Show All Boards",
    'top_hboard'                => "Hide All Boards",

    # Links on TOC
    'toc_post'                  => "post",
    'toc_refresh'               => "refresh",
    'toc_show'                  => "show",
    'toc_hide'                  => "hide",
    'toc_previews'              => "previews",
    'toc_markallread'           => "mark all as read",
);

# Descriptions & Other Text
%_str_text = (
    'post'                      => "Post Message",          # Submit button text
    'clear'                     => "Clear",                     
    'login'                     => "Go Login!",             # Login link after registering
    'remember'                  => "Remember my ID on this computer",
    'login_note'                => "To bookmark the boards, bookmark this page",
    'statistics_desc'           => "Number of Posts and Hits for the Board System.",
    'query_desc'                => "Get information on a particular user.",
    'preferences_desc'          => "Change your personal preferences.",
    'search_desc'               => "Search the boards by keyword.",
    'backtoclink'               => "Back to TOC",            # Link from various sections $_str_text{'backtoclink'}
    'backthreadlink'            => "Back to this thread",    # Link back to the current thread (from reply page)
    'password_changed'          => "Password changed. If you had auto-login enabled, you will have to log off and log back in.",
    'bb_add_fav'                => "BB added to favorites.",
    'bb_del_fav'                => "BB removed from favorites.",
    'prefs_updated'             => "Preferences updated.",
    'link_option'               => "Turn URLs into links",
    'emoticon_option'           => "Use emoticons",
    'fuzzydate_today'           => "Today,",
    'fuzzydate_yesterday'       => "Yesterday,",
    'board_id'                  => "Board ID",
    'article_id'                => "Article ID",
    'reply_id'                  => "Reply ID",

);


# Error messages
%_str_errors = (
    'login'                     => qq{Invalid password, your cookie expired, or you haven't logged on before. Make sure cookies are enabled before continuing.},
    'subject_length'            => qq{Subjects must be $_subjectminlen-$_subjectmaxlen characters in length.},
    'writer_length'             => qq{The "From" field must be $_writerminlen-$_writermaxlen characters in length.},
    'guest_no_post'             => qq{Guests are restricted from posting.},
    'locked_out'                => qq{Your account has been disabled by a board administrator.},
    'pass_too_short'            => qq{Passwords mut be between $_passwordminlen-$_passwordmaxlen characters in length.},
    'pass_too_long'             => qq{Passwords mut be between $_passwordminlen-$_passwordmaxlen characters in length.},
    'user_too_short'            => qq{Usernames must be $_usernameminlen-$_usernamemaxlen characters in length.},
    'user_too_long'             => qq{Usernames must be $_usernameminlen-$_usernamemaxlen characters in length.},
    'user_characters'           => qq{Usernames can only contain letters A-Z, numbers 0-9, and the underscore (_) character.},
    'password_mismatch'         => qq{Passwords don't match.},
);

$_emoticonheight = 15;
$_emoticonwidth = 15;

%_emoticons = (
    ':-)'                       => "smile1.gif",
    ':)'                        => "smile1.gif",
    ':-('                       => "sad1.gif",
    ':('                        => "sad1.gif",
    '>:0'                       => "angry1.gif",
);

$_censor = "*";

# Censored words are replaced ONLY if they're surrounded by spaces or other whitespace (eg ass vs. associate)
@_censored_words = (

    #... add words here
);

# Superwords are replaced no matter where they're located in the string
@_censored_superwords = (

    #... add words here
);



# --- primary routines ---

sub _header {

    # Parameter List
    my $Title = shift;          # Page Title
    my $Background = shift;     # (optional) It overrides the global bgcolor
    my $CSSExtra = shift;       # (optional) Extra CSS code for specific layouts
    my $BGExtra = shift;        # (optional) Extra background information

    # if background wasn't specified, default to global
    if (! $Background) {$Background = $_bgcolor;}

    return qq{
        <html>
        <head>
        <title>$Title</title>

        <style type="text/css">
        <!--
        A:link     { color: $_linkcolor; text-decoration: none }
        A:visited  { color: $_vlinkcolor; text-decoration: none }
        A:hover   {text-decoration: underline; color: $_hovercolor}
        $CSSExtra
        -->
        </style>
        </head>
        <body bgcolor="$Background" leftmargin="0" marginwidth="0" topmargin="0" marginheight="0" link="$_linkcolor" vlink="$_vlinkcolor" alink="$_alinkcolor" $BGExtra>
    };

}



sub _footer {

    my $timecode = "";
    if ($_usetime != 0) {
        use Time::HiRes qw(gettimeofday);
        ## TIME CODE
        my $time = gettimeofday - $_initial_time;
        $time = int($time * 100000) / 100000;
        

        if ($_initial_time != "") {
            $timecode = qq{
                <br><br><br>
                <table bgcolor="white" align="center"><tr><td align="center"><font face="Verdana" size="-2">
                [ $time seconds ]<br>
                </font></td></tr></table> 
            };
        }
        ##
    }

    return qq{
        $timecode
        </body></html>
    };

}



sub _error
{

    my $Error_Msg = shift;

    print "Content-type: text/html\n\n";
    print qq{
        <html>
            <body>
                <hr>
                <h1>Script Error</h1>
                <P>$Error_Msg
            </body>
        </html>
    };

    exit;

}




sub _descerror
{

    my $Error_Title = shift;
    my $Error_Msg = shift;

    print "Content-type: text/html\n\n";
    print &_header("$_title Error",$_altbgcolor);

    print qq{
        <center><br><br><br>
        <table border="0" cellpadding="15" cellspacing="0" width="100%" height="90%">
        <tr><td>
        <table border="0" cellpadding="0" cellspacing="2" width="100%" bgcolor="$_tableborder" align="center">
            <tr>
                <td>
                <table border="0" cellpadding="3" cellspacing="2" width="100%">
                    <tr>
                        <td bgcolor="$_tablebg">
                        <center><br>
                        <font face="$_font" size="$_fontregular" color="$_errorcolor">
                        <b>$Error_Title</b><BR><BR><i>$Error_Msg</i></font><br><br>
                        </center>
                        </td>
                    </tr>
                </table>
                </td>
            </tr>
        </table>
        </td></tr></table>
    };

    print &_footer;

    exit;

}







# --- specific routines ---



sub _layoutTopWithHeader() {

    # Parameters
    my $Home = shift;           # Home link
    my $Administration = shift; # Link to Admin Console (blank if user isnt admin)
    my $Search = shift;         # Search link
    my $Preferences = shift;    # Preferences link
    my $UserQuery = shift;      # User Query link
    my $Statistics = shift;     # Statistics link
    my $Header = shift;         # Show/Hide Header link
    my $Sidebar = shift;        # Show/Hide Sidebar link
    my $All_Boards = shift;     # Show/Hide All Boards link
    my $Logoff = shift;         # Logoff link

    # Add a separator if the link actually exists
    if ($Administration ne "") {
        $Administration .= $_linkbarseparator;
    }

    print "Content-type: text/html\n\n";
    print &_header("$_header Login","","","background=\"$_imgpath/topbg.gif\"");

    print qq{
        <table border="0" cellpadding="0" cellspacing="0" width="100%">
            <tr>
                <td width="587"><img src="$_imgpath/top.gif" border="0"></td>
                <td width="100%"></td>
            </tr>
        </table>
        <table border="0" cellpadding="0" cellspacing="0" width="100%"  background="$_imgpath/menubg.gif">
            <tr height="$_frametophide">
                <td align="left" height="$_frametophide"><img src="$_imgpath/blank.gif" width="$_linkbarmargin" height="15" border="0" align="absmiddle"><font size="1" face="$_font">
                    $Home$_linkbarseparator
                    $Administration
                    $Search$_linkbarseparator 
                    $Preferences$_linkbarseparator 
                    $UserQuery$_linkbarseparator
                    $Statistics$_linkbarseparator
                    $Header$_linkbarseparator
                    $Sidebar$_linkbarseparator
                    $All_Boards$_linkbarseparator
                    $Logoff
                </td>
            </tr>
        </table>
    };

    print &_footer;

}





sub _layoutTopWithoutHeader() {

    # Parameters
    my $Home = shift;           # Home link
    my $Administration = shift; # Link to Admin Console (blank if user isnt admin)
    my $Search = shift;         # Search link
    my $Preferences = shift;    # Preferences link
    my $UserQuery = shift;      # User Query link
    my $Statistics = shift;     # Statistics link
    my $Header = shift;         # Show/Hide Header link
    my $Sidebar = shift;        # Show/Hide Sidebar link
    my $All_Boards = shift;     # Show/Hide All Boards link
    my $Logoff = shift;         # Logoff link

    # Add a separator if the link actually exists
    if ($Administration ne "") {
        $Administration .= $_linkbarseparator;
    }

    print "Content-type: text/html\n\n";
    print &_header("$_header Login");

print qq{
        <table border="0" cellpadding="0" cellspacing="0" width="100%" background="$_imgpath/menubg.gif" >
            <tr height="$_frametophide">
                <td align="left" height="$_frametophide"><img src="$_imgpath/blank.gif" width="$_linkbarmargin" height="15" border="0" align="absmiddle"><font size="1" face="$_font">
                    $Home$_linkbarseparator
                    $Administration
                    $Search$_linkbarseparator 
                    $Preferences$_linkbarseparator 
                    $UserQuery$_linkbarseparator
                    $Statistics$_linkbarseparator
                    $Header$_linkbarseparator
                    $Sidebar$_linkbarseparator
                    $All_Boards$_linkbarseparator
                    $Logoff
                </td>
            </tr>
        </table>
    };

    print &_footer;

}



sub _layoutLeft() {

    # Parameters
    my $BoardList = shift;        # List of BBs

    print "Content-type: text/html\n\n";
    print &_header("$_header","E6E6E6","body {background: url($_imgpath/side.gif) fixed; margin-left: 10px; margin-top: 10px;}");

    print qq{
        <p><font size="$_fontmenusize" face="$_fontmenu">$BoardList</font>
    };

    print &_footer;

}




sub _layoutMain() {

    # Parameters
    my $Sidebar = shift;            # To display boards in left table column if sidebar is hidden
    my $FileContents = shift;       # Contents of news/announcements file, editable through admin interface

    print "Content-type: text/html\n\n";
    print &_header("$_header",$_altbgcolor);

    print qq{
        <table width="100%" cellpadding="4" border="0"><tr><td>
        $Sidebar
        </td><td>
        <br><table border="0" cellpadding="15" cellspacing="0" width="100%" height="100%">
        <table border="0" width="90%" height="90%" bgcolor="$_tableborder" align="center">
            <tr>
                <td bgcolor="$_tablebg" valign="top">
                    <br>
                    <center><font face="$_font" size="$_fonttitlesize" color="$_titlecolor"><b>$_title</b></font></center><br>
                    <table>
                        <tr>
                            <td>
                                $FileContents
                            </td>
                        </tr>
                    </table>
                </td>
            </tr>
            <tr>
                <td bgcolor="$_tablebg" height="100">
                <table width="100%">
                    <tr align="center">
                        <td>
                            <a href="http://www.binary-concepts.com" target="_new"><img src="$_imgpath/bcproject.gif" border=0></a>
                        </td>
                        <td>
                            <font face="$_font" size="$_fontsmall">
                            <b>$_header</b><br>
                            Version $_version<br>
                            $_release Release<br>
                            <br>
                            &copy; 2002 <a href="http://binary-concepts.com/kirkland" target="_blank">Brian Kirkland</a> &amp; <a href="http://binary-concepts.com/cory" target="_blank">Cory Mickelson</a><br>
                        </td>
                        <td>
                            <a href="http://www.perl.com" target="_new"><img src="$_imgpath/perlpower.gif" border=0></a>
                        </td>
                    </tr>
                </table>
                </td>
            </tr>
        </table>
        </table>

        </td></tr></table>

    };


    print &_footer;

}





sub _layoutPrefs() {

    # Parameters
    my $Username = shift;
    my $First_Name = shift;
    my $Last_Name = shift;
    my $Location = shift;
    my $Email = shift;
    my $Show_Header = shift;    # Drop-down box, show header, yes/no
    my $Show_Sidebar = shift;   # Drop-down box, show sidebar, yes/no
    my $Show_Boards = shift;    # Drop-down box, show all boards, yes/no
    my $Del_Fav_List = shift;   # Drop down options for deleting from favorites
    my $Add_Fav_List = shift;   # Drop-down options for adding to favorites
    my $Status_Message = shift; # Message describing a completed action, if any

    print "Content-type: text/html\n\n";
    print &_header("$_header",$_altbgcolor);

    print qq{
        <table border="0" cellpadding="15" cellspacing="0" width="100%" height="100%">
            <tr>
                <td>
                <table border="0" width="60%" cellpadding="0" cellspacing="2" bgcolor="$_tableborder" align="center">
                <tr><td>
                <table border="0" width="100%" height="100%" cellpadding="5"><tr><td bgcolor="$_tablebg">
                <font face="$_font" size="$_fonttitlesize" color="$_titlecolor"><b>$_str_names{'preferences'}</b></font><br>
                <font face="$_font" size="$_fontsmall">$_str_text{'preferences_desc'}</font>
                </td></tr>
                 <tr><td bgcolor="$_tablebg" align="center">
                <form method="POST" action="prefs.pl">
                <input type="hidden" name="action" value="changeprefs">
                <center><font face="$_font" size="$_fontregular"><b>$Status_Message</b></font></center>
                <table cellspacing="10" cellpadding="5">
                    <tr>
                        <td valign="top">
                        <table>
                            <tr>
                                <td align="right"><B><font face="$_font" size="$_fontregular">$_str_names{'username'}:</font></b></td>
                                <td><font face="$_font" size="$_fontregular">$Username</font></td>
                            </tr>
                            <tr>
                                <td align="right"><B><font face="$_font" size="$_fontregular">First Name:</font></b></td>
                                <td><font face="$_font" size="$_fontregular"><input type="text" name="first_name" value="$First_Name"></font></td>
                            </tr>
                            <tr>
                                <td align="right"><B><font face="$_font" size="$_fontregular">Last Name:</font></b></td>
                                <td><font face="$_font" size="$_fontregular"><input type="text" name="last_name" value="$Last_Name"></font></td>
                            </tr>
                            <tr>
                                <td align="right"><B><font face="$_font" size="$_fontregular">Location:</font></b></td>
                                <td><input type="text" name="location" value="$Location"></td>
                            </tr>
                            <tr>
                                <td align="right"><B><font face="$_font" size="$_fontregular">E-mail:</font></b></td>
                                <td><input type="text" name="email" value="$Email"></td>
                            </tr>
                            <tr>
                                <td align="right"><B><font face="$_font" size="$_fontregular">Show Header?</font></b></td>
                                <td>$Show_Header</td>
                            </tr>
                            <tr>
                                <td align="right"><B><font face="$_font" size="$_fontregular">Show Sidebar?</font></b></td>
                                <td>$Show_Sidebar</td>
                            </tr>
                            <tr>
                                <td align="right"><B><font face="$_font" size="$_fontregular">Show All Boards?</font></b></td>
                                <td>$Show_Boards</td>
                            </tr>
                            <tr>
                                <td colspan="2" align="center"><input type="submit" name="submit" value="Submit"></form></td>
                            </tr>
                            <tr height="20"><td colspan="2">&nbsp;</td></tr>
    };

    if ($_requireoldpw != 0) {
        print qq{
                            <tr>
                                <td align="right">
                                <form method="POST" action="prefs.pl">
                                    <input type="hidden" name="action" value="changepass">
                                    <B><font face="$_font" size="$_fontregular">Old Password:</font></b>
                            </td>
                                <td><input type="password" name="oldpass"></td>
                            </tr>
        };
    }

    print qq{
        
                            <tr>
                                <td align="right"><B><font face="$_font" size="$_fontregular">New Password:</font></b></td>
                                <td><input type="password" name="newpass"></td>
                            </tr>
                            <tr>
                                <td align="right"><B><font face="$_font" size="$_fontregular">Confirm Password:</font></b></td>
                                <td><input type="password" name="confpass"></td>
                            </tr>
                            <tr>
                                <td colspan="2" align="center"><input type="submit" name="submit" value="Change Password"></form></td>
                            </tr>
                            <tr height="20"><td colspan="2">&nbsp;</td></tr>
    };

    if (($Del_Fav_List ne "")&&($_favoriteslist != 0)) {
        print qq{
                            <tr><td colspan="2">
                                <form method="POST" action="prefs.pl">
                                    <input type="hidden" name="action" value="delfavbb">
                                    <font face="$_font" size="$_fontregular">Delete BB from Favorites:</font>
                            </td></tr>
                            <tr><td colspan="2">
                                    <select name="bb">$Del_Fav_List</select>
                                    <input type="submit" name="submit" value="Delete">
                                </form>
                            </td></tr>
        };
    }

    if (($Add_Fav_List ne "")&&($_favoriteslist != 0)) {
        print qq{
                            <tr><td colspan="2">
                                <form method="POST" action="prefs.pl">
                                    <input type="hidden" name="action" value="addfavbb">
                                    <font face="$_font" size="$_fontregular">Add BB to Favorites:</font>
                            </td></tr>
                            <tr><td colspan="2">
                                    <select name="bb">$Add_Fav_List</select>
                                    <input type="submit" name="submit" value="Add">
                                </form>
                            </td></tr>
        };
    }

    print qq{
                        </table>
                        </td>
                    </tr>
                </table>
                </td></tr></table>
                </td></tr></table>
                </td>
            </tr>
        </table>
    };

    print &_footer;


}



sub _layoutQuery() {

    # Parameters
    my $Search_Error_Msg = shift;   # If they get an error in the search

    print "Content-type: text/html\n\n";
    print &_header("$_header",$_altbgcolor);

    print qq{
        <table border="0" cellpadding="15" cellspacing="0" width="100%" height="100%">
            <tr>
                <td>
                <table border="0" width="60%" cellpadding="0" cellspacing="2" bgcolor="$_tableborder" align="center">
                <tr><td>
                <table border="0" width="100%" height="100%" cellpadding="5"><tr><td bgcolor="$_tablebg">
                <font face="$_font" size="$_fonttitlesize" color="$_titlecolor"><b>$_str_names{'userquery'}</b></font><br>
                <font face="$_font" size="$_fontsmall" color="$_fontregular">$_str_text{'query_desc'}</font>
                </td></tr>
                <tr><td bgcolor="$_tablebg">

                <div align="center"><font face="$_font" size="$_fontregular" color="$_errorcolor"><B>$Search_Error_Msg</B></font></div>
                <form method="POST" action="query.pl">
                <table align="center" border="0"><tr>
                <td><font face="$_font" size="$_fontregular" color="$_fontregular">$_str_names{'username'}: </font></td><td>
                <input type="hidden" name="guest" value="$guest">
                <input type="text" name="user"></td><td>
                <input type="submit" name="submit" value="Query">
                </td></tr></table></form>
                </td></tr></table>
                </td></tr></table>
                </td>
            </tr>
        </table>
    };

    print &_footer;


}




sub _layoutQueryUser() {

    # Parameters
    my $Username = shift;
    my $First_Name = shift;
    my $Last_Name = shift;
    my $Location = shift;
    my $Email = shift;
    my $Posts = shift;      # Number of posts user has made

    print "Content-type: text/html\n\n";
    print &_header("$_header",$_altbgcolor);

    print qq{
        <table border="0" cellpadding="15" cellspacing="0" width="100%" height="100%">
            <tr>
                <td>
                <table border="0" width="60%" cellpadding="0" cellspacing="2" bgcolor="$_tableborder" align="center">
                <tr><td>
                <table border="0" width="100%" height="100%" cellpadding="5"><tr><td bgcolor="$_tablebg">
                <font face="$_font" size="$_fonttitlesize" color="$_titlecolor"><b>$_str_names{'userquery'}</b></font><br>
                <font face="$_font" size="$_fontsmall" color="$_fontregular">$_str_text{'query_desc'}</font>
                </td></tr>
                <tr><td bgcolor="$_tablebg">
                <table align="center">
                    <tr>
                        <td align="right" width="50%"><B><font face="$_font" size="$_fontregular">$_str_names{'username'}:</font></b></td>
                        <td><font face="$_font" size="$_fontregular">$Username</font></td>
                    </tr>
                    <tr>
                        <td align="right"><B><font face="$_font" size="$_fontregular">First Name:</font></b></td>
                        <td><font face="$_font" size="$_fontregular">$First_Name</font></td>
                    </tr>
                    <tr>
                        <td align="right"><B><font face="$_font" size="$_fontregular">Last Name:</font></b></td>
                        <td><font face="$_font" size="$_fontregular">$Last_Name</font></td>
                    </tr>
                    <tr>
                        <td align="right"><B><font face="$_font" size="$_fontregular">Location:</font></b></td>
                        <td><font face="$_font" size="$_fontregular">$Location</font></td>
                    </tr>
                    <tr>
                        <td align="right"><B><font face="$_font" size="$_fontregular">E-mail:</font></b></td>
                        <td><font face="$_font" size="$_fontregular"><a href="mailto:$Email">$Email</a></font></td>
                    </tr>
                    <tr>
                        <td align="right"><B><font face="$_font" size="$_fontregular">$_str_names{'posts'}:</font></b></td>
                        <td><font face="$_font" size="$_fontregular">$Posts</font></td>
                    </tr>
                </table>
                </td></tr></table>
                </td></tr></table>
                </td>
            </tr>
        </table>
    };

    print &_footer;


}



sub _layoutStats() {

    my $Branch_Posts_Link = shift;
    my $Branch_Hits_Link = shift;
    my $Board_Posts_Link = shift;
    my $Board_Hits_Link = shift;
    my $User_Posts_Link = shift;
    my $User_Hits_Link = shift;

    print "Content-type: text/html\n\n";
    print &_header("$_header",$_altbgcolor);

    print qq{
        <table border="0" cellpadding="15" cellspacing="0" width="100%" height="100%">
            <tr>
                <td>
                <table border="0" width="90%" cellpadding="0" cellspacing="2" bgcolor="$_tableborder" align="center">
                <tr><td>
                <table border="0" width="100%" height="100%" cellpadding="5"><tr><td bgcolor="$_tablebg">
                <font face="$_font" size="$_fonttitlesize" color="$_titlecolor"><b>$_str_names{'statistics'}</b></font><br>
                <font face="$_font" size="$_fontsmall" color="$_fontregular">$_str_text{'statistics_desc'}</font>
                </td></tr>
                <tr><td bgcolor="$_tablebg">

                <table border="0" cellspacing="5" cellpadding="5" align="center" width="100%">
                    <tr>
                        <td width="33%" align="center"><font face="$_font" size="$_fontregular"><B><U>$_str_names{'branch'} Activity</U></B></font></td>
                        <td width="34%" align="center"><font face="$_font" size="$_fontregular"><B><U>$_str_names{'board'} Activity</U></B></font></td>
                        <td width="33%" align="center"><font face="$_font" size="$_fontregular"><B><U>$_str_names{'user'} Activity</U></B></font></td>
                    </tr>
                    <tr>
                        <td width="33%" align="center"><font face="$_font" size="$_fontregular">
                            <a href="$Branch_Posts_Link">Sort by $_str_names{'posts'}</a>
                            <br><a href="$Branch_Hits_Link">Sort by $_str_names{'hits'}</a>
                        </font></td>
                        <td width="34%" align="center"><font face="$_font" size="$_fontregular">
                            <a href="$Board_Posts_Link">Sort by $_str_names{'posts'}</a>
                            <br><a href="$Board_Hits_Link">Sort by $_str_names{'hits'}</a>
                        </font></td>
                        <td width="33%" align="center"><font face="$_font" size="$_fontregular">
                            <a href="$User_Posts_Link">Sort by $_str_names{'posts'}</a>
                            <br><a href="$User_Hits_Link">Sort by $_str_names{'hits'}</a>
                        </font>                
                 </td></tr></table></form>
                </td>
                    </tr>
                </table>
                </td>
            </tr>
        </table>
    };

    print &_footer;


}






sub _layoutStatsDetail() {

    # Parameters
    my $Navigation = shift;     # Navigate between statistic sections
    my $Section_Title = shift;  # Title of current section
    my $Column_Define = shift;  # Column title
    my $Column_Posts = shift;   # Posts title/link
    my $Column_Hits = shift;    # Hits title/link
    my $Entries = shift;        # All table entries
    my $Column_Define_2 = shift;# (optional) Column title for a second table
    my $Column_Posts_2 = shift; # (optional) Posts title/link for a second table
    my $Column_Hits_2 = shift;  # (optional) Hits title/link for a second table
    my $Entries_2 = shift;      # (optional) All table entries for a second table

    print "Content-type: text/html\n\n";
    print &_header("$_header",$_altbgcolor);

    print qq{
        <table border="0" cellpadding="15" cellspacing="0" width="100%" height="100%">
            <tr>
                <td>
                <table border="0" width="60%" cellpadding="0" cellspacing="2" bgcolor="$_tableborder" align="center">
                <tr><td>
                <table border="0" width="100%" height="100%" cellpadding="5"><tr><td bgcolor="$_tablebg">
                <font face="$_font" size="$_fonttitlesize" color="$_titlecolor"><b>$_str_names{'statistics'}</b></font><br>
                <font face="$_font" size="$_fontsmall" color="$_fontregular">$_str_text{'statistics_desc'}</font>
                            <p align="center"><b><font face="$_font" size="$_fontregular">$Navigation</font></b>
                </td></tr>
                <tr><td bgcolor="$_tablebg">
                <center>
                <P align="center">
                <B><U><font face="$_font" size="$_fontregular">$Section_Title</U></B></font><br>
                <table width="400" align="center">
                    <tr>
                        <td width="50%"><font face="$_font" size="$_fontregular"><B>$Column_Define</B></font></td>
                        <td width="25%"><font face="$_font" size="$_fontregular"><B>$Column_Posts</B></font></td>
                        <td width="25%"><font face="$_font" size="$_fontregular"><B>$Column_Hits</B></font></td>
                    </tr>
                    $Entries
                </table>
                <p><table width="400" align="center">
                    <tr>
                        <td width="50%"><font face="$_font" size="$_fontregular"><B>$Column_Define_2</B></font></td>
                        <td width="25%"><font face="$_font" size="$_fontregular"><B>$Column_Posts_2</B></font></td>
                        <td width="25%"><font face="$_font" size="$_fontregular"><B>$Column_Hits_2</B></font></td>
                    </tr>
                    $Entries_2
                </table>
                </td></tr></table>
                </td></tr></table>
                </td>
            </tr>
        </table>
    };

    print &_footer;


}




sub _layoutLogin() {

    # Parameters
    my $Form_Data = shift;      # Necessary form data, must be inside <form> tags
    my $Quicknav_Data = shift;  # Necessary for quicknav functionality

    print "Content-type: text/html\n\n";
    print &_header("$_header");

    print qq{
        <center>
        <br><br><br>
        <img src="$_imgpath/logintitle.gif" border="0">
        <br><br><br>
        <table border="0" cellpadding="0" cellspacing="2" bgcolor="$_tableborder">
            <tr>
                <td>
                <table border="0" cellpadding="3" cellspacing="2" width="100%">
                    <tr>
                        <td bgcolor="$_tablebg">
                        <center><form action="login.pl" method="POST">
                        $Form_Data
                        <font face="$_font" size="$_fontregular"><b>$str_names{'login'}</b><BR><i>$_str_text{'login_note'}</i></font><br><br>
                        <table cellpadding=2 cellspacing=0>
                            <tr>
                                <td><font face="$_font" size="$_fontregular"><b>$_str_names{'username'}:</td><td><input type="text" name="user" size="20"></td>
                            </tr>
                            <tr>
                                <td><font face="$_font" size="$_fontregular"><b>Password:</td><td><input type="password" name="pass" size="20"></td>
                            </tr>
                            <tr>
                                <td colspan="2"><input type="checkbox" name="remember" value="ON"><font face="$_font" size="$_fontregular">$_str_text{'remember'}</td>
                            </tr>
                            <tr>
                                <td colspan="2"><font face="$_font" size="$_fontregular"><a href="frameset.pl?guest=yes$Quicknav_Data">Login as a $_str_names{'guest'}</a> - <a href="register.pl">Create a $_str_names{'user'}</a></td>
                            </tr>
                        </table>
                        <br>
                        <input name="Submit" type="Submit" value="Login">
                        <br>
                        </center>
                        </form>
                        </td>
                    </tr>
                </table>
                </td>
            </tr>
        </table>
    };

    print &_footer;

}




sub _layoutPost() {

    # Parameters
    my $Board_Name = shift; # Title of board posting to
    my $Form_Data = shift;  # Necessary form data for submission, must be inside <form> tags
    my $Options = shift;    # Extra posting options
    my $Back_To_TOC = shift;# $_str_text{'backtoclink'} link
    my $Users_Name = shift; # User's first and last name (if enabled)

    print "Content-type: text/html\n\n";
    print &_header("$_header");

    print qq{
        <table border="0" cellpadding="0" cellspacing="0" width="100%">
            <tr>
                <td bgcolor="$_topbarcolor">
                <table cellpadding="2" cellspacing="0" width="100%" background="$_imgpath/topostbg.gif">
                <tr><td>
                <img src="$_imgpath/blank.gif" width="5" height="5" border="0"><br>
                <img src="$_imgpath/blank.gif" width="5" height="5" border="0">
                <img src="$_imgpath/blank.gif" width="5" height="20" border="0">
                <font face="$_font" size="$_fontheadersize" color="$_titlecolor"><b>
                $_str_names{'post_to'} $Board_Name<br>
                </b></font>
                </td></tr></table>
                </td>
            </tr>
            <tr>
                <td background="$_imgpath/line.gif"><img src="$_imgpath/blank.gif" width="20" height="10" border="0"></td>
            </tr>
        </table>

        <img src="$_imgpath/blank.gif" width="10" height="10" border="0">
        <font face="$_font" size="$_fontregular"><B>$Back_To_TOC</B></font><br>

        <table border="0" cellpadding="15" cellspacing="0" width="100%">
            <tr>
                <td>
                <table border="0" cellpadding="0" cellspacing="0" bgcolor="$_tableborder" width="550" align="center">
                    <tr>
                        <td>
                        <table border="0" cellpadding="3" cellspacing="2" width="100%">
                            <tr>
                                <td bgcolor="$_tablebg">
                                <form method="POST" action="post.pl">
                                <input type="hidden" name="action" value="post">
                                $Form_Data
                                    <table><tr><td>
                                    <font face="$_font" size="$_fontregular"><b>Subject: </b> </font>
                                    </td><td align="right">
                                    <input type="text" name="subject" size="30">
                                    </td></tr>
                                    <tr><td align="right">
                                    <font face="$_font" size="$_fontregular"><b>From: </b> </font>
                                    </td><td>
                                    <input type="text" name="writer" size="30" value="$Users_Name"><br>
                                    </td></tr>
                                    </table>
                                </td>
                            </tr>
                            <tr>
                                <td bgcolor="$_tablebg" align="center">
                                <textarea name="message" rows="$_textarearows" cols="$_textareacols"></textarea>
                                </td>
                            </tr>
                            <tr>
                                <td bgcolor="$_tablebg" align="center">
                                <font face="$_font" size="$_fontsmall">
                                $Options
                                </font>
                                <center>$guestinput<input type="submit" name="submit" value="Post Message"></center></td>
                            </tr>
                        </table>
                        </td>
                    </tr>
                </table>
                </td>
            </tr>
        </td>
    </table>
    </form>
   
    };

    print &_footer;

}




sub _layoutReply() {

    # Parameters
    my $Reply_To = shift;       # Title of article replying to
    my $Form_Data = shift;      # Necessary form data for submission, must be inside <form> tags
    my $Options = shift;        # Additional Posting Options
    my $Back_To_TOC = shift;    # $_str_text{'backtoclink'} link
    my $Back_To_Thread = shift; # Back to current thread link
    my $Users_Name = shift;     # User's first and last name (if enabled)

    print "Content-type: text/html\n\n";
    print &_header("$_header");

    print qq{
        <table border="0" cellpadding="0" cellspacing="0" width="100%">
            <tr>
                <td bgcolor="$_topbarcolor">
                <table cellpadding="2" cellspacing="0" width="100%" background="$_imgpath/topostbg.gif">
                <tr><td>
                <img src="$_imgpath/blank.gif" width="5" height="5" border="0"><br>
                <img src="$_imgpath/blank.gif" width="5" height="5" border="0">
                <img src="$_imgpath/blank.gif" width="5" height="20" border="0">
                <font face="$_font" size="$_fontheadersize" color="$_titlecolor"><b>
                $_str_names{'reply_to'} $Reply_To<br>
                </b></font>
                </td>
                </td></tr></table>
                </td>
            </tr>
            <tr>
                <td background="$_imgpath/line.gif"><img src="$_imgpath/blank.gif" width="20" height="10" border="0"></td>
            </tr>
        </table>

        <img src="$_imgpath/blank.gif" width="10" height="10" border="0">
        <font face="$_font" size="$_fontregular"><B>$Back_To_TOC $_linkbarseparator $Back_To_Thread</B></font><br>

        <table border="0" cellpadding="15" cellspacing="0" width="100%">
            <tr>
                <td>
                <br>
                <table border="0" cellpadding="0" cellspacing="0" bgcolor="$_tableborder" width="550" align="center">
                    <tr>
                        <td>
                        <table border="0" cellpadding="3" cellspacing="2" width="100%">
                            <tr>
                                <td bgcolor="$_tablebg">
                                <form method="POST" action="reply.pl">
                                <input type="hidden" name="action" value="post">
                                $Form_Data
                                $guestinput
                                <table>
                                <tr>
                                <td>
                                    <font face="$_font" size="$_fontregular"><b>From: </b> </font>
                                </td>
                                <td>
                                    <input type="text" name="writer" size="30" value="$Users_Name">
                                </td>
                                </tr>
                                </table>
                                </td>
                            </tr>
                            <tr>
                                <td bgcolor="$_tablebg" align="center">
                                <textarea name="message" rows="$_textarearows" cols="$_textareacols"></textarea>
                                </td>
                            </tr>
                            <tr>
                                <td bgcolor="$_tablebg" align="center">
                                <font face="$_font" size="$_fontsmall">
                                $Options
                                </font>
                                <center><input type="submit" name="submit" value="$_str_text{'post'}"> &nbsp; &nbsp; <input type="reset" name="clear" value="$_str_text{'clear'}"></center></td>
                            </tr>
                        </table>
                        </td>
                    </tr>
                </table>
                </td>
            </tr>
        </td>
    </table>
    </form>
   
    };

    print &_footer;

}







sub _layoutTOC() {

    # Parameters 
    my $Board_Name = shift;     # Name of the TOC
    my $Board_ID = shift;       # Board ID
    my $Function_List = shift;  # post, refresh, access list, homepage, etc
    my $Hits = shift;           # Number of hits to the board
    my $Article_List = shift;   # List of articles for the TOC
    my $Bottom_Nav = shift;     # Next/prev links on bottom

    print "Content-type: text/html\n\n";
    print &_header("$_header");

    print qq{
        <table border="0" cellpadding="0" cellspacing="0" width="100%">
            <tr><td bgcolor="$_topbarcolor">
                <table width="100%" cellpadding="0" cellspacing="0" background="$_imgpath/forumbg.gif"><tr><td>
                <img src="$_imgpath/blank.gif" width="5" height="5" border="0"><br>
                <img src="$_imgpath/blank.gif" width="5" height="5" border="0">
                <img src="$_imgpath/blank.gif" width="15" height="20" border="0">
                <font face="$_font" size="$_fonttitlesize" color="$_titlecolor"><b>
                $Board_Name
                </b></font>
                </td>
    };

    if ($_showids != 0) {
        print qq{
                <td valign="bottom" align="right">
                    <img src="$_imgpath/blank.gif" width="15" height="20" border="0">
                    <font face="$_font" size="$_fontsmall" color="$_idcolor">$_str_text{'board_id'} $Board_ID
                    <img src="$_imgpath/blank.gif" width="10" height="5" border="0"></font>
                </td>
        };
    }

    print qq{
                </tr></table>
                </td>
            </tr>
        </table>
        <table border="0" cellpadding="2" cellspacing="0" width="100%">
            <tr>
                <td bgcolor="$_linkbarcolor" background="$_imgpath/line.gif">
                    <table border="0" cellspacing="0" cellpadding="0" width="100%">
                        <tr>
                            <td align="left">
                                <font size="1" face="$_font">
                                <img src="$_imgpath/blank.gif" width="20" height="10" border="0">
                                $Function_List
                                </font>
                            </td>
    };

    if ($_hitcounters != 0) {
        print qq{
                            <td align="right" valign="bottom">
                                <font face="$_font" size="$_fontsmall">$Hits $_str_names{'hits'}
                                <img src="$_imgpath/blank.gif" width="10" height="10" border="0">
                                </font>
                            </td>
        };
    }

    print qq{
                        </tr>
                    </table>
                </td>
            </tr>
            <tr>
                <td background="$_imgpath/line.gif"><img src="$_imgpath/blank.gif" width="20" height="10" border="0"></td>
            </tr>
        </table>
        
        <table cellpadding="20" width="100%"><tr><td>
        <table width="100%" cellpadding="0" cellspacing="0"><tr><td bgcolor="$_tableborder">
            <table width="100%" cellpadding="2" cellspacing="1">
            <tr>
            <td bgcolor="$_alttablebg">
            <div align="center"><font face="$_font" size="$_fontsmall"><b>
            &nbsp;
            </b></font></div>
            </td>
            <td bgcolor="$_alttablebg">
            <div align="center"><font face="$_font" size="$_fontsmall"><b>
            Subject
            </b></font></div>
            </td>
            <td width="100" bgcolor="$_alttablebg">
            <div align="center"><font face="$_font" size="$_fontsmall"><b>
            Author
            </td>
            <td bgcolor="$_alttablebg" width="140">
            <div align="center"><font face="$_font" size="$_fontsmall"><b>
            Date &amp; Time
            </td>
            <td bgcolor="$_alttablebg" width="50">
            <div align="center"><font face="$_font" size="$_fontsmall"><b>
            Replies
            </td>
            </tr>
            
        $Article_List
            </table>
        </table>
            </td></tr></table>

        <br><br>
        <img src="$_imgpath/blank.gif" width="10" height="1" border="0">

        $Bottom_Nav
   
    };

    print &_footer;

}


sub _formatPostEntry {

    # Parameters
    my $Board_ID = shift;
    my $Article_ID = shift;
    my $gueststring = shift;    # Necessary for all links, leave alone
    my $Bold = shift;           # Link is bold if unread
    my $Subject = shift;
    my $End_Bold = shift;       # End's Bold tag if unread
    my $Writer = shift;
    my $Time = shift;
    my $Reply_Count = shift;    # Number of replies
    my $Replies = shift;        # Contains gramatically correct word "reply" or "replies"

    return qq{
            <tr>
                <td width="16" bgcolor="$_tablebg">
                    <img src="$_imgpath/msg.gif">
                </td>
                <td bgcolor="$_tablebg">
                    <font face="$_font" size="$_fontregular">
                    <a href="article.pl?action=viewarticle&board=$Board_ID&article=$Article_ID$gueststring">$Bold$Subject$End_Bold</a>
                    </font>
                </td>
                <td width="100" bgcolor="$_tablebg">
                    <div align="center"><font face="$_font" size="$_fontsmall">
                    $Writer
                    </font></div>
                </td>
                <td width="140" bgcolor="$_tablebg">
                    <div align="center">
                    <font face="$_font" size="$_fontsmall">
                    $Time
                    </font>
                    </div>
                </td>
                <td width="50" bgcolor="$_tablebg">
                    <div align="center">
                    <font face="$_font" size="$_fontsmall">
                    $Reply_Count
                    </font>
                    </div>
                </td>
            </tr>
    };


}

sub _formatPostPreviewEntry {

    # Parameters
    my $Board_ID = shift;
    my $Article_ID = shift;
    my $gueststring = shift;    # Necessary for all links, leave alone
    my $Bold = shift;           # Link is bold if unread
    my $Subject = shift;
    my $End_Bold = shift;       # End's Bold tag if unread
    my $Writer = shift;
    my $Time = shift;
    my $Reply_Count = shift;    # Number of replies
    my $Replies = shift;        # Contains gramatically correct word "reply" or "replies"
    my $Preview = shift;        # Contains Article Preview
    
    return qq{
            <tr>
                <td width="16" bgcolor="$_tablebg" rowspan="1">
                    <img src="$_imgpath/msg.gif">
                </td>
                <td bgcolor="$_tablebg">
                    <font face="$_font" size="$_fontregular">
                    <a href="article.pl?action=viewarticle&board=$Board_ID&article=$Article_ID$gueststring">$Bold$Subject$End_Bold</a>
                    </font><br>
                    <table cellpadding="1" cellspacing="0"><tr><td valign="top"><font face="$_font" size="$_fontsmall"></font></td><td valign="top">
                    <font face="$_font" size="$_fontsmall">
                    <i>$Preview</i></font>
                    </td></tr></table>
                    
                </td>
                <td width="100" bgcolor="$_tablebg" align="center">
                    <font face="$_font" size="$_fontsmall">
                    $Writer
                    </font>
                </td>
                <td width="140" bgcolor="$_tablebg" align="center">
                    <div>
                    <font face="$_font" size="$_fontsmall">
                    $Time
                    </font>
                    </div>
                </td>
                <td width="50" bgcolor="$_tablebg" align="center">
                    <font face="$_font" size="$_fontsmall">
                    $Reply_Count
                    </font>
                </td>
            </tr>
    };

}

sub _layoutArticle() {

    # Parameters
    my $Base_Post = shift;      # Hash reference to first post in the thread
    my $Replies = shift;        # Array reference to a hash of each individual reply
    my $Num_Of_Replies = shift; # Number of replies

    # Add link separator before value, to separate from TOC link in body
    if (($guest ne "yes")&&($flag ne "unread")) {
        $Back_To_TOC = qq{<a href="main.pl?action=viewboard&board=$board$gueststring">$_str_text{'backtoclink'}</a>};
        $Mark_As_Unread = qq{$_linkbarseparator <a href="article.pl?action=viewarticle&board=$board&article=$article&flag=unread">Mark as Unread</a>};
    } else {
        $Back_To_TOC = qq{<a href="main.pl?action=viewboard&board=$board$gueststring">$_str_text{'backtoclink'}</a>};
        $Mark_As_Unread = qq{};
    }

    print "Content-type: text/html\n\n";
    print &_header("$_header");

    print qq{
        <a name=\"top\"></a>
        <table border="0" cellpadding="0" cellspacing="0" width="100%">
            <tr>
                <td bgcolor="$_topbarcolor">
                <table width="100%" cellpadding="0" cellspacing="0"><tr><td background="$_imgpath/postbg.gif">
                <img src="$_imgpath/blank.gif" width="5" height="5" border="0"><br>
                <img src="$_imgpath/blank.gif" width="5" height="5" border="0">
                <img src="$_imgpath/blank.gif" width="5" height="20" border="0">
                <font face="$_font" size="$_fontheadersize" color="$_titlecolor"><b>
                $Base_Post->{Subject}<br>
                </b></font>
                </td></tr></table>
                </td>
            </tr>
            <tr>
                <td background="$_imgpath/line.gif"><img src="$_imgpath/blank.gif" width="20" height="10" border="0"></td>
            </tr>
        </table>

        <img src="$_imgpath/blank.gif" width="20" height="10" border="0">
        <font face="$_font" size="$_fontregular"><B>$Back_To_TOC$Mark_As_Unread</B></font>

        <BR><BR>

        <font face="$_font" size="$_fontregular"><img src="$_imgpath/blank.gif" width="20" height="10" border="0">
        <B><a href="#base">$Base_Post->{Subject}</a></B></font>
        <font face="$_font" size="$_fontsmall"> by $Base_Post->{Writer}<BR></font>
    };

    # For each reply, display a preview
    for ($i = 0; $i < $Num_Of_Replies; $i++) {
        print qq{
            <font face="$_font" size="$_fontregular"><img src="$_imgpath/blank.gif" width="40" height="10" border="0">
            <a href="#$Replies->[$i]->{PostCount}">$Replies->[$i]->{IndexMessage}</a></font>
            <font face="$_font" size="$_fontsmall"> by $Replies->[$i]->{Writer}<BR></font>
        };
    }

    if (($guest ne "yes")&&($_viewremotename != 0)) {
        $Base_Post->{RemoteName} = "<b>Remote Name:</b> $Base_Post->{Remote_Name}<BR>";
    } else {
        $Base_Post->{RemoteName} = "";
    }
    
    print qq{
        <br><a name="base"></a><table border="0" cellpadding="15" cellspacing="0" width="100%">
        <tr>
        <td>
            <table border="0" cellpadding="0" cellspacing="1" width="100%" bgcolor="$_tableborder">
            <tr>
            <td>
                <table border="0" cellpadding="3" cellspacing="1" width="100%">
                    <tr>
                        <td bgcolor="$_alttablebg">
                            <table border="0" cellspacing="0" cellpadding="0" width="100%">
                            <tr>
                            <td>
                                <font face="$_font" size="$_fontregular"><b>Base Post</b></font><br>
                            </td>
    };

    if ($_showids != 0) {
        print qq{
                            <td bgcolor="$_alttablebg" align="right">
                                <font face="$_font" size="$_fontsmall" color="$_idcolor">$_str_text{'article_id'} $Base_Post->{BPostsID}
                                </font>
                            </td>
        };
    }

    print qq{
                            </tr>
                            </table>
                        </td>
                    </tr>
                    <tr>
                        <td bgcolor="$_tablebg">
                        <font face="$_font" size="$_fontsmall"><b>Posted By:</b> $Base_Post->{Writer} (<a href="query.pl?user=$Base_Post->{Username}$gueststring">$Base_Post->{Username}</a>)
                        <br>
                        <b>Time:</b> $Base_Post->{Time}<br>$Base_Post->{RemoteName}</font>
                        </td>
                    </tr>
                    <tr>
                        <td bgcolor="$_tablebg">
                        <font face="$_font" size="$_fontregular">
                        $Base_Post->{Message}
                        </font>
                        <br><br>
                        <font face="$_font" size="$_fontsmall"><a href="reply.pl?board=$board&article=$article&inreplyto=0$gueststring"><img src="$_imgpath/reply.gif" border="0" align="absmiddle">reply to this thread</a>
                        <img src="$_imgpath/blank.gif" width="20" height="10" border="0"><a href="#top"><img src="$_imgpath/topicon.gif" border="0" align="absmiddle">go to top</a></font>
                        </td>
                    </tr>
                </table>
            </tr>
            </td>
            </table>
        </td>
        </tr>
        </table>
    };


    # For each reply, display entire reply
    for (my $i = 0; $i < $Num_Of_Replies; $i++) {
        # Whether or not to display remote name
        if (($guest ne "yes")&&($_viewremotename != 0)) {
            $Replies->[$i]->{RemoteName} = "<BR><b>Remote Name:</b> $Replies->[$i]->{Remote_Name}";
        } else {
            $Replies->[$i]->{RemoteName} = "";
        }
        print qq{
            <a name="$Replies->[$i]->{PostCount}"></a><table border="0" cellpadding="15" cellspacing="0" width="100%">
            <tr>
            <td>
                <table border="0" cellpadding="0" cellspacing="1" width="100%" bgcolor="$_tableborder">
                <tr>
                <td>
                    <table border="0" cellpadding="3" cellspacing="1" width="100%">
                        <tr>
                            <td bgcolor="$_alttablebg">
                                <table border="0" cellspacing="0" cellpadding="0" width="100%">
                                <tr><td>
                                    <font face="$_font" size="$_fontregular"><b>Reply $Replies->[$i]->{PostCount} of $Num_Of_Replies</b></font>
                                </td>
        };

        if ($_showids != 0) {
            print qq{
                                <td bgcolor="$_alttablebg" align="right">
                                    <font face="$_font" size="$_fontsmall" color="$_idcolor">$_str_text{'reply_id'} $Replies->[$i]->{BRepliesID}
                                    </font>
                                </td>
            };
        }

        print qq{
                                </tr>
                                </table>
                            </td>
                        </tr>
                        <tr>
                            <td bgcolor="$_tablebg">
                            <font face="$_font" size="$_fontsmall"><b>Posted By:</b> $Replies->[$i]->{Writer} (<a href="query.pl?user=$Replies->[$i]->{Username}$gueststring">$Replies->[$i]->{Username}</a>)<BR><b>Time:</b> $Replies->[$i]->{Time}$Replies->[$i]->{RemoteName}<br>
                                </font>
                            </td>
                        </tr>   
                        <tr>
                            <td bgcolor="$_tablebg">
                                <font face="$_font" size="$_fontregular">
                                $Replies->[$i]->{Message}
                                </font>
                                <br><br>
                                <font face="$_font" size="$_fontsmall"><a href="reply.pl?board=$board&article=$article&inreplyto=$Replies->[$i]->{PostCount}$gueststring"><img src="$_imgpath/reply.gif" border="0" align="absmiddle">reply to this thread</a>
                                <img src="$_imgpath/blank.gif" width="20" height="10" border="0"><a href="#top"><img src="$_imgpath/topicon.gif" border="0" align="absmiddle">go to top</a></font>
                            </td>
                         </tr>
                     </table>
                </td>
                </tr>
                </table>
            </td>
            </tr>
            </table>
        };
    }

    print &_footer;

}





sub _layoutAccess() {

    # Parameters
    my $Board_Name = shift;
    my $Function_List = shift;         # Post, Refresh, etc
    my $Access_List = shift;           # List of users that have access to the BB
    my $Add_Delete_Forms = shift;      # Only applies if the user is an administrator

    print "Content-type: text/html\n\n";
    print &_header("$_header");

    print qq{
        <table border="0" cellpadding="0" cellspacing="0" width="100%">
            <tr>
                <td bgcolor="$_topbarcolor">
                <table width="100%" cellpadding="0" cellspacing="0"><tr><td background="$_imgpath/forumbg.gif">
                <img src="$_imgpath/blank.gif" width="5" height="5" border="0"><br>
                <img src="$_imgpath/blank.gif" width="5" height="5" border="0">
                <img src="$_imgpath/blank.gif" width="15" height="20" border="0">
                <font face="$_font" size="$_fonttitlesize" color="$_titlecolor"><b>
                $Board_Name
                </b></font>
                </td></tr></table>
                </td>
            </tr>
            </table>
            <table border="0" cellpadding="2" cellspacing="0" width="100%">
            <tr>
                <td bgcolor="$_linkbarcolor" background="$_imgpath/line.gif">
                    <table border="0" cellspacing="0" cellpadding="0" width="100%">
                        <tr>
                            <td align="left"><font size="1" face="$_font">
                            <img src="$_imgpath/blank.gif" width="20" height="10" border="0">
                            $Function_List
                            </font>
                            </td>
                        </tr>
                    </table>
                </td>
            </tr>
            <tr>
                <td background="$_imgpath/line.gif"><img src="$_imgpath/blank.gif" width="20" height="10" border="0"></td>
            </tr>
        </table>

        <font face="$_font" size="$_fontregular"><B>$_str_names{'access_list'}:</B><BR><BR>
        $Access_List
        </font>

        <br><br>

        $Add_Delete_Forms
    };

    print &_footer;

}




sub _layoutRegister() {

    # Parameters
    my $Error_Message = shift;
    my $First_Name = shift;
    my $Last_Name = shift;
    my $Username = shift;

    print "Content-type: text/html\n\n";
    print &_header("$_header");

    print qq{
        <center><br><br><br>
        <img src="$_imgpath/logintitle.gif" border="0"><br><br><br>
        <table border="0" cellpadding="0" cellspacing="2" width="500" bgcolor="$_tableborder">
            <tr>
                <td>
                    <table border="0" cellpadding="3" cellspacing="2" width="100%">
                        <tr>
                            <td bgcolor="$_tablebg">
                            <center><br>
                            <form method="POST" action="register.pl">
                            <input type="hidden" name="action" value="create">
                            <font face="$_font" size="$_fontregular"><b>$Error_Message<BR><BR>$_str_names{'account_create'}</b></font><br><br>
                            <table cellpadding="2" cellspacing="0">
                            <tr><td><font face="$_font" size="$_fontregular"><b>First Name:</b></font></td><td><input type="text" name="firstname" size="20" value="$First_Name"></td></tr>
                            <tr><td><font face="$_font" size="$_fontregular"><b>Last Name:</b></font></td><td><input type="text" name="lastname" size="20" value="$Last_Name"></td></tr>
                            <tr><td><font face="$_font" size="$_fontregular"><b>$_str_names{'username'}:</b></font></td><td><input type="text" name="username" size="20" value="$Username"></td></tr>
                            <tr><td><font face="$_font" size="$_fontregular"><b>Password:</b></font></td><td><input type="password" name="password" size="10"></td></tr>
                            <tr><td><font face="$_font" size="$_fontregular"><b>Confirm Password:</b></font></td><td><input type="password" name="confirmpass" size="10"></td></tr>
                            </table>
                            <br>
                            <input name="Submit" type="Submit" value="Create"></form>
                            <br>
                            </center>
                            </td>
                        </tr>
                    </table>
                </td>
            </tr>
        </table>
    };

    print &_footer;

}




sub _layoutRegisterConfirm() {

    # Parameters
    my $Username = shift;

    print "Content-type: text/html\n\n";
    print &_header("$_header");

    print qq{
        <center><br><br><br>
        <img src="$_imgpath/logintitle.gif" border="0"><br><br><br>
        <table border="0" cellpadding="0" cellspacing="2" width="500" bgcolor="$_tableborder">
            <tr>
                <td>
                    <table border="0" cellpadding="3" cellspacing="2" width="100%">
                        <tr>
                            <td bgcolor="$_tablebg">
                            <center><br><font face="$_font" size="$_fontregular"><B>$_str_names{'account_created'}</B>
                            <BR><BR>$_str_names{'username'}: $Username
                            <BR><BR><a href="login.pl">$_str_text{'login'}</a><br></font></center>
                            </td>
                        </tr>
                    </table>
                </td>
            </tr>
        </table>
    };

    print &_footer;

}




sub _layoutLoginError() {

    print "Content-type: text/html\n\n";
    print &_header("$_header");

    print qq{
        <center><br><br><br>
        <img src="$_imgpath/logintitle.gif" border="0"><br><br><br>
        <table border="0" cellpadding="0" cellspacing="2" width="500" bgcolor="$_tableborder">
            <tr>
                <td>
                    <table border="0" cellpadding="3" cellspacing="2" width="100%">
                        <tr>
                            <td bgcolor="$_tablebg">
                            <center><br>
                            <form action="login.pl" method="POST">
                            <input type="hidden" name="action" value="login">
                            <font face="$_font" size="$_fontregular"><b>$_str_names{'login'}</b></font><br><br><font face="$_font" size="$_fontregular">$_str_errors{'login'}</font><br>
                            <br>
                            <table cellpadding="2" cellspacing="0">
                                <tr><td><font face="$_font" size="$_fontregular"><b>$_str_names{'username'}:</b></font></td><td><input type="text" name="user" size="20"></td></tr>
                                <tr><td><font face="$_font" size="$_fontregular"><b>Password:</b></font></td><td><input type="password" name="pass" size="20"></td></tr>
                                <tr><td colspan="2"><input type="checkbox" name="remember" value="ON"><font face="$_font" size="$_fontregular"> Remember my ID on this computer</font></td></tr>
                                <tr><td colspan="2"><font face="$_font" size="$_fontregular"><a href="frameset.pl?guest=yes">Login as a $_str_names{'guest'}</a> - <a href="register.pl">Create a $_str_names{'user'}</a></font></td></tr>
                            </table>
                            <br>
                            <input name="Submit" type="Submit" value="Login"></form>
                            <br>
                            </center>
                            </td>
                        </tr>
                    </table>
                </td>
            </tr>
        </table>
    };

    print &_footer;

}


sub _layoutSearch() {

    # Parameters
    my $Form_Data = shift;
    my $Board_List = shift;

    print "Content-type: text/html\n\n";
    print &_header("$_header",$_altbgcolor);

    print qq{
        <table width="100%" height="90%">
            <tr><td>

            <form method="POST" action="search.pl">
            $Form_Data
            
            <table border="0" width="60%" cellpadding="0" cellspacing="2" bgcolor="$_tableborder" align="center">
            <tr><td>
            <table border="0" width="100%" height="100%" cellpadding="5"><tr><td bgcolor="$_tablebg">

            <font face="$_font" size="$_fonttitlesize" color="$_titlecolor"><b>$_str_names{'search'}</b></font><br>
            <font face="$_font" size="$_fontsmall" color="$_fontregular">$_str_text{'search_desc'}</font>
            
            </td></tr>
        
            <tr><td bgcolor="$_tablebg" align="left">
                <table align="center">
                    <tr><td align="left">
                    <font face="$_font" size="$_fontregular">Search in: </font><select name="board">$Board_List</select>
                    </td></tr><tr><td>
                    <font face="$_font" size="$_fontregular">Search for: </font><input type="text" name="keywords" size="40">
                    <input type="submit" name="submit" value="Search">
                    </td></tr><tr><td align="left">
                    <font face="$_font" size="$_fontregular">
                    Results per page:</font>
                    <input type="text" name="numrecs" value="10" size="3" maxlength="2">
                    </td></tr><tr><td align="center">
                    </td></tr>
                </table>
            </form>
            </td></tr><tr><td bgcolor="$_tablebg">

            <font face="$_font" size="$_fontregular"><b>Search Examples...</b></font>
            <dl compact>
                <dt><code>apple banana</code>
                <dd><font face="$_font" size="$_fontsmall">find rows that contain at least one of these words.</font></dd>
                <dt><code>+apple +juice</code>
                <dd><font face="$_font" size="$_fontsmall">... both words.</font></dd>
                <dt><code>+apple macintosh</code>
                <dd><font face="$_font" size="$_fontsmall">... word "apple," but rank it higher if it also contain "macintosh."</font></dd>
                <dt><code>+apple -macintosh</code>
                <dd><font face="$_font" size="$_fontsmall">... word "apple" but not "macintosh".</font></dd>
                <dt><code>+apple +(&gt;pie &lt;strudel)</code>
                <dd><font face="$_font" size="$_fontsmall">... "apple" and "pie", or "apple" and "strudel" (in any order),
                but rank "apple pie" higher than "apple strudel."</font></dd>
                <dt><code>apple*</code>
                <dd><font face="$_font" size="$_fontsmall">... "apple," "apples," "applesauce," and "applet."</font></dd>
                <dt><code>&quot;some words&quot;</code>
                <dd><font face="$_font" size="$_fontsmall">... "some words of wisdom," but not "some other words."</font></dd>
            </dl>

           </td></tr></table></td></tr></table>
        </td></tr>
        </table>

    };

    print &_footer;

}

sub _layoutSearchResults() {

    # Results is a reference to an array
    # $Results->[$element]->{Field}
    # Valid Fields: Message, Time (Epoch), BRepliesID (doesn't really need displayed),
    #               BPostsID, Writer ("From" field), Username, Subject, BUsersID
    #               BBoardsID, PostCount (Used after pound sign in URL as anchor to actual reply)

    # Parameters
    my $Results = shift;             # Reference to search results array
    my $Num_Results = shift;         # Number of results search returned
    my $Start_At_Rec = shift;        # Start at record #
    my $Show_Num_Records = shift;    # Show # of records per page
    my $New_Search_Form_Data = shift;# Must go between <form> tags
    my $Next_Form_Data = shift;      # Must go between <form> tags
    my $Keywords = shift;            # Keywords
    my $Board_List = shift;          # List of BBs to Search

    $Start_At_Rec++;
    $End_At_Rec = $Start_At_Rec + $Show_Num_Records - 1;
    if ($End_At_Rec > $Num_Results) {$End_At_Rec = $Num_Results;}

    print "Content-type: text/html\n\n";
    print &_header("$_header",$_altbgcolor);

    print qq{
        <table width="100%" height="90%">
            <tr><td>
            
            <table border="0" width="60%" cellpadding="0" cellspacing="2" bgcolor="$_tableborder" align="center">
            <tr><td>
            <table border="0" width="100%" height="100%" cellpadding="5"><tr><td bgcolor="$_tablebg">

            <font face="$_font" size="$_fonttitlesize" color="$_titlecolor"><b>$_str_names{'search'}</b></font><br>
            <font face="$_font" size="$_fontsmall" color="$_fontregular">$_str_text{'search_desc'}</font>
            
            </td></tr>
            
            <tr><td bgcolor="$_tablebg" align="left">
            <font face="$_font" size="$_fontregular">
            Your query returned $Num_Results results. Displaying results $Start_At_Rec through $End_At_Rec.
            </font>
    };

    my $num = 1;
    for (my $i=$Start_At_Rec-1; $i < $End_At_Rec; $i++) {
        # use $num to number them
        print qq{
            <p><font face="$_font" size="$_fontregular">
            <b><a href="article.pl?action=viewarticle&board=$Results->[$i]->{BBoardsID}&article=$Results->[$i]->{BPostsID}$gueststring#$Results->[$i]->{PostCount}">
            $Results->[$i]->{Subject}</a></b><br></font><font face="$_font" size="$_fontsmall"> by $Results->[$i]->{Writer} (<a href="query.pl?user=$Results->[$i]->{Username}">$Results->[$i]->{Username}</a>) $Results->[$i]->{Time}</font>
        };
        $num++;
    }

    # If not all records were displayed, show next page button
    if ($End_At_Rec < $Num_Results) {
        print qq{
            <center>
            <form method="POST" action="search.pl">
            $Next_Form_Data
            <input type="submit" name="submit" value="Next $Show_Num_Records">
            </form></center>
        };
    } else {
        print qq{
            <br><br>
        };
    }

    print qq{
        <tr><td bgcolor="$_tablebg" align="left">
        <form method="POST" action="search.pl">
        $New_Search_Form_Data
            <table align="center">
                    <tr><td align="left">
                    <font face="$_font" size="$_fontregular">Search in: </font><select name="board">$Board_List</select>
                    </td></tr><tr><td>
                    <font face="$_font" size="$_fontregular">Search for: </font><input type="text" name="keywords" size="30" value="$Keywords">
                    <input type="submit" name="submit" value="Search">
                    </td></tr><tr><td align="left">
                    <font face="$_font" size="$_fontregular">
                    Results per page:</font>
                    <input type="text" name="numrecs" value="$Show_Num_Records" size="3" maxlength="2">
                    </td></tr><tr><td align="center">
                    </td></tr>
                </table>
            </form>
    };



    print qq {
        </td></tr></table></td></tr></table>
        </td></tr>
        </table>
    };
    print &_footer;

}


sub _formatFuzzyDateTime () {
    
    my $Sec = shift;
    my $Min = shift;
    my $Hour = shift;
    my $TOD = shift;    # This will still display the correct AM/PM even if you're in 24-hour mode
    my $Timezone = shift;
    my $FuzzyDate = shift;


    return 
        qq{
            $FuzzyDate &nbsp;$Hour:$Min$TOD $Timezone
        };

}

sub _formatDateTime () {
    
    my $Sec = shift;
    my $Min = shift;
    my $Hour = shift;
    my $TOD = shift;    # This will still display the correct AM/PM even if you're in 24-hour mode
    my $Day = shift;
    my $Month = shift;
    my $Year = shift;
    my $Timezone = shift;

    return 
        qq{
            $Month.$Day.$Year &nbsp;$Hour:$Min$TOD $Timezone
        };

}

1;