-- This file should created all the tables you need. 
-- You'll also need to run zip-inserts.sql if you want geo-spatial searching

create table cas_category (
	category_id					serial PRIMARY KEY,
	name				varchar(64),
	description			text,
	meta_description	text,
	meta_keywords		text,
	header				text,
	footer				text,
	sort_key			varchar(255) unique,
	parent_id			int,
	-- a welcome message template to send to the email address of newly added entries in the directory
	insert_msg			text,
	-- template to let people know their entry has been updated. 
	update_msg			text
);

create table cas_item (
	item_id		    		serial PRIMARY KEY,
	url					varchar(300),
	title				varchar(100) NOT NULL,
    description    		text,
    points          	int4,						-- a rating, from 0 to 5 (0 is inactive)
	--
	-- Further Contact info, if available. 
	--
	first_names			varchar(32),
	last_name			varchar(32),
	org_name			varchar(200),
    add1				varchar(32),
    add2				varchar(32),
    city				varchar(32),  
    state				varchar(2),
    country				varchar(64),
    postal_code			varchar(16),
	phone				varchar(32),
	fax					varchar(32),
	email				varchar(100),
	-- email address of the person submitting the entry, so we can send them a welcome message
	submitor_email		varchar(100),
	--  date it was suggested by a user
	insert_date			date DEFAULT CURRENT_DATE,
	approval_state			varchar(20) check(approval_state in ('needs_approval','approved','deleted')) default 'needs_approval',
	-- editor to approve the last insert or update. 
    approved_date           date,
    approving_user          integer,
    approving_note           varchar(4000),

    email_verified_date      date,
    url_verified_date		 date,

    -- used if the item rejected before it reachs
    -- the approved state
    rejected_date        date,
    rejecting_user       integer,
    rejecting_note       varchar(4000),
    -- item was active but is now deleted from the system 
    -- (but can sell be revived)
    deleted_date        date,    
    deleting_user       integer,
    deleting_note       varchar(4000)
);

-- houses suggested updates to the item table that need editor or admin approval 
create table cas_item_suggested_updates (
	item_update_id		serial,
	date_added			date default CURRENT_DATE,
	item_id					int4 not null, -- the id of the item we are suggesting the update for.
	category_id			int4,
	url					varchar(300),
	title				varchar(100) NOT NULL,
    description    		text,
    points          	int4,						-- a rating, from 0 to 5 (0 is inactive)
	--
	-- Further Contact info, if available. 
	--
	first_names	varchar(32),
	last_name	varchar(32),
	org_name	varchar(200),
    add1		varchar(32),
    add2		varchar(32),
    city		varchar(32),  
    state		varchar(2),
    country		varchar(64),
    postal_code	varchar(16),
	phone		varchar(32),
	fax			varchar(32),
	email		varchar(64),
	-- email address of person suggesting the update, so we can let them know it's been done. 
	submitor_email		varchar(100)
);

-- Postgres 6.5 complains about these: "rule plan string too big."
create view cas_item_approved as SELECT * from cas_item WHERE approval_state = 'approved';
create view cas_item_needs_approval as 
	SELECT * from cas_item 
	WHERE approval_state = 'needs_approval';

create table cas_category_item_map (
    category_id        	int NOT NULL,
    item_id             int NOT NULL,
    unique(category_id,item_id)
); 

insert into cas_category (category_id,name,description,sort_key) values
	(0,'Top','This is the top node in the hierarchy. The public should never see it.','aa');

-- This is used for geo-spatial searching
CREATE TABLE cas_zipcodes (
	zipcode 		varchar(5) NOT NULL unique,
	state_code 		char(2),
	
	 -- strategically the field size in the zipcode file I use. -mls
	city 			varchar(28),
	lon_lat 		point
);


create table cas_country_codes (
	code		char(2) unique,
	name		varchar(64)
);
copy "cas_country_codes" from stdin;
ad	Andorra
ae	United Arab Emirates
af	Afghanistan
ag	Antigua and Barbuda
ai	Anguilla
al	Albania
am	Armenia
an	Netherlands Antilles
ao	Angola
aq	Antarctica
ar	Argentina
as	American Samoa
at	Austria
au	Australia
aw	Aruba
az	Azerbaijan
ba	Bosnia and Herzegovina
bb	Barbados
bd	Bangladesh
be	Belgium
bf	Burkina Faso
bg	Bulgaria
bh	Bahrain
bi	Burundi
bj	Benin
bm	Bermuda
bn	Brunei Darussalam
bo	Bolivia
br	Brazil
bs	Bahamas
bt	Bhutan
bv	Bouvet Island
bw	Botswana
by	Belarus
bz	Belize
ca	Canada
cc	Cocos (Keeling) Islands
cf	Central African Republic
cg	Congo
ch	Switzerland
ck	Cook Islands
cl	Chile
cm	Cameroon
cn	China
co	Colombia
cr	Costa Rica
cs	Czechoslovakia (former)
cu	Cuba
cv	Cape Verde
cx	Christmas Island
cy	Cyprus
cz	Czech Republic
de	Germany
dj	Djibouti
dk	Denmark
dm	Dominica
do	Dominican Republic
dz	Algeria
ec	Ecuador
ee	Estonia
eg	Egypt
eh	Western Sahara
er	Eritrea
es	Spain
et	Ethiopia
fi	Finland
fj	Fiji
fk	Falkland Islands (Malvinas)
fm	Micronesia
fo	Faroe Islands
fr	France
fx	France, Metropolitan
ga	Gabon
gb	Great Britain (UK)
gd	Grenada
ge	Georgia
gf	French Guiana
gh	Ghana
gi	Gibraltar
gl	Greenland
gm	Gambia
gn	Guinea
gp	Guadeloupe
gq	Equatorial Guinea
gr	Greece
gs	S. Georgia and S. Sandwich Isls.
gt	Guatemala
gu	Guam
gw	Guinea-Bissau
gy	Guyana
hk	Hong Kong
hm	Heard and McDonald Islands
hn	Honduras
hr	Croatia (Hrvatska)
ht	Haiti
hu	Hungary
id	Indonesia
ie	Ireland
il	Israel
in	India
io	British Indian Ocean Territory
iq	Iraq
ir	Iran
is	Iceland
it	Italy
jm	Jamaica
jo	Jordan
jp	Japan
ke	Kenya
kg	Kyrgyzstan
kh	Cambodia
ki	Kiribati
km	Comoros
kn	Saint Kitts and Nevis
kp	Korea (North)
kr	Korea (South)
kw	Kuwait
ky	Cayman Islands
kz	Kazakhstan
la	Laos
lb	Lebanon
lc	Saint Lucia
li	Liechtenstein
lk	Sri Lanka
lr	Liberia
ls	Lesotho
lt	Lithuania
lu	Luxembourg
lv	Latvia
ly	Libya
ma	Morocco
mc	Monaco
md	Moldova
mg	Madagascar
mh	Marshall Islands
mk	Macedonia
ml	Mali
mm	Myanmar
mn	Mongolia
mo	Macau
mp	Northern Mariana Islands
mq	Martinique
mr	Mauritania
ms	Montserrat
mt	Malta
mu	Mauritius
mv	Maldives
mw	Malawi
mx	Mexico
my	Malaysia
mz	Mozambique
na	Namibia
nc	New Caledonia
ne	Niger
nf	Norfolk Island
ng	Nigeria
ni	Nicaragua
nl	Netherlands
no	Norway
np	Nepal
nr	Nauru
nt	Neutral Zone
nu	Niue
nz	New Zealand (Aotearoa)
om	Oman
pa	Panama
pe	Peru
pf	French Polynesia
pg	Papua New Guinea
ph	Philippines
pk	Pakistan
pl	Poland
pm	St. Pierre and Miquelon
pn	Pitcairn
pr	Puerto Rico
pt	Portugal
pw	Palau
py	Paraguay
qa	Qatar
re	Reunion
ro	Romania
ru	Russian Federation
rw	Rwanda
sa	Saudi Arabia
sb	Solomon Islands
sc	Seychelles
sd	Sudan
se	Sweden
sg	Singapore
sh	St. Helena
si	Slovenia
sj	Svalbard and Jan Mayen Islands
sk	Slovak Republic
sl	Sierra Leone
sm	San Marino
sn	Senegal
so	Somalia
sr	Suriname
st	Sao Tome and Principe
su	USSR (former)
sv	El Salvador
sy	Syria
sz	Swaziland
tc	Turks and Caicos Islands
td	Chad
tf	French Southern Territories
tg	Togo
th	Thailand
tj	Tajikistan
tk	Tokelau
tm	Turkmenistan
tn	Tunisia
to	Tonga
tp	East Timor
tr	Turkey
tt	Trinidad and Tobago
tv	Tuvalu
tw	Taiwan
tz	Tanzania
ua	Ukraine
ug	Uganda
uk	United Kingdom
um	US Minor Outlying Islands
us	United States
uy	Uruguay
uz	Uzbekistan
va	Vatican City State (Holy See)
vc	Saint Vincent and the Grenadines
ve	Venezuela
vg	Virgin Islands (British)
vi	Virgin Islands (U.S.)
vn	Viet Nam
vu	Vanuatu
wf	Wallis and Futuna Islands
ws	Samoa
ye	Yemen
yt	Mayotte
yu	Yugoslavia
za	South Africa
zm	Zambia
zr	Zaire
zw	Zimbabwe
\.
update cas_country_codes set code  = upper(code); 

create table cas_state_codes (
	code		char(2) unique,
	name		varchar(64)
);
copy "cas_state_codes" from stdin;
al	Alabama
ak	Alaska
as	American Samoa
az	Arizona
ar	Arkansas
ca	California
co	Colorado
ct	Connecticut
de	Delaware
dc	District Of Columbia
fm	Federated States Of Micronesia
fl	Florida
ga	Georgia
gu	Guam
hi	Hawaii
id	Idaho
il	Illinois
in	Indiana
ia	Iowa
ks	Kansas
ky	Kentucky
la	Louisiana
me	Maine
mh	Marshall Islands
md	Maryland
ma	Massachusetts
mi	Michigan
mn	Minnesota
ms	Mississippi
mo	Missouri
mt	Montana
ne	Nebraska
nv	Nevada
nh	New Hampshire
nj	New Jersey
nm	New Mexico
ny	New York
nc	North Carolina
nd	North Dakota
mp	Northern Mariana Islands
oh	Ohio
ok	Oklahoma
or	Oregon
pw	Palau
pa	Pennsylvania
pr	Puerto Rico
ri	Rhode Island
sc	South Carolina
sd	South Dakota
tn	Tennessee
tx	Texas
ut	Utah
vt	Vermont
vi	Virgin Islands
va	Virginia
wa	Washington
wv	West Virginia
wi	Wisconsin
wy	Wyoming
aa	Armed Forces Americas
ae	Armed Forces
ap	Armed Forces Pacific
\.
update cas_state_codes set code  = upper(code);

-- to allow links to other categories and items in the directory
create table cas_link (
	link_id				serial primary key,
	from_cat_id 		int4 NOT NULL,
	to_cat_id 			int4,
	to_item_id  		int4,
	name 				varchar(64),
	active  			bool NOT NULL default 't'
); 

-- allows links to related categories
create table cas_related_category (
	from_cat_id			int4 NOT NULL,
	to_cat_id			int4 NOT NULL,
	unique(to_cat_id,from_cat_id)
);   


create table cas_users (
    user_id            serial primary key,
    first_names        varchar(100) not null,
    last_name        varchar(100) not null,
    screen_name        varchar(100),
    constraint cas_users_screen_name_unique unique(screen_name),
    email            varchar(100) not null unique,
    email_bouncing_p    char(1) default 'f' check(email_bouncing_p in ('t','f')),
    password        varchar(30) not null,
    -- set when user reappears at site
    last_visit        date,
    -- this is what most pages query against (since the above column
    -- will only be a few minutes old for most pages in a session)
    second_to_last_visit    date,
    -- how many times this person has visited
    n_sessions        integer default 1,
    registration_date    date DEFAULT CURRENT_DATE,
    registration_ip      inet,
    -- state the user is in in the registration process
    user_state        varchar(100) check(user_state in ('need_email_verification_and_admin_approv', 'need_admin_approv', 'need_email_verification', 'rejected', 'authorized', 'banned', 'deleted')),
    -- admin approval system
    approved_date           date,
    approving_user          integer,
    approving_note           varchar(4000),
    -- email verification system
    email_verified_date      date,
    -- used if the user rejected before they reach 
    -- the authorized state
    rejected_date        date,
    rejecting_user        integer,
    rejecting_note          varchar(4000),
    -- user was active but is now deleted from the system 
    -- may be revived
    deleted_date        date,    
    deleting_user       integer,
    deleting_note          varchar(4000),
    -- user was active and now not allowed into the system
    banned_date        date,
    -- who and why this person was banned
    banning_user        integer,
    banning_note        varchar(4000),
    bio            varchar(4000),
    role		   varchar(100) check(role in ('editor','admin','user')) default 'user'
);

--- users who are not deleted or banned

create view cas_users_active
as
select * 
 from cas_users 
 where user_state = 'authorized';
 
-- users who've signed up in the last 30 days
-- useful for moderators since new users tend to 
-- be the ones who cause trouble
create view cas_users_new
as
select * 
 from cas_users 
 where registration_date > (CURRENT_DATE - 30);
 
create table cas_users_preferences (
    user_id            integer primary key,
    prefer_text_only_p    char(1) default 'f' check (prefer_text_only_p in ('t','f')),
    -- an ISO 639 language code (in lowercase)
    language_preference    char(2) default 'en',
    dont_spam_me_p        char(1) default 'f' check (dont_spam_me_p in ('t','f')),
    email_type        varchar(64)
);

CREATE TABLE "cas_user_item_map" (
        "user_id" integer not null,
        "item_id" integer not null,
        "rating" integer CHECK (rating in (0,1,2,3,4,5)),
        "comment" text,
	"comment_add_date" date default CURRENT_DATE,
        unique(user_id,item_id)
);

-- create a view for easy access to the item ratings
create view cas_item_rating_summary as
	select item_id, trim(to_char(avg(rating), '9.9')) as rating_raw,
	 round(avg(rating)) as rating_rounded,
 	 count(comment) as comment_count
	 from cas_user_item_map
	 group by item_id;

-- explains which editors go with which category. 
-- categories may have multiple editors
create table cas_category_editor_map (
	category_id				int4, -- refers to category table
	editor_id				int4, -- refers to users table 
	unique (category_id,editor_id),
	-- which admin approved this relationship? 
    approved_date           date default CURRENT_DATE,
    approving_user          integer
);

-- holds session information for users
CREATE TABLE "cas_user_session" (
        "session_id" varchar(50) primary key,
        "user_id" int4,
        "real_user_id" int4,
        "remote_ip" inet,
        "start_time" timestamp default CURRENT_TIMESTAMP,
        "mod_time" timestamp,
        "end_time" timestamp
);
create index session_user_id on cas_user_session(user_id);

-- IMPORTANT: personalize the below information or you've got a security risk!
insert into cas_users
    (first_names, last_name, email, password, user_state,role)
   values 
    ('YourFirstName', 'YourLastName', 'YourEmail', 'YourPassword', 'authorized','admin');

-- if you want to get mail from the system when people suggest updates
--  set the below preference to 'f'
insert into cas_users_preferences (user_id,dont_spam_me_p) values (currval('cas_users_user_id_seq'),'t');
