#!/usr/bin/perl
# Copyright (C) 2001 Mark Stosberg <mark@stosberg.com>
# Licensed under the the GNU GPL, available here: http://www.gnu.org/copyleft/gpl.html

=pod

=head1 NAME

  Cascade::Item -- An OO module to manipulate item related data
  for Cascade. The current docs are incomplete.

=head1 SYNOPSIS

  my $item = Cascade::Item->new(id=>$item_id);
  return $item->output_html;

=head1 DESCRIPTION

=cut

package Cascade::Item;
use Cascade;

@ISA = qw(Exporter);

# Make just the constructors available by default
@EXPORT = qw(
 	&new
);
$VERSION = substr(q$Revision: 1.9 $, 10);  

use strict;
use vars qw/%DefaultItemFields/;
use CGI qw(popup_menu);

my @DefaultItemFields = (qw/
	item_id
	update_id
	url
	title
	description
	points
	first_names
	last_name
	org_name
	add1
	add2
	city
	state
	country
	postal_code
	phone
	fax
	email
	submitor_email
	rating_raw
	rating_rounded
	comment_count
	approved_date
/);
@DefaultItemFields{@DefaultItemFields} = ();

=pod

=head1 FUNCTION DOCUMENTATION

=head2 new

$item = Cascade::Item->new(id=>$item_id, mode=>'static');

INPUT

A hash of options with the following keys: 

=over 4

=item * id

An item id. If this is not defined, populating the item with data
based on the id no longer makes sense. However, in some cases you
might want to create an item object without an item_id and later
create html for the item from another source, such as info passed
through a form.  So we don't require the C<id> field here, but you
should Know What You're Doing if you don't use one  

=item * category_id

Including the category_id is sometimes useful to give some context to 
where the item is appearing, since it can appear in multiple categories. 
It's required for the C<html_form> method.

=item * mode

can be 'static' or 'dynamic'. Choose 'static' to produce output that
integrates with a stic site (ie, links go to html files) or 'dynamic'
to integrate with the dynamic version of the site (i.e., links go CGI
scripts).  Defaults the value of the CGI parameter 'mode', or
'dymanic' if that is not present.

=item * data

a reference to a hash of data, if you want to supply your own. defaults to false. 

=item * table

The table to draw your data from. Defaults to 'cas_item'

=item * change_category

Set to true if you want to be able to manipulate the category this
item belongs to when editting it, false otherwise. 

=back

RETURN VALUE

returns a Cascade::Item object. 

=cut

sub new {
    my $class = shift;
    my %in = (
    	id 		=> undef,
    	category_id 	=> undef, # if we care what category the item is in  
    	mode 		=> $FORM{'mode'} || 'dynamic',
    	table		=> 'cas_item', # default to displaying info from the item table
	change_category => 1, # can we change the category when editting? Usually so
    	data		=> undef, # usually we get the data based on the item's id
    	@_,
    );
    
	my ($key_col,$seq);
	if ($in{table} eq 'cas_item_suggested_updates') {
	   $key_col = 'item_update_id';
	} 
	else {
	   $key_col =  'item_id';
	   #$seq = 'item_id_seq';
	}
    
	if (!$in{data} and length $in{id}) {
       if ($CFG{DRIVER} eq "mysql") {
	   # This is a workaround for the "view" that Postgres uses, this is really
	   # horrible, and will introduce timing problems and lots of other weird
	   # stuff. But it worked! ;-) Simon Lindsay <simon@iseek.ws>
	   my $sth = $DBH->prepare(
			 "drop table if exists cas_item_rating_summary"
			 );
	   my $rv = $sth->execute;
	   $sth = $DBH->prepare(
			 "create temporary table cas_item_rating_summary " .
			 "select item_id, avg(rating) as rating_raw, " .
			 "round(avg(rating)) as rating_rounded, " .
			 "count(comment) as comment_count " .
			 "from cas_user_item_map " .
			 "group by item_id"
			 );
	   $rv = $sth->execute;
	   $in{data} = $DBH->selectrow_hashref("
               select  *,rating_raw,rating_rounded,coalesce(comment_count,0) as comment_count
                 from $in{table} item 
                 left join cas_item_rating_summary USING (item_id)
                 where item.$key_col = $in{id}") or
		 (warn 'invalid item id' && return undef);
	   $sth = $DBH->prepare(
			 "drop table if exists cas_item_rating_summary"
			 );
	   $rv = $sth->execute;
       } else {
         $in{data} = $DBH->selectrow_hashref("
               select  *,rating_raw,rating_rounded,coalesce(comment_count,0) as comment_count
                 from $in{table} item 
                 left join cas_item_rating_summary USING (item_id)
                 where item.$key_col = $in{id}") or
		 (warn 'invalid item id' && return undef);
       }
	}
	
	my %self = (
	  # XXX The "id" is really the item_update_id if $in{table} eq 'item_suggested_updates'
	  id	 	=> $in{id},
	  mode 		=> $in{mode},
	  data 		=> $in{data},
	  category_id 	=> $in{category_id},
	  key_col		=> $key_col,
	  table		=> $in{table},
	  change_category => $in{change_category},
	);

    $self{category_id} = $in{data}->{category_id} unless length $self{category_id};


#    carp $self{data};
    
    bless (\%self, $class);
    return \%self;   
}    

# An accessor mutator method
sub field {
	my $self = shift;
	my $field = shift;
	my $new_val = shift;
	$self->{data}->{$field} = $new_val if defined $new_val;
	return $self->{data}->{$field};
}

# return or set the category_id
# used on twice in Item.pm, can easily go away 
sub category_id {
	my $self = shift;
	if (@_) { $self->{category_id} = shift }
	return $self->{category_id};
}

# return the whole data hash
sub data {
	my $self = shift;
	return %{ $self->{data} };
} 

# returns reference to an array of item columns that hold public data
#  eventually different types of items will have different values here.. 
# ...and at the moment, we don't need this anyway.
# sub public_cols {
# 	return  \@DefaultItemFields;
# }

# returns an html link to this particular item
# we need it's category id to this-- we'll save this for later. :) -mls
# sub link {
#	my $self = shift;
#	
#	return $html;
#}

=pod 

=head2 output_html

$html = $item->output_html(date_added=>0,style=>'long');

INPUT

=over 4 

=item * date_added

Displays the date the item was added if true. Defaults to false. 

=item * style

Display. Can be 'short' for just a hyperlinked title. Displays full
information by default.

=item * data

a reference to a hash containing the name/value pairs for the item,
where the names are those in the item table. Use this when you want to
supply your own data for formatting, rather than using the data
provided by the 'new' method. This is useful when you want to build
the html for a category rather a single item. optional.

=item * display_comments_link

Set to a true value if you want to display the comments link, false
otherwise. Defaults to a true value.

=item * prefix

If you are supplying your own data with the C<data> attribute, you can
also pass in a custom prefix that will be stripped off the name of the
keys you passed in your data hash. I consider this kludgy so it's
deprecated. It may go away in a future version, in favor of use
L<Data::FormValidator> style handling instead.

=back

RETURN VALUE

an HTML formatted string. 

=cut

sub insert {
   my $self = shift or return undef;
	my %in = (
		data	   => $self->{data},	
		#state	   => 'needs_approval',
		prefix	   => 'item_', # prefix to strip off when data is supplied
	 	@_,
	 );
   # strip off a prefix out of the %data keys if we need to. 
   # also, delete un-usable fields
   my %data = _clean_data($in{prefix},%{ $in{data} });

   # delete  null labels for states and country codes if they aren't exactly 2 characters long. 
   foreach my $field ('state','country') {
      delete $data{$field} if ((length $data{$field}) != 2);
   }
    
   my $cat_id =  $FORM{category_id};
    
   # Turning AutoCommit off and back on effectively does a "begin" and "commit"
   # MYSQL doesnt need it
    
   $DBH->{PrintError} = 1;
   $DBH->{RaiseError} = 1;
   eval {
	    
      $DBH->{AutoCommit} = 0 if ( $CFG{DRIVER} eq 'Pg' ) ; 
	    
      # If we already have an item_id, then we are
      # making a link between an existing category and an item, 
      # so we can skip the rest of the stuff.
      # In this case, the resource is not counted twice-- just by 
      # the original parent. 

      if (length $FORM{item_id}) {
	 $DBH->do("insert into cas_category_item_map (category_id,item_id) 
	                  values ($FORM{category_id},$FORM{item_id})"); 
      } else {
	 if ($CFG{DRIVER} eq 'Pg') {
	    $data{item_id} =  $DBH->selectrow_array("SELECT nextval('cas_item_item_id_seq')");
	 }	 
			 
	 $data{approval_state} = 'approved' if ($SES{role_admin} or $SES{role_editor});

	 # We pre-select the CURRENT_DATE because MySQL doesn't currently allow us to default to CURRENT_DATE
	# This has been tested to work with both MySQL and Postgres. -mls
	 my $cur_date = $DBH->selectrow_array("SELECT CURRENT_DATE");
	 $data{insert_date} = $cur_date;
	$data{approved_date} = $cur_date if $data{approval_state} eq 'approved'; 

	 require DBIx::Abstract;
	 my $db = DBIx::Abstract->connect($DBH);
	 $db->insert($self->{table},\%data) or die;
	 if ( $CFG{DRIVER} eq 'Pg') {
	    $DBH->do("insert into cas_category_item_map 
		                  values ($cat_id, $data{item_id})");
	 } else {
	    # XXX should this really be "max(id)+1"? -mls
	    $data{item_id} = $DBH->selectrow_array("SELECT  max(item_id) FROM cas_item ") ;
	    $DBH->do("insert into cas_category_item_map 
		                  values ($cat_id, $data{item_id} )");
	 }
      } 	  
   };
   if ($@) {
     # XXX We really shouldn't be outputing HTML from this module,
     # what's good way to handle this? -mls
      err(title=>'Error Inserting', 
	  msg=>"There was an error inserting the item. The database returned the error:
				  $DBI::errstr"); 
   }
   else {
       $DBH->commit if ( $CFG{DRIVER} eq 'Pg' ); 
   }
   
   return $data{item_id}
   
   
}

sub output_html {
   my $self = shift || return undef;
   my %in = (
      date_added => 0,
      style	   => 'long',
      data	   => $self->{data},	
      display_comments_link => 1, 
      prefix	   => 'item_',	# prefix to strip off when data is supplied
      @_,
     );
	 
   # strip off a prefix out of the %data keys if we need to. 
   # also, delete un-usable fields
   my %data = _clean_data($in{prefix},%{ $in{data} });

   # we break the standard of using load_tmpl() here
   # because we don't have the CGI::App  object available. -mls
   require HTML::Template;
   my $t = $TMPL{'items/Default-display.tmpl'} ||= 
       HTML::Template->new(
	 %{ $CFG{HTML_TMPL_DEFAULTS} },
	 filename=>'items/Default-display.tmpl',
	 ); 
	$t->clear_params;
   delete $data{approved_date} unless $in{date_added};

   $t->param(display_comments_link=>$in{display_comments_link});
	
   # If they are an editor or admin, include the edit button
   if (($SES{role_admin} or $SES{role_editor}) && (length $data{item_id}) && ($self->{mode} ne 'static')) {
      $t->param(edit_link=>qq^$CFG{CASCADE_CGI}/item_edit_form?update_id=^.
             $self->{data}->{item_update_id}.qq^&item_id=$data{item_id}&category_id=^.
             $self->{category_id}.'&table='.$self->{table});
   } 
	
    if ($in{style} eq 'short') {
        my $cat = new Cascade::Category(id=>$self->{category_id});
		$t->param(direct_link=>$cat->name('url').'#'.$self->enc_title);
		$t->param(style_short=>1);
   } 
   else { 
      $t->param(anchor=>$self->get_anchor); 

      if ($data{rating_raw}) {
	 $t->param(graphical_rating=>graphical_rating($data{rating_rounded}));
      } 

      $data{comment_count} ||= 0;
      $t->param(postal_comma=>',') if ($data{city} && ($data{state} || $data{country} || $data{postal_code}) );

      if ( $data{state} || $data{country} ) {
	 $t->param(state_or_country=>1);
      }
       
      # we don't want to show the query string in the email links
      my ($linkable_email) = ($data{email} =~ m/(.*)\?/) ? $1 : $data{email};
      $t->param(linkable_email=>$linkable_email);
      $t->param(city_or_state=>1) if ($data{city} or $data{state});
   } 
   # prefix all the item fields with "item_" to avoid namespace clashes. -mls
   $t->param(
	   map {
	      my $k = $_;
	      'item_'.$k =>  $data{$k};
	   } keys (%data)
     );
   return $t->output;
}

# strips off prefixs if needed and removes unuseable data. 
sub _clean_data {
	my $prefix = shift;
	   $prefix or return @_;
	my %data = @_;
	
	foreach my $key (keys %data) {
		# strip off the prefix
	        # XXX with a hackish exception for item_id. -mls
		if (($key =~ m/^$prefix(.*)/) && ($key ne 'item_id')) {
		  my $new_key = $1;
		  $data{$new_key} = $data{$key};
		  delete 	$data{$key};
		  
		  # delete unuseable columns
		  unless (exists  $DefaultItemFields{$new_key}){
		     delete $data{$new_key}
		  }
		}
		else {
		  # delete unuseable columns
		  unless ((exists  $DefaultItemFields{$key})
		    and (length $data{$key})){
		     delete $data{$key}
		  }
		}
	}
	
	return %data;
}

sub output_text {
	my $self = shift or return undef;
	my %in = (
		data	=> $self->{data},
		prefix	=> 'item_', # prefix to strip off when data is supplied
		@_
	);
	
	my %d = _clean_data($in{prefix},%{ $in{data} });

	my $txt;
	foreach my $key (%d) {
		$txt .= "\t$key: $d{$key}\n" if $d{$key};	
	}
	

	return $txt;
}

# Encode the title
sub enc_title {
	my $self = shift;
	my $title = $self->{data}->{title};
	my $enc_title;
	($enc_title = CGI::escape($title) ) =~ s/%20/_/g;
	return $enc_title;
}

# create an html anchor
sub get_anchor {
	my $self = shift;
	return '<A NAME="'.$self->enc_title().'"></A>';
}

# returns an html form for this item, used for inserting and updating it. 
sub html_form {
   my $self = shift;
	
   my %in = (
	     change_category=> $self->{change_category},
	     @_
	);

   # XXX At the moment, we are dependent on having a category_id
   # I'd like to get of this dependency, which means not being able to 
   # to edit the categories than an item appears in on some screens, which I think
  # is OK. -mls 
   my $category_id = $self->{category_id};
   length $category_id or return undef;
   
   # Are we being called by an editor or admin?
   my $editor_context = is_cat_editor_or_admin($category_id);

   my (%item, $action);
		
   require Cascade::Category;
   my $cat = Cascade::Category->new(id=>$category_id) or
     die err(title=>'Bad Category Id',msg=>'The category id used does not appear to be valid');
		
   # If there's data, we assume it's an update
   if ($self->{id}) { 
      $action = 'edit';			   
      %item = %{ $self->{data} };

      # We need to test for a form category ID in case it's a link. 
      unless (defined $category_id) {
	 $category_id = $DBH->selectrow_array("select category_id from cas_category_item_map where category_id = $FORM{'id'}");
      }
      
   } else {
      $action = 'add';	
      $item{points} ||= $CFG{DEFAULT_POINTS};
      $item{state} = $CFG{DEFAULT_STATE_CODE};
      $item{country} = $CFG{DEFAULT_COUNTRY_CODE};
      # We give people a hint about what we expect in the URLs. -mls 
      $item{url} = 'http://';
   }			

   require DBIx::Abstract;
   my $db = DBIx::Abstract->connect($DBH);

   my $states = $db->select_all_to_hashref("code,name","cas_state_codes");
   $states->{''} = 'N/A';
   my $countries = $db->select_all_to_hashref("code,name","cas_country_codes");
   $countries->{''} = 'N/A';

	my %params = (
		mode=>$FORM{mode},
		action	=> $action,
		%item,
		page_title   => ucfirst $action,
		plain	=> $cat->name('plain'),
		category_id => $category_id,
		change_category=>$in{change_category},
		new_category_id_box => $cat->all_categories_box(
		   name=>'category_id',
		   default=>$category_id,
		   valid_children=>1,
		  ),
		editor_context => $editor_context,						
		item_points_popup_menu => popup_menu('item_points',[0,1,2,3,4,5],$item{points}),
		states_popup_menu => 
		  CGI::popup_menu(
		     -name=>'item_state',
		     -default=>$item{state} || '',
		     -values=>[ sort { $states->{$a} cmp $states->{$b} } keys %$states  ],
		     -labels=>$states,
		    ),
		country_popup_menu => 
		  CGI::popup_menu(
		     -name=>'item_country',
		     -default=>$item{country} || '',
		     -values=>[  sort { $countries->{$a} cmp $countries->{$b} } keys %$countries ],
		     -labels=>$countries,
		    ),
	     edit	=> ($action eq 'edit') && 1,
		&footer_html_tmpl
	);
   return %params;
}

sub already_exists {
	my $self = shift;
	
	# We check URL, email, phone number, and org name for dupes in the database, plus title within this category
    my $sql = "select item.item_id, category_item_map.category_id 
                FROM cas_item item, 
                     cas_category_item_map category_item_map
               where (category_item_map.item_id = item.item_id)  AND (";
    
    my @sql_pieces; 

# I think for now, I just want to check on the item title. -mark
#    push @sql_pieces, "lower(item.url) = ". lc base_url( $DBH->quote($FORM{item_url}) ) if $FORM{item_url};
#    push @sql_pieces, "lower(item.org_name) = ".lc( $DBH->quote($FORM{item_org_name}) ) if $FORM{item_org_name};
#    push @sql_pieces, "lower(item.email) = ".lc( $DBH->quote($FORM{item_email}) ) if $FORM{item_email};
    
    push @sql_pieces, "lower(item.title) = ".lc( $DBH->quote($FORM{item_title}) );

    $sql .= join ' OR ', @sql_pieces;
	
	$sql .= ")";
	# if we are doing an update, we don't want to find the item we are updating!
	$sql .= " AND item.item_id != ".$self->{data}->{item_id} if $self->{data}->{item_id};

    my $tbl_ref = $DBH->selectall_arrayref($sql);
    my @tbl = @$tbl_ref if defined $tbl_ref;
	 
    # If there's a match found
    if (scalar @tbl) {
	my (@results, $cat_name, %item);
	require Cascade::Category;
	
	# We're going to set our own action, so we delete the old instead of passing it through. 
	delete $FORM{rm};
	# set up some parameters that we will use in all cases
	my %params = (
		     self_html => $self->output_html(data=>\%FORM),
		     hidden_fields => (join "\n", map { CGI::hidden($_) if ($FORM{$_} ne "") } keys %FORM),
		     is_public     => (not ($SES{role_admin} or $SES{role_editor}) && 1),
		     category_id => $FORM{category_id},
		     # at this point the data has been validated, so we can add it directly 
		     #action => 'direct_add',
        );
	foreach my $row_ref (@tbl) {
	    my ($item_id, $cat_id) = @$row_ref;
	    my $cat = Cascade::Category->new(id=>$cat_id);
	    my $item = Cascade::Item->new(id=>$item_id,category_id=>$self->{category_id});
	    $cat_name = $cat->name('single_link');
	    
	    # Is there a conflict is this category? There should be only one, if so
	    if ($cat_id == $self->{category_id}) {
	       %params = (
			  %params,
			  category_conflict=>1,
			  item_html => $item->output_html(),
			  plain => $cat->name('plain'),
			    
	       );
	    } 
	    else {
	      
	       push @results, { CAT_NAME => $cat_name, 
				ITEM_ID => $item_id,
			        ITEM_HTML => $item->output_html()
			      }		
	    }
	}
	$params{items} = \@results;
	return %params;
    }
    else {
	   return undef;
	 }
}

sub validate_form {
   my $self = shift; 
   require Data::FormValidator;
   my $validator = new Data::FormValidator({
      form => {
	 optional => \@DefaultItemFields,	
	 required => 'title',
	 filters => 'trim',
	},		
     });
 
   # Get rid of multiple spaces, which get condensed when displayed in HTML anyway
   $FORM{'item_title'} =~ s/\s+/ /g;

   # empty out the URL field, if it just contains the default stuff. 
   delete $FORM{item_url} if ($FORM{item_url} =~ m!http://$!);

   my ($missing,$invalid);
   ($self->{data},$missing,$invalid) = $validator->validate({ _clean_data('item_',%FORM) }, 'form');
   return ($missing,$invalid);
}


# send alerts to cat editors and admins for a suggested insert or update 
sub send_alerts {
   my $self = shift; 
   my $update_id = shift;
   my $style = 'update' if $update_id;

   # If there are no email address to send to, we're done here. 
   require Cascade::Category;
   my $cat = Cascade::Category->new(id=>$FORM{category_id});
   my $to_emails = $cat->get_admin_and_cat_emails || return undef;
 
   my $from_email =  $FORM{item_submitor_email} || "alertbot\@$CFG{SITE_DOMAIN}";
   my $item_from_form = Cascade::Item->new(data=>\%FORM);

   # We have to create another item object to get the original contents, because the
   # current item objects contains the _suggested_update_ to the data
   my $item_from_db = Cascade::Item->new(id=>$self->{data}->{item_id});

   my ($approve_url,$edit_url);
   if ($style eq 'update') {
       $approve_url = "$CFG{CASCADE_CGI}/item_approve_update?update_id=$update_id&cat_id=$FORM{category_id}";
       $edit_url = "$CFG{CASCADE_CGI}/item_edit_suggestion_form?update_id=$update_id&category_id=$FORM{category_id}";
   }
   else {
       $approve_url = "$CFG{CASCADE_CGI}/item_approve_insert?item_id=".$self->{id}."&cat_id=$FORM{category_id}";
       $edit_url   =  "$CFG{CASCADE_CGI}/item_edit_suggestion_form?category_id=$FORM{category_id}&item_id=".$self->{id};
   }

   return {
        to_emails => $to_emails,
	from_email => 	$from_email,
	title	   => $FORM{item_title},
	cat_name_plain => $cat->name('plain'),
	item_from_form  => $item_from_form->output_text,
	item_from_db	=> $item_from_db->output_text,
	notes => $FORM{notes},
	approve_url => $approve_url,
	edit_url => $edit_url,
       };
}

sub update {
  my $self = shift;
  my %in = (
      data => $self->{data},
     );	

  # XXX I've hardcoded 'item_' as the prefix, because I expect this whole style to go away soon. -mls
  my %data = _clean_data('item_',%{ $in{data} });

  # XXX this hack may be removed with Data::FormValidator later. -mls
  $data{item_id} = $data{id} if (length $data{id});
  delete $data{id};
  $data{item_update_id} = $data{update_id} if (length $data{update_id});
  delete $data{update_id};

  require DBIx::Abstract;
  my $db = DBIx::Abstract->connect($DBH);

  $DBH->{AutoCommit} = 0;
  my $rv = $db->update($self->{table},\%data,$self->{key_col}." = ".$data{ $self->{key_col} });

    # if the old and new category ids don't match, we update the category_item_map table as well
    # This only needs to happen in the item table
   if (($self->{category_id} != $FORM{orig_category_id}) and ($self->{table} eq 'cas_item')) {
       # actually, first we try to update a link, and then if that doesn't work, we 
       # update the item
       $rv = $DBH->do("
            UPDATE cas_link
               SET from_cat_id = ".$self->{category_id}."
               WHERE to_item_id = ".$self->{id}."
                  AND from_cat_id = $FORM{orig_category_id}
       ");
       unless ($rv > 0) {
		$rv = $DBH->do("
			UPDATE cas_category_item_map
				SET category_id = ".$self->{category_id}."
				WHERE item_id = ".$self->{id}."
					AND category_id = $FORM{orig_category_id} ");    
       }
    }
   	$DBH->{AutoCommit} = 1;
    return $rv;	
}

sub delete {
  my $self = shift;
    # Maybe we can use cascading with foreign keys in Postgres to clean this up? -mark
    # Turning AutoCommit off and back on effectively does a "begin" and "commit"
    $DBH->{AutoCommit} = 0 if ( $CFG{DRIVER} eq 'Pg' )  ; 
    $DBH->do("delete from ".$self->{table}." where ".$self->{key_col}." = ".$self->{id});
    $DBH->do("delete from cas_category_item_map where item_id = ".$self->{id}) if $self->{table} eq 'item';
    $DBH->do("DELETE FROM cas_user_item_map WHERE item_id = ".$self->{id}) if $self->{table} eq 'item';
    $DBH->commit if ( $CFG{DRIVER} eq 'Pg' ) ; 
    return 1;
}

sub suggest_update {
    my $self = shift; 
    my %in = (
	data => $self->{data},
       );	
    
    # XXX I've hardcoded 'item_' as the prefix, because I expect this whole style to go away soon. -mls
    my %data = _clean_data('item_',%{ $in{data} });
    require DBIx::Abstract;
    my $db = DBIx::Abstract->connect($DBH);
    
    $DBH->{AutoCommit} = 0;
	
    if  ($CFG{DRIVER} eq 'mysql') {
       # do nothing for now, we'll get the item_update_id later
    } 
    else {
       $data{item_update_id} = $DBH->selectrow_array("SELECT nextval('cas_item_sugg_item_update_i_seq')");
    }      

    # make an exception for the category id
    $data{category_id} = $FORM{category_id};

    # XXX and a hackish switch for the item id:
    $data{item_id} ||= $data{id};
    delete $data{id};

    $db->insert('cas_item_suggested_updates',\%data) || die;
	
    $DBH->{AutoCommit} = 1;
	
    if ($CFG{DRIVER} eq 'mysql') {
       $data{item_update_id} = $DBH->selectrow_array("SELECT LAST_INSERT_ID()");
    }
    return $data{item_update_id};
}

# returns reference to array of real and virtual categories that this item appears in 
sub parent_cat_ids {
   my $self = shift;
   my $id= $self->{data}->{item_id};

   if ($CFG{DRIVER} eq "mysql") {
     my ($select1, $select2);
     $select1 = $DBH->selectcol_arrayref(
       "SELECT category_id from cas_category_item_map as m " .
       "WHERE item_id = $id"
       );
     $select2 = $DBH->selectcol_arrayref(
       "SELECT from_cat_id from cas_link " .
       "WHERE to_item_id = $id"
       );
     return [ @$select1, @$select2 ];
   } else {
     return $DBH->selectcol_arrayref("
      SELECT category_id from cas_category_item_map JOIN cas_item USING (item_id) WHERE item_id = $id
      UNION
      SELECT from_cat_id FROM cas_link WHERE to_item_id = $id ");
   }
}

# A class method to return the most recent items
sub most_recent {
   my $cutoff = shift || return undef;
    my $tbl = {};
	
    # find all items ordered by date
    if ($CFG{DRIVER} eq "mysql") {
	$tbl = $DBH->selectall_arrayref("
		   SELECT item.item_id, category_item_map.category_id
		  	 FROM cas_item as item, cas_category_item_map category_item_map
			WHERE item.item_id = category_item_map.item_id and approval_state = 'approved'
  			ORDER BY item.approved_date DESC, category_item_map.category_id
	  		LIMIT $cutoff
	  ");
    } else {
      $tbl = $DBH->selectall_arrayref("
           SELECT item.item_id, category_item_map.category_id
             FROM cas_item_approved as item, cas_category_item_map category_item_map
            WHERE item.item_id = category_item_map.item_id
            ORDER BY item.approved_date DESC, category_item_map.category_id
            LIMIT $cutoff
      ");
                                                                             
    }
   return $tbl;
}

1;

=pod

=head1 AUTHOR

Copyright (C) 2000-2001 Mark Stosberg <mark@stosberg.com>

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.
 
Address bug reports and comments to: mark@stosberg.com.  When sending
bug reports, please provide the version of Cascade, the version of
Perl, the name and version of your Web server, the name and version of
the operating system you are using, and the name and version of the
database you are using.  If the problem is even remotely browser
dependent, please provide information about the affected browers as
well.

=head1 SEE ALSO

perl(1).

=cut
