#!/usr/bin/perl
# Copyright (C) 2001 Mark Stosberg <mark@stosberg.com>
# Licensed under the the GNU GPL, available here: http://www.gnu.org/copyleft/gpl.html

# $Header: /cvsroot/cascade/cascade/lib/Cascade/Category.pm,v 1.6 2001/10/08 23:58:08 markjugg Exp $

=pod

=head1 NAME

  Cascade::Category - Functions to manipulate category related data for Cascade. 

=head1 SYNOPSIS

=head1 DESCRIPTION

=cut

package Cascade::Category;
use Cascade;
use Cascade::Item;
use CGI::Carp;

@ISA = qw(Exporter);

# Make just the constructor available by default
@EXPORT = qw(
	&new
);
$VERSION = substr(q$Revision: 1.6 $, 10);
use strict;

# Use this constructor when you want all the info readily available
sub new {
    my $class = shift;
    my %in = (
    	id	=> undef,
    	mode => $FORM{'mode'} || 'dynamic',
    	populate => 1, # populate the object with data? usually. 
    	@_,
    );
    
    my $id = $in{id};
    
    my (@multi_links,  $name,$single_link, $url,$enc_name,$relative,
    @plain_bits, @parents, $plain, %self, $sort_key, $full_url);
    
	if ($in{populate}) {    
	    # Get all the relevent information about this category and it's parents.
	    # We want to exclude the top category UNLESS it happens
	    # to be the top category where are interested in.. 
	    
	    $sort_key = $DBH->selectrow_array("
	    	SELECT sort_key
	    		FROM cas_category
	    		WHERE category_id = $in{id}") or return undef; # if there's no sort key, the category id is bad.
	    
	    my $tbl = $DBH->selectall_arrayref("
	    	SELECT category_id  as id, name
				FROM cas_category
				WHERE sort_key IN ('".(join "','", _get_ancestry_sort_keys($sort_key) )."')".
				(($id != 0) && ' and category_id <> 0').
				' ORDER BY sort_key');

	    while ($tbl && @$tbl) {
	       my $row = shift (@$tbl);
	       ($id,$name) = @$row;
	       push @parents, { id => $id, name => $name  };
	       
	       push @plain_bits, $name;

	       if ($in{mode} eq 'static') {
		  my @url_bits = @plain_bits;
		  $url = $CFG{HTML_ROOT_URL}.'/'
		    .(join '/', (map { $_= CGI::escape($_); s/%20/_/g; $_ } @url_bits))
		    .'/';
	       }
	       else {
		  $url = $CFG{CASCADE_CGI}.'/category_page?category_id='.$id
	       }	
	       # XXX There's a probably a good reason that $full_url and $url actually used to be different.
	       # Help me think of it. -mls
	       $full_url = $url;

	       push @multi_links, qq^<a href="$full_url">$name</a>^;
	    } 
	  
	    $plain = (join ': ', @plain_bits);

	    $enc_name = '/'.(join '/',  @plain_bits);
	    $enc_name = $CFG{DOWNCASE_DIRS} ? lc $enc_name : $enc_name;

# I think this sort of thing is probably the best way to handle internationalization issues
# but for now I'm disabling it because it breaks skatepark.org
# (it's also the case that at the moment the directory names don't mirror the url names
# so it would break your site, too. -mark
#		$enc_name =~ s/[^abcdefghijklmnopqrstuvwxyz0-9\ _\/]//gi ;
		$enc_name =~ s/\s+/_/g ;

	    # $relative links just the name for this category, instead of the full path of the name
	    if ($in{mode} =~ /dynamic/i) { 
		  	$relative = '<A HREF="'.$CFG{CASCADE_CGI}.'/category_page?category_id='.$id.'">'.$name.'</A>';	
		  	$single_link = '<a href="'.$CFG{CASCADE_CGI}.'/category_page?category_id='.$id.'">'.$plain.'</A>';
		} else {
			$relative = '<a href="'.$full_url.'">'.$name.'</A>';
			$single_link = '<a href="'.$full_url.'">'.$name.'</A>';
		}
  }

    %self = ( 	
    	multi_link 	=> (join ': ', @multi_links),  
	     name            => $name ,
	     single_link 	=> $single_link, 

	     # $url used to be relative, but now it's the same
	     # as full_url. I should probably deprecate one of them. - mls
	     url		=> $url,
	     full_url		=> $full_url,
	     enc_name        => $enc_name,
	     plain 			=> $plain,
	     relative		=> $relative,
	     mode			=> $in{mode},
	     id				=> $id,
	     sort_key		=> $sort_key,
	     parents	        => [@parents],
	);

    bless (\%self, $class);
    return \%self;
}

# with no arguments, returns the sort key
# if there is one argument, we assume it's new parent_id and return an appropriate key for a new child under this parent
sub sort_key {
	my $self = shift;
	my $parent_id = shift;
	defined $parent_id || return $self->{sort_key};
	
	my $parent_sort_key = $DBH->selectrow_array("select sort_key from cas_category where category_id = $parent_id");
	
	my $max_sibling_sort_key = $DBH->selectrow_array("
		SELECT max(sort_key) 
					FROM cas_category 
					WHERE parent_id = $parent_id
		".($self->{id} &&	"AND category_id <> ".$self->{id})
		);
			
	my $sort_key;
	# If there are siblings, we increment the sort key
	if ($max_sibling_sort_key) {			
		$sort_key = _inc_sort_key($max_sibling_sort_key) or
			croak qq^max sort key reached. 
				That's not going to work because we are out of sort keys for this category^;
	}
	# otherwise, we add 'add to the parent's sort key
	else { 				
		$sort_key = $parent_sort_key.'aa';
	}
	return $sort_key;
}


# return just the parent_id
sub parent_id {
	my $self = shift;
	pop @{ $self->{parents} }; # the first element is our id, so we discard it. 
	my $parent = pop @{ $self->{parents} } ;

	return $parent->{id};
}


sub name {
	my ($self, $style) = @_;
	
	# If it's the URL style and the category id is 0, we need to trim of the left piece of the url
	if ((lc $style eq 'url') and ($self->{id} == 0) and ($self->{mode} eq 'static')) {
		return $CFG{HTML_ROOT_URL};
	} else {
		return $self->{ lc $style }
	}	
}

# set or return the mode. 
# I don't actually use this now, since changing the mode would cause many things to change.
# sub mode {
#    my $self = shift;
#    if (@_) { $self->{mode} = shift }
#    return $self->{mode};
# }

# The difference between this and name('relative'), and this expects you to provide the name
# This should be N times faster when building the dynamic sub cats. 
sub relative {
    my ($self, $name) = @_;
	
    if ($self->{mode} eq 'dynamic') {
	return '<A HREF="'.$CFG{CASCADE_CGI}.'/category_page?category_id='. $self->{id}.'">'.$name.'</A>';	
    } else {
	return '<a href="'.$self->{url}.'">'.$name.'</A>';
    }
	
}

# returns hash of data to display a html_page for this category. Designed for HTML::Template
sub html_page {
  my $cat = shift || return undef;
  my %in = (
  	user_id	=> 0, # default to the public's view
  	@_
  );
  
  my $id = $cat->name('id');	
  my $static_cat = Cascade::Category->new(id=>$id, mode=>'static');

  # If we are in static mode, we turn off the editor and admin features
  # XXX this probably isn't the best way to do this. :) -mls
  if ($cat->{mode} eq 'static') {
      delete $SES{role_admin};
      delete $SES{role_editor};
  }

  return (
	  plain_name			=> $cat->name('plain'),
	  description			=> $DBH->selectrow_array("select description from cas_category where category_id = $id"),
	  top_url			=> $FORM{mode} eq 'static' ? $CFG{HTML_ROOT_URL} : $CFG{CASCADE_CGI},
	  update_item_link		=> "$CFG{CASCADE_CGI}/item_suggest_update_form?category_id=$id&mode=".$cat->{mode},
	  add_item_link			=> "$CFG{CASCADE_CGI}/item_add_form?category_id=$id&mode=".$cat->{mode},
	  multi_link			=> $cat->name('multi_link'),
	  related_cats		=> $cat->related_cats,
	  sub_cats			=> $cat->sub_cats,
	  &footer_html_tmpl,
	  items				=> $cat->get_item_html,
	  page_title				=> "$CFG{SYSTEM_NAME}: ".$cat->name('PLAIN'),
	  id					=> $id,
	  static_url			=> $static_cat->name('url'),
	  %SES
	 );
}


sub footer_cats {
	my $top_cat = Cascade::Category->new(id=>0);
	return $top_cat->sub_cats;
}

# returns a reference to an array of hash references with info about the subcats
sub sub_cats {
  my $self = shift; 
  
  my @sths;
  if ($CFG{DRIVER} eq "mysql") {
     # We get the sort_key here to help figure out the items_below count below. -mls
      push @sths, $DBH->prepare("
	  	 SELECT link.link_id, child.category_id as cat_id, link.name, child.description,sort_key
	       FROM cas_category as child, cas_link link
	      WHERE link.from_cat_id = ".$self->{id}."
	      	AND child.category_id = link.to_cat_id
	      	AND link.to_item_id is null
	      ORDER BY name
	  "), 
		  $DBH->prepare("
         SELECT child.category_id as cat_id, child.name, child.description,sort_key
           FROM cas_category as child
          WHERE child.parent_id = ".$self->{id}."
	      ORDER BY name
	  ");
  } 
  else {
      push @sths, $DBH->prepare("
	  	SELECT link.link_id, child.category_id as cat_id, link.name, (
	    	SELECT count(*)
                 FROM cas_item item,cas_category category_b, cas_category category, cas_category_item_map category_item_map
				WHERE item.item_id = category_item_map.item_id 
					AND category_item_map.category_id = category.category_id 
					AND category.sort_key like category_b.sort_key || '%'
					AND category_b.category_id = child.category_id
			) as items_below, child.description
	      FROM cas_category as child, cas_link link
	      WHERE link.from_cat_id = ".$self->{id}."
	      	AND child.category_id = link.to_cat_id
	      	AND link.to_item_id is null
	      ORDER BY name
	  "), 
	    $DBH->prepare("
               SELECT
                        child.category_id as cat_id,
                        child.name,
                                (SELECT count(*)
                                FROM cas_category category_b, cas_category_item_map
				    JOIN cas_category USING (category_id)
                                    JOIN cas_item_approved USING  (item_id)
                                 WHERE cas_category.sort_key like category_b.sort_key || '%' 
                                 AND category_b.category_id = child.category_id
                        ) AS
                        items_below,
                        child.description
              FROM cas_category as child
              WHERE child.parent_id = ".$self->{id}."
	      ORDER BY name
	  ");
  }

  my @subcats;
  foreach my $sth (@sths) {
     $sth->execute;
     while (my $row = $sth->fetchrow_hashref) {
	if ($CFG{DRIVER} eq "mysql") {
	   $row->{items_below} = $DBH->selectrow_array("
             SELECT count(*)
             FROM cas_item item,cas_category category, cas_category_item_map category_item_map
		WHERE item.item_id = category_item_map.item_id 
		AND category_item_map.category_id = category.category_id 
		AND category.sort_key like '".$row->{sort_key}."%'");
           # We're done with the sort_key now, so we'll get rid of it to simplify things. -mls
	   delete $row->{sort_key};
	}

	# If the mode is dynamic, we don't need to populate the category with data
	my $cat;
	if ($self->{mode} eq 'dynamic') {
	   $cat = Cascade::Category->new(id=>$row->{cat_id},mode=>$self->{mode},populate=>0);
	} else {
	   $cat = Cascade::Category->new(id=>$row->{cat_id},mode=>$self->{mode});
	}	
		   
	my $at;
	if ($row->{link_id}) {
	   $at = '@';
	   if ($SES{role_admin} or $SES{role_editor}) {
	      $row->{link} = 
		qq^[<A HREF="$CFG{CASCADE_CGI}/link_form?link_id=^.$row->{link_id}.
                '"><font size="-1" face="arial,helvetica" >edit</font></A>] '
				}
			}		   
		    $row->{link} .= $cat->relative($row->{name})."$at";
		    push @subcats, $row
	  }
  }
 
 #  for (@subcats) { carp "this: $_\n" };
  return \@subcats;	
}

sub related_cats {
    my $self = shift;
    my $to_cat_ids = $DBH->selectcol_arrayref("select to_cat_id from cas_related_category where from_cat_id = ".$self->{id});
	
    my @related_cats;
    require Cascade::Category;
    foreach my $id (@$to_cat_ids) {
	my $cat = Cascade::Category->new(id=>$id);
	push @related_cats, { single_link => $cat->name('single_link') };
    }
    return \@related_cats;
}

sub get_item_html {
	my $self = shift || return undef;
	my %in = (
		date_added => 0,
		style	   => undef,
	 	@_,
	 );

	require Cascade::Item;
	 
	
	my @asql;
	if ($CFG{DRIVER} eq "mysql") {
	    my $sth = $DBH->prepare(
				    "drop table if exists cas_item_rating_summary"
				    );
	    my $rv = $sth->execute;
	    $sth = $DBH->prepare(
				 "create temporary table cas_item_rating_summary " .
				 "select item_id, avg(rating) as rating_raw, " .
				 "round(avg(rating)) as rating_rounded, " .
				 "count(comment) as comment_count " .
				 "from cas_user_item_map group by item_id"
				 );	
	    $rv = $sth->execute;
	    @asql = ("
 SELECT item.*, rating_raw,
        rating_rounded,coalesce(comment_count,0) as comment_count
	FROM cas_category_item_map category_item_map,
	cas_item item 
	LEFT JOIN cas_item_rating_summary 
		ON cas_item_rating_summary.item_id = item.item_id
	WHERE item.item_id = category_item_map.item_id
		AND item.approval_state = 'approved'
		AND category_item_map.category_id = ".$self->name('id'),
	    " SELECT item.* from cas_item as item,cas_link as link
	     WHERE (item.item_id = link.to_item_id   
              AND item.approval_state = 'approved'
              AND link.from_cat_id = ".$self->name('id').") 
	     ORDER by points DESC" ) ;
    } 
    else {
		@asql =  ("
 SELECT item.*, rating_raw,
        rating_rounded,coalesce(comment_count,0) as comment_count
	FROM cas_category_item_map category_item_map,
	cas_item_approved item 
	LEFT JOIN cas_item_rating_summary 
		ON cas_item_rating_summary.item_id = item.item_id
	WHERE item.item_id = category_item_map.item_id
		AND category_item_map.category_id = ".$self->name('id')."
UNION
SELECT item.*, rating_raw,
       rating_rounded,coalesce(comment_count,0) as comment_count
	FROM cas_link link,
	cas_item_approved item
	LEFT JOIN cas_item_rating_summary 
		ON cas_item_rating_summary.item_id = item.item_id 
	WHERE item.item_id =link.to_item_id
	AND link.from_cat_id =  ".$self->name('id')." 
	ORDER by rating_raw DESC, comment_count DESC, points DESC ") ;
    }
    
    my @items;
	foreach my $sql (@asql) 	{
	    my $sth = $DBH->prepare($sql);
	    my $rv = $sth->execute;
	    while (my $h_ref = $sth->fetchrow_hashref) {
	       my $item = Cascade::Item->new(data=>$h_ref,category_id=>$self->{id});
	       push @items, $item->output_html(%in,date_added=>0);
		}	
	}
	
	return CGI::li([@items]);
}

# ironically, this doesn't handle category editors yet.
# ...funny, since  it's already in Cascade::Category.pm
# later, when I actually finish this, let's make it go only editor, one group of editor
# the first group of editors for this category above that is not null. 
# it excludes the email address of the logged in user.
sub get_admin_and_cat_emails {
	my $self = shift;

	# user_id defaults to zero to make valid SQL. Zero would never be a valid user_id, right? 
	my $user_id = is_cat_editor_or_admin($self->{id}) || 0;
	my $email_ref = $DBH->selectcol_arrayref("
		SELECT email
		FROM cas_users users, cas_users_preferences users_preferences
		WHERE role = 'admin'
			AND users.user_id = users_preferences.user_id
			AND users_preferences.dont_spam_me_p = 'f'
			AND users.user_id != $user_id
	") or return undef;
	my $email_string = join ',', @$email_ref;
	$email_string =~ s/\n//g;
	return $email_string;
}

sub update {
    my $self = shift;
    eval {
	require DBIx::Abstract;
	my $db = DBIx::Abstract->connect($DBH);
    
	# For now we get our data to update using the
	# form fields that start with "cat_"
	# This will probably change in the future
	my %cat_data = %FORM;
	foreach my $key (keys %cat_data) {
	    if ($key =~ /^cat_(.*)/i) {
		$cat_data{$1} = $cat_data{$key};
	    }
	    delete$cat_data{$key};
	}
	
	# do a switcheroo with the category id name
	# This hack will probably removed with Data::FormValidator at some point. -mls
	$cat_data{category_id} = $cat_data{id};
	delete $cat_data{id};
	$db->update('cas_category',\%cat_data,"category_id = ".$self->{id});
    };
    return $@ ? 0 : 1;
}

# Updates the sort_keys for this category and below.
# useful when moving and deleting categories
sub update_sort_keys {
	my $self = shift;
    
    # We need to get a new parent_id in case we were moving this category
    # ( it took me a long time to figure this out.... -mark )
    my $parent_id = $DBH->selectrow_array("SELECT parent_id FROM cas_category WHERE category_id = ".$self->{id});
    $DBH->do(" UPDATE cas_category
			SET sort_key = '".$self->sort_key($parent_id)."'
			WHERE category_id = ".$self->{id});

    _update_subcats_sort_key( $self->{id} );

}

# increment the sort_key
# return undef on error 
sub _inc_sort_key {
	my $old_key = shift || 'aa'; # start with 'aa' if we don't have anything
	my $new_key = ++$old_key;
	return ( length($old_key) == length($new_key) ) ? $new_key : undef;
}

# get the ancestors of this sort key, starting from the top
sub _get_ancestry_sort_keys {
	my $sort_key = shift;
	my @ancestry_sort_keys;
	my $i = length($sort_key); 
	until ($i == 0) {
		push @ancestry_sort_keys, (substr($sort_key,0,$i));
		$i -= 2;
	}	
	return @ancestry_sort_keys;
}

sub _update_subcats_sort_key {
	my $parent_id = shift;
	my $cats_ref = $DBH->selectcol_arrayref("
		SELECT category_id 
			FROM cas_category 
			WHERE parent_id = $parent_id
			ORDER BY sort_key");
	foreach my $child_id (@$cats_ref){
		my $cat = new Cascade::Category (id=>$child_id);
		$cat->update_sort_keys;
	}
	return;
}


# used for selecting a new parent category id
# display a form widget to select categories from
# input is hash of options to pass through to CGI.pm including:
# name, multiple, size 
sub all_categories_box {
   my $cat = shift;		
   my %in = (
      name		=> undef,
      multiple	=> 0,
      size		=> 1, 
      default		=> $cat->parent_id,
      # Is valid to select a child of the category? 
      # Usually not. It doesn't make sense to to attach a category to 
      # to one of it's own children, but if you are moving an item
      # there, that's fine. 
      valid_children  => 0,	
      # sometimes it doesn't make sense to the top category, like when we are 
      # choosing a virtual subcategory, but by default, we include it. 
      include_top     => 1, 
      @_
     );

   my @where_bits;
   push @where_bits, "sort_key not like '".$cat->sort_key."%'" if not $in{valid_children};
   push @where_bits, "category_id != 0" if not $in{include_top};
   my $where_sql = @where_bits ? "WHERE ".(join " AND ", @where_bits) : "";
   my $tbl = $DBH->selectall_arrayref("
      SELECT category_id, name, (length(sort_key)/2)-1 as level 
	FROM cas_category $where_sql ORDER BY sort_key");

   my %parents;	
   my $q = new CGI qw(autoEscape scrolling_list);
   my @p_ids;
   foreach my $row (@$tbl) {
      my ($id,$name,$level) = @$row;
      $parents{$id} = '--&nbsp;'x$level.CGI::escapeHTML($name);
      push @p_ids, $id;		# we keep the ids in a seperate array to preserve sorting
   }		

   $q->autoEscape(undef);	# we need to turn this off for the spaces in the list to show up correctly. 
	
   # use $in{size} or scalar @p_ids, whichever is smaller
   my $size = ($in{size} <= @p_ids) ? $in{size} : scalar @p_ids;
	
   return $q->scrolling_list(
      -name		=>$in{name},
      -values		=>[@p_ids],
      -labels		=>\%parents,
      -default	=>$in{default},
      -multiple	=>$in{multiple},
      -size		=>$size,
     );
}

1;

=pod

=head1 AUTHOR

Copyright (C) 2000-2001 Mark Stosberg <mark@stosberg.com>

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.
 
Address bug reports and comments to: mark@stosberg.com.  When sending
bug reports, please provide the version of Cascade, the version of
Perl, the name and version of your Web server, the name and version of
the operating system you are using, and the name and version of the
database you are using.  If the problem is even remotely browser
dependent, please provide information about the affected browers as
well.

=head1 SEE ALSO

perl(1).

=cut
