/***********************************************************************
 *                  wsmake - A website pre-processor                   *
 *           Copyright (C) 1999,2000,2001 Michael Brownlow             *
 *                                                                     *
 * This program is free software; you can redistribute it and/or modify*
 * it under the terms of the GNU General Public License as published by*
 * the Free Software Foundation; either version 2 of the License, or   *
 * (at your option) any later version.                                 *
 *                                                                     *
 * This program is distributed in the hope that it will be useful,     *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of      *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the       *
 * GNU General Public License for more details.                        *
 *                                                                     *
 * You should have received a copy of the GNU General Public License   *
 * along with this program; if not, write to the Free Software         *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.           *
 *                                                                     *
 * For questions and comments, please email the author at:             *
 * mike@wsmake.org                                                     *
 ***********************************************************************/

#if HAVE_CONFIG_H
# include "config.h"
#endif

/* For OSX and Fink... */
#if HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#include <dirent.h>
#include <unistd.h>

#include <cstdlib>

#include <iostream>
#include <list>

#include "wsWebSite.h"
#include "wsmake.h"

#if HAVE_GETOPT_LONG
# if LOCAL_GETOPT
#  include "../getopt/getopt.h"
# else
#  include <getopt.h>
# endif
#endif

using namespace std;

struct setup {
  int debugLevel;
  int force;
  int quiet;
  int have_targets;
  int target_clean;
  int match_end;
  int clean;
  int match_sub;
  int times;
  list< string > targets;
};

void printOptions();
void printVersion();
void applyToWebsite(string, setup *);

int main(int argc, char *argv[])
{
  DIR *dp;
  struct dirent *entry;
  string config_file;
  char *p;
  list< string > configs;
  list< string >::const_iterator i;
  int opt;
  setup su;
#if HAVE_GETOPT_LONG
  int option_index = 0;
  static struct option long_options[] =
  {
    {"force",0,0,'F'},
    {"silent",0,0,'s'},
    {"quiet",0,0,'s'},
    {"help",0,0,'h'},
    {"version",0,0,'v'},
    {"clean",0,0,'c'},
    {"times",0,0,'t'},
    {"match",1,0,'m'},
    {"end-match",1,0,'e'},
    {"debug",1,0,'d'},
    {"file",1,0,'f'},
    {"wsmakefile",1,0,'f'},
  };
#endif /* HAVE_GETOPT_LONG */

  su.target_clean = 0;
  su.debugLevel = 1;
  su.force = 0;
  su.quiet = 0;
  su.match_end = 0;
  su.match_sub = 0;
  su.clean = 0;
  su.times = 0;
  
#if HAVE_GETOPT_LONG
  while((opt=getopt_long(argc,argv,"hcetmvsqFd:f:",
			 long_options,&option_index)) != -1) {
#else
    while((opt=getopt(argc,argv,"hcetmvsqFd:f:")) != -1) {
#endif /* HAVE_GETOPT_LONG */
      switch(opt) {
    case 'h':
      printOptions();
      exit(0);
      break;
    case 'c':
      su.clean = 1;
      break;
    case 'e':
      su.match_end = 1;
      break;
    case 'm':
      su.match_sub = 1;
      break;
    case 'q':
    case 's':
      su.quiet = 1;
      break;
    case 't':
      su.times = 1;
      break;
    case 'd':
      su.debugLevel = atoi(optarg);
      break;
    case 'F':
      su.force = 1;
      break;
    case 'f':
      config_file.assign(optarg);
      break;
    case ':':
    case '?': // Shouldn't really need this, if no value is given for option
              // d then it should return a ':', but my machine returns a '?'.
              // This also means that unknown options will cause the program
              // to exit as well.
      printOptions();
      exit(1);
      break;
    case 'v':
      printVersion();
      exit(0);
      break;
    }
  }
  
  if(su.match_end && su.match_sub) {
    __wsmake_print(1,"warning: using substring matching.\n");
    su.match_end = 0;
  }

  su.have_targets=0;
  while(optind <argc) {
    // Accumulate targets
    if(!strcmp(argv[optind],"clean")) {
      su.target_clean = 1;
      su.clean = 1;
    }
    su.targets.push_back(argv[optind++]);
    su.have_targets=1;
  }

  if(config_file.length() == 0) {
    if((dp = opendir(".")) == NULL) {
      __wsmake_print_error("couldn't load '.' directory to search for "
			   ".ws files.\n");
      exit(EXIT_FAILURE);
    }
    while((entry = readdir(dp)) != NULL) {
      if(!strcmp("wsmakefile", entry->d_name)) {
	configs.push_back("wsmakefile");
      } else {
	if((p=strstr(entry->d_name, ".ws")) != NULL) {
	  if(strlen(p) == 3) {
	    configs.push_back(entry->d_name);
	  }
	}
      }
    }
    closedir(dp);
  } else {    
    configs.push_back(config_file);
  }


  if(configs.size() == 0) {
    __wsmake_print_error("no configuration files specified or found.\n");
    exit(EXIT_FAILURE);
  } else {
    for(i=configs.begin(); i!=configs.end(); i++) {
      wsmake_init();
      applyToWebsite(*i, &su);
      wsmake_exit();
    }
  }

  exit(EXIT_SUCCESS);
}
  
void applyToWebsite(string config_file, setup *su)
{
  wsWebSite *ws;

  ws = new wsWebSite(su->debugLevel, su->quiet, su->times);
  __wsmake_print(2, "Loading configuration (%s):\n", config_file.c_str());
  if(!ws->load(&config_file, su->force)) {
    __wsmake_print_error("loading of config file failed: %s\n",
			 config_file.c_str());
  } else {
    ws->print();
    __wsmake_print(2, "Synchronizing database with filesystem:\n");
    ws->sync();
    __wsmake_print(2, "Making website:\n");
    if(su->have_targets && !su->target_clean) {
      if(su->match_end) {
	if(!su->clean) {
	  ws->make(su->targets, MATCH_END);
	} else {
	  ws->clean(su->targets, MATCH_END);
	}
      } else if(su->match_sub) {
	if(!su->clean) {
	  ws->make(su->targets, MATCH_SUB);
	} else {
	  ws->clean(su->targets, MATCH_SUB);
	}
      } else {
	if(!su->clean) {
	  ws->make(su->targets, MATCH_BEG);
	} else {
	  ws->clean(su->targets, MATCH_BEG);
	}
      }
    } else {
      if(!su->clean) {
	ws->make();
      } else {
	ws->clean();
      }
    }
  }

  delete ws;
}

void printOptions()
{
#if !HAVE_GETOPT_LONG
  cout << PACKAGE << " [-Fshvct] [-m|-e] [-d #] [-f configfile] [target] ..."
       << endl;
  cout <<
" -F                    Force files to be updated regardless of their state."
       << endl;
  cout <<
" -t                    Sync timestamps between source and output files."
       << endl;
  cout <<
" -s                    Silent. No output." << endl;
  cout <<
" -h                    This help." << endl;
  cout <<
" -v                    Show version and copyright information." << endl;
  cout <<
" -c                    Clean the website." << endl;
  cout <<
" -m                    Targets match anywhere in web paths." << endl;
  cout <<
" -e                    Targets match at end of web paths." << endl;
  cout <<
" -d #                  Debug output level. (0-3)" << endl <<
"                       0 = No output except for errors" << endl <<
"                       1 = 0 and minimal output (default)" << endl <<
"                       2 = 1 and verbose output" << endl <<
"                       3 = 2 and config file loading info" << endl;
  cout <<
" -f file               Configuration filename (looks for 'wsmakefile' and "
       << endl <<
"                       files ending with '.ws' if option not given)."
       << endl;
  cout <<
" targets               Specific web paths, matching from front, to make. If"
       << endl <<
"                       not specified, all pages will be made." << endl;
#else
  cout << PACKAGE << " [-Fshvct] [-m|-e] [-d #] [-f configfile] [target] ..." << endl;
  cout <<
" -F,--force            Force files to be updated regardless of their state."
       << endl;
  cout <<
" -t,--times            Sync timestamps between source and output files."
       << endl;
  cout <<
" -s,--silent,--quiet   Silent. No output." << endl;
  cout <<
" -h,--help             This help." << endl;
  cout <<
" -v,--version          Show version and copyright information." << endl;
  cout <<
" -c,--clean            Clean the website." << endl;
  cout <<
" -m,--match            Targets match anywhere in web paths." << endl;
  cout <<
" -e,--end-match        Targets match at end of web paths." << endl;
  cout <<
" -d #,--debug=#        Debug output level. (0-3)" << endl <<
"                       0 = No output except for errors" << endl <<
"                       1 = 0 and minimal output (default)" << endl <<
"                       2 = 1 and verbose output" << endl <<
"                       3 = 2 and config file loading info" << endl;
  cout <<
" -f file,--file=file,  Configuration filename (looks for 'wsmakefile' and "
       << endl <<
"   --wsmakefile=file   files ending with '.ws' if option not given)."
       << endl;
  cout <<
" targets               Specific web paths, matching from front, to make. If"
       << endl <<
"                       not specified, all pages will be made." << endl;

#endif /* !HAVE_GETOPT_LONG */
  cout << "Report bugs to <bugs@wsmake.org>." << endl;
}

void printVersion()
{
  cout << PACKAGE << " " << VERSION << " (" << __DATE__ << "), "
       << "Copyright (C) 1999-2001 Michael Brownlow" << endl << endl;
  cout << "Wsmake may be copied only under the terms of either the Artistic "
       << "License or " << endl;
  cout << "the GNU General Public License, which may be found in "
       << "the Wsmake " << VERSION << " source " << endl;
  cout << "distribution." << endl;
}
