/* $Id: wsWebSite.cpp,v 1.2 2002/05/10 09:00:39 mike Exp $
 ***********************************************************************
 *         libwsmake - Core functionality library of wsmake            *
 *           Copyright (C) 1999,2000,2001 Michael Brownlow             *
 *                                                                     *
 * This program is free software; you can redistribute it and/or modify*
 * it under the terms of the GNU General Public License as published by*
 * the Free Software Foundation; either version 2 of the License, or   *
 * (at your option) any later version.                                 *
 *                                                                     *
 * This program is distributed in the hope that it will be useful,     *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of      *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the       *
 * GNU General Public License for more details.                        *
 *                                                                     *
 * You should have received a copy of the GNU General Public License   *
 * along with this program; if not, write to the Free Software         *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.           *
 *                                                                     *
 * For questions and comments, please email the author at:             *
 * mike@wsmake.org                                                     *
 ***********************************************************************/
#include "dba.h"
#include "wsWebPage.h"
#include "wsWebSite.h"
#include "wsUtil.h"
#include "wsmake.h"

using namespace std;

wsWebSite::wsWebSite(int debug_level, int quiet, int times)
{
  this->setDebugLevel(debug_level);
  __wsmake_set_quiet(quiet);
  this->setForce(0);
  this->setConfigFileName("");
  this->setTimes(times);

  pagegroup_list = new wsPageGroupList();

  DBGPRINT(("O:created: wsWebSite\n"));
}

wsWebSite::~wsWebSite()
{
  pagegroup_list->free();
  delete pagegroup_list;

  DBGPRINT(("O:destroyed: wsWebSite\n"));
}

int wsWebSite::sync()
{
  pagegroup_list->sync(force);
  if(name.length() > 0) {
    __wsmake_print(2,"\nWebsite: %s\n", name.c_str());
  }
  return 1;
}

int wsWebSite::makePages()
{
  if(!pagegroup_list->make())
    return -1;

  return 0;
}

int wsWebSite::makePages(list<string> targets, int matchtype)
{
  if(!pagegroup_list->make(targets,matchtype))
    return -1;

  return 0;
}

int wsWebSite::cleanPages()
{
  if(!pagegroup_list->clean())
    return 1;

  return 0;
}

int wsWebSite::cleanPages(list<string> targets, int matchtype)
{
  if(!pagegroup_list->clean(targets,matchtype))
    return 1;

  return 0;
}

void wsWebSite::setDebugLevel(int debug_level)
{
  if(debug_level > MAX_DEBUG_LEVEL) {
    __wsmake_set_level(MAX_DEBUG_LEVEL);
    __wsmake_print_error("warning: debug level lowered to maximum - %d\n",
	     MAX_DEBUG_LEVEL);
  } else {
    if(debug_level < MIN_DEBUG_LEVEL) {
      __wsmake_set_level(MIN_DEBUG_LEVEL);
      __wsmake_print_error("warning: debug level raised to minimum - %d\n",
	       MIN_DEBUG_LEVEL);
    } else {
      __wsmake_set_level(debug_level);
    }
  }
}

void wsWebSite::setForce(int force)
{
  if(force!=0) {
    this->force = 1;
  } else {
    this->force = 0;
  }
}

void wsWebSite::setTimes(int times)
{
  this->times = times;
}

void wsWebSite::setConfigFileName(string config_filename)
{
  this->config_filename.assign(config_filename);
}

string wsWebSite::getConfigFileName(void)
{
  return this->config_filename;
}

string wsWebSite::getConfigFileDir(void)
{
  string config_dir("");

  if((getConfigFileName()[0] != '/') && (getConfigFileName().find("/") !=
					 string::npos)) {
    config_dir = getConfigFileName().substr(0, getConfigFileName().rfind("/"));
  }

  if((config_dir == "") || (config_dir[0] != '/')) {
    char *buf = new char[512];
    buf = getFullPath(buf);
    if(config_dir != "") {
      config_dir.insert(0,"/");
    }
    config_dir.insert(0,buf);
    delete [] buf;
  }

  return config_dir;
}

int wsWebSite::make(void)
{
  int error;

  if((error=makePages()) != 0) {
    __wsmake_print_error("make error = %d\n", error);
  }

  return error;
}

int wsWebSite::make(list<string> targets, int matchtype)
{
  int error;

  if((error=makePages(targets,matchtype)) != 0) {
    __wsmake_print_error("make error = %d\n", error);
  }

  return error;
}

int wsWebSite::clean(void)
{
  int error;

  if((error=cleanPages()) != 0) {
    __wsmake_print_error("make error = %d\n", error);
  }

  return error;
}

int wsWebSite::clean(list<string> targets, int matchtype)
{
  int error;

  if((error=cleanPages(targets,matchtype)) != 0) {
    __wsmake_print_error("make error = %d\n", error);
  }

  return error;
}

int wsWebSite::dump(ofstream *output)
{
  *output << "dump: " << name << endl;

  return 1;
}

void wsWebSite::print(void) const
{
  __wsmake_print(3,"Website\n");
  __wsmake_print(3,"====================\n");
  if(this->name.length() != 0) {
    __wsmake_print(3,"  Name             : %s\n", this->name.c_str());
  }
  if(this->maintainer.length() != 0) {
    __wsmake_print(3,"  Maintainer       : %s\n", this->maintainer.c_str());
  }
  if(this->URL.length() != 0) {
    __wsmake_print(3,"  URL              : %s\n", this->URL.c_str());
  }
  __wsmake_print(3,"====================\n");
  __wsmake_print(3,"\n");
  pagegroup_list->print();
  __wsmake_print(3,"\n");
}

int wsWebSite::getSectionID(const string *tag)
{
  if(strcmp(tag->c_str(),"Website") == 0)
    return SECTION_WEBSITE;
  else if(strcmp(tag->c_str(), "include") == 0)
    return TAG_INCLUDE;

  return UNKNOWN;
}

int wsWebSite::load(string *config_filename, int force)
{
  this->setForce(force);
  this->setConfigFileName(*config_filename);

  return load();
}

int wsWebSite::load()
{
  string buff;
  string tag;
  string value;
  int done = 0;
  int line = 0;
  list<ifstream *> config;
  ifstream *ifs = NULL;

  ifs = new ifstream(config_filename.c_str(), std::ios::in);
  if(!*ifs) return 0;

  config.push_front(ifs);

  while(!done) {
    line=getNextConfigLine(config.front(), &buff);

    if(line == 0) {
      config.pop_front();
      if(config.size() == 0) {
	done = 1;
      }
    } else {
      splitString(&tag, &value, &buff);
      
      switch(this->getSectionID(&tag)) {
      case SECTION_WEBSITE :
	if(!this->loadWebsite(config.front()))
	  return 0;
	break;
      case TAG_INCLUDE :
	chopQuotes(&value);
	if(value[0] != '/') {
	  value = this->getConfigFileDir() + "/" + value;
	}
	config.push_front(new ifstream(value.c_str(), ios::in));
	break;
      case UNKNOWN :
      default :
	__wsmake_print_error("unknown config section (line %d): %s\n",line,
			     tag.c_str());
	return 0;
	break;
      };
    }
  }

  return 1;
}

int wsWebSite::getWebsiteTagID(const string *tag)
{
  if(strcmp(tag->c_str(),"name") == 0)
    return TAG_NAME;
  else if(strcmp(tag->c_str(),"maintainer") == 0)
    return TAG_MAINTAINER;
  else if(strcmp(tag->c_str(),"url") == 0)
    return TAG_URL;
  else if(strcmp(tag->c_str(),"include") == 0)
    return TAG_INCLUDE;
  else if(strcmp(tag->c_str(),"PageGroup") == 0)
    return SECTION_PAGEGROUP;

  return UNKNOWN;
}

int wsWebSite::loadWebsite(ifstream *input)
{
  string buff;
  string tag;
  string value;
  int done = 0;
  int line = 0;
  int pagegroups = 0;
  int pages = 0;
  list<ifstream *> config;

  config.push_front(input);

  while(!done) {
    line=getNextConfigLine(config.front(), &buff);

    if(line == 0) {
      config.pop_front();
      if(config.size() == 0) {
	done = 1;
      }
    } else {
      
      splitString(&tag, &value, &buff);
      
      if(tag.find("}") != string::npos) {
	done = 1;
      } else {
	
	switch(this->getWebsiteTagID(&tag)) {
	case TAG_NAME :
	  chopQuotes(&value);
	  this->name = value;
	  break;
	case TAG_MAINTAINER :
	  chopQuotes(&value);
	  this->maintainer = value;
	  break;
	case TAG_URL :
	  chopQuotes(&value);
	  this->URL = value;
	  break;
	case TAG_INCLUDE :
	  chopQuotes(&value);
	  if(value[0] != '/') {
	    value = this->getConfigFileDir() + "/" + value;
	  }
	  config.push_front(new ifstream(value.c_str(), ios::in));
	  break;
	case SECTION_PAGEGROUP :
	  if(!pagegroup_list->pushBackPageGroup
	     (new wsPageGroup(this->getConfigFileDir(),config.front(),times))) {
	    __wsmake_print_error("unable to add pagegroup (line %d)\n", line);
	  }
	  pagegroups++;
	  pages = pagegroup_list->numPages();
	  break;
	case UNKNOWN :
	default :
	  __wsmake_print_error("unknown website section skipped "
			       "(line %d): %s\n",line,tag.c_str());
	  return 0;
	  break;
	};
      }
    }
  }

  __wsmake_print(2,"%d PageGroup%s, %d Page%s.\n",
		 pagegroups,(pagegroups==1)?"":"s",
		 pages,(pages==1)?"":"s");

  return 1;
}
