/* $Id: wsTheme.cpp,v 1.2 2001/09/01 20:15:01 mike Exp $
 ***********************************************************************
 *         libwsmake - Core functionality library of wsmake            *
 *           Copyright (C) 1999,2000,2001 Michael Brownlow             *
 *                                                                     *
 * This program is free software; you can redistribute it and/or modify*
 * it under the terms of the GNU General Public License as published by*
 * the Free Software Foundation; either version 2 of the License, or   *
 * (at your option) any later version.                                 *
 *                                                                     *
 * This program is distributed in the hope that it will be useful,     *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of      *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the       *
 * GNU General Public License for more details.                        *
 *                                                                     *
 * You should have received a copy of the GNU General Public License   *
 * along with this program; if not, write to the Free Software         *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.           *
 *                                                                     *
 * For questions and comments, please email the author at:             *
 * mike@wsmake.org                                                     *
 ***********************************************************************/
#include "wsTheme.h"
#include "wsUtil.h"
#include "wsmake.h"

using namespace std;

wsTheme::wsTheme(ifstream* config, wsPageGroup *pg)
  : updated(false), pagegroup(pg)
{
  this->subtag_list = new wsSubTagList();
  this->subtaggroup_list = new wsSubTagGroupList();
  this->pageorder_list = new wsPageOrderList();
  this->depend_list = new wsDependList();
  this->setCommand("");
  this->setOptions("");

  this->load(config, "", 0);

  DBGPRINT(("O:created: wsTheme\n"));
}

wsTheme::wsTheme(ifstream *config, wsPageGroup *pg,
		 string db_file, int db_type)
  : updated(false), pagegroup(pg)
{
  this->subtag_list = new wsSubTagList();
  this->subtaggroup_list = new wsSubTagGroupList();
  this->pageorder_list = new wsPageOrderList();
  this->depend_list = new wsDependList();
  this->setCommand("");
  this->setOptions("");

  this->load(config, db_file, db_type);

  DBGPRINT(("O:created: wsTheme\n"));
}

wsTheme::~wsTheme()
{
  subtag_list->free();
  depend_list->free();

  delete subtag_list;
  delete subtaggroup_list;
  delete pageorder_list;
  delete depend_list;

  DBGPRINT(("O:destroyed: wsTheme\n"));
}

int wsTheme::getThemeTagID(const string *tag)
{
  if(strcmp(tag->c_str(),"name") == 0)
    return TAG_NAME;
  else if(strcmp(tag->c_str(),"SubTag") == 0)
    return TAG_SUBTAG;
  else if(strcmp(tag->c_str(),"subtaggroup") == 0)
    return TAG_SUBTAGGROUP;
  else if(strcmp(tag->c_str(),"pageorder") == 0)
    return TAG_PAGEORDER;
  else if(strcmp(tag->c_str(),"command") == 0)
    return TAG_COMMAND;
  else if(strcmp(tag->c_str(),"options") == 0)
    return TAG_OPTIONS;
  else if(strcmp(tag->c_str(),"depend") == 0)
    return TAG_DEPEND;

  return UNKNOWN;
}

int wsTheme::load(ifstream *config, string db_file, int db_type)
{
  string buff;
  string tag;
  string value;
  int done = 0;
  int line = 0;

  while((!done) && (line=getNextConfigLine(config, &buff))!=0) {
    splitString(&tag, &value, &buff);

    if(tag.find("}") != string::npos) {
      done = 1;
    } else {
      switch(this->getThemeTagID(&tag)) {
      case TAG_NAME :
        chopQuotes(&value);
        this->setName(&value);
        break;
      case TAG_SUBTAG :
	if(db_file.length() != 0) {
	  this->pushBackSubTag(&value, pagegroup->getSubTagTagFront(),
			       pagegroup->getSubTagTagBack(),
			       this->getDatabaseKey(), db_file, db_type);
	} else {
	  this->pushBackSubTag(&value, pagegroup->getSubTagTagFront(),
			       pagegroup->getSubTagTagBack());
	}
        break;
      case TAG_SUBTAGGROUP :
        chopQuotes(&value);
        if(!subtaggroup_list->pushBackSubTagGroup
	   (pagegroup->findSubTagGroupWithName(value))) {
	  __wsmake_print_error("Error adding subtaggroup `%s' (line %d)\n",
			       value.c_str(),line);
        }
        break;
      case TAG_PAGEORDER :
        chopQuotes(&value);
        if(!pageorder_list->pushBackPageOrder
	   (pagegroup->findPageOrderWithName(value))) {
	  __wsmake_print_error("Error adding pageorder `%s' (line %d)\n",
			       value.c_str(),line);
        }
        break;
      case TAG_COMMAND :
        chopQuotes(&value);
        this->setCommand(value);
        break;
      case TAG_OPTIONS :
        chopQuotes(&value);
        this->setOptions(value);
        break;
      case TAG_DEPEND :
	chopQuotes(&value);
	if(!depend_list->pushBackDepend
           (wsDepend::findOrCreate(pagegroup->getConfigDir(), value,
				   pagegroup))) {
          __wsmake_print_error("Error adding depend `%s' (line %d)\n",
                               value.c_str(),line);
        }
        break;
      case UNKNOWN :
      default :
        __wsmake_print_error("unknown theme tag skipped (line %d): %s\n",
			     line,tag.c_str());
        return 0;
        break;
      }
    }
  }

  // Set the command and options from the pagegroup if the command length is 0
  if(command.length() == 0) {
    this->setCommand(pagegroup->getCommand());
    //    setOptions(pagegroup->options + " " + options);
    if(options.length() == 0) {
      this->setOptions(pagegroup->getOptions());
    } else {
      this->setOptions(options);
    }
  }

  return 1;
}

int wsTheme::sync(int force)
{
  DBGPRINT(("S: syncing wsTheme `%s'\n", name.c_str()));

  if(force) { updated = true; return 1; }

  if(!pagegroup->usingSourceTimestamp()) {
    depend_list->sync(force);
    subtag_list->sync(force);
    subtaggroup_list->sync(force);
    
    if(subtag_list->isUpdated()) {
      updated = true;
    } else if(subtaggroup_list->isUpdated()) {
      updated = true;
    } else if(pageorder_list->isUpdated()) {
      updated = true;
    } else if(depend_list->isUpdated()) {
      updated = true;
    }
  }

  return 1;
}

int wsTheme::make(void)
{
  DBGPRINT(("S: making wsTheme `%s'\n", name.c_str()));

  if(!depend_list->make()) return 0;

  return 1;
}

int wsTheme::parse(string *in)
{
  if(subtag_list->parse(in))
    return 1;

  if(subtaggroup_list->parse(in))
    return 1;

  return 0;
}

int wsTheme::setPageList(wsPageList *prepend, wsPageList *append)
{
  if(!pageorder_list->setPageList(prepend,append))
    return 0;

  return 1;
}

int wsTheme::pushBackSubTag(const string *subtag,
			    const string tagfront,
			    const string tagback)
{
  return subtag_list->pushBackSubTag(subtag, &tagfront, &tagback);
}

int wsTheme::pushBackSubTag(const string *value,
			    const string tagfront,
			    const string tagback,
			    const string db_key,
			    const string db_file, int db_type)
{
  return subtag_list->pushBackSubTag(value, tagfront, tagback,
				     db_key, db_file, db_type);
}

void wsTheme::print() const
{
  __wsmake_print(3,"  Theme\n");
  __wsmake_print(3,"  =============\n");
  __wsmake_print(3,"    Name      : %s\n", this->getName().c_str());
  __wsmake_print(3,"    Command    : %s\n", this->getCommand().c_str());
  __wsmake_print(3,"    Options    : %s\n", this->getOptions().c_str());
  __wsmake_print(3,"    SubTags   : ");
  if(__wsmake_get_level() >= 3) {
    subtag_list->print();
  }
  __wsmake_print(3,"\n");
  __wsmake_print(3,"  =============\n");
}

void wsTheme::printSync() const
{
  pageorder_list->printSyncs();
}

void wsTheme::printName() const
{
  __wsmake_print("%s", this->getName().c_str());
}
