/* $Id: wsSubTag.cpp,v 1.1 2001/05/30 13:38:56 mike Exp $
 ***********************************************************************
 *         libwsmake - Core functionality library of wsmake            *
 *           Copyright (C) 1999,2000,2001 Michael Brownlow             *
 *                                                                     *
 * This program is free software; you can redistribute it and/or modify*
 * it under the terms of the GNU General Public License as published by*
 * the Free Software Foundation; either version 2 of the License, or   *
 * (at your option) any later version.                                 *
 *                                                                     *
 * This program is distributed in the hope that it will be useful,     *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of      *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the       *
 * GNU General Public License for more details.                        *
 *                                                                     *
 * You should have received a copy of the GNU General Public License   *
 * along with this program; if not, write to the Free Software         *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.           *
 *                                                                     *
 * For questions and comments, please email the author at:             *
 * mike@wsmake.org                                                     *
 ***********************************************************************/
#include <cstdio>
#include <cstring>

#include <unistd.h>
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#include "wsSubTag.h"

using namespace std;

wsSubTag::wsSubTag()
  : wsDBObject(), command(0), update(false), split_buff(NULL),
    command_output("")
{
  *argv = NULL;

  DBGPRINT(("O:created: wsSubTag `%s'\n", name.c_str()));
}

wsSubTag::wsSubTag(const string *buff)
  : wsDBObject(), update(false), split_buff(NULL)
{
  *argv = NULL;

  this->load(buff);

  DBGPRINT(("O:created: wsSubTag `%s'\n", name.c_str()));
}

wsSubTag::wsSubTag(const string *buff, const string *tagfront,
		   const string *tagback)
  : wsDBObject(), update(false), split_buff(NULL)
{
  *argv = NULL;

  this->load(buff, tagfront, tagback);

  DBGPRINT(("O:created: wsSubTag `%s'\n", name.c_str()));
}

wsSubTag::wsSubTag
(const string *buff,
 const string *prefix,
 const string *db_file,
 int db_type)
  : wsDBObject(db_file->c_str(), db_type), update(false), split_buff(NULL)
{
  *argv = NULL;

  this->load(buff, prefix);

  if(db_type != DBA_DB_UNKNOWN) {
    this->setDatabaseKey(this->getDatabaseKey().c_str());
    this->setValueType(DBA_DT_STRING);
  }

  DBGPRINT(("O:created: wsSubTag `%s'\n", name.c_str()));
}

wsSubTag::wsSubTag
(const string *buff,
 const string *tagfront,
 const string *tagback,
 const string *prefix,
 const string *db_file,
 int db_type)
  : wsDBObject(db_file->c_str(), db_type), update(false), split_buff(NULL)
{
  *argv = NULL;

  this->load(buff, tagfront, tagback, prefix);

  if(db_type != DBA_DB_UNKNOWN) {
    this->setDatabaseKey(this->getDatabaseKey().c_str());
    this->setValueType(DBA_DT_STRING);
  }

  DBGPRINT(("O:created: wsSubTag `%s'\n", name.c_str()));
}

wsSubTag::wsSubTag
(const string *buff, const string *buff2)
  : wsDBObject(), update(false), split_buff(NULL)
{
  string tag;
  string value;

  tag = *buff;
  value = *buff2;

  this->setTag(tag);
  this->setValue(value);

  *argv = NULL;

  DBGPRINT(("O:created: wsSubTag `%s'\n", name.c_str()));
}

wsSubTag::~wsSubTag()
{
  if(split_buff != NULL) {
    free(split_buff);
  }

  DBGPRINT(("O:destroyed: wsSubTag `%s'\n", name.c_str()));
}

int wsSubTag::sync(int force)
{
  char *tmpval;

  if(!this->dbKeyExists(this->getDatabaseKey().c_str())) {
    DBGPRINT(("S: adding new wsSubTag to database `%s':`%s'\n",
	      this->tag.c_str(), this->value.c_str()));
    this->setDatabaseValue(value.c_str());
    update = true;
  } else {
    if((tmpval=(char *)this->getDatabaseValue(&tmpval))==NULL) {
      __wsmake_print_error("key value could not be found! (`%s')\n",
			   name.c_str());
    } else {
      if((force) || (this->value != tmpval)) {
	DBGPRINT(("S: wsSubTag updated `%s' -> `%s'\n",
		  tmpval, this->value.c_str()));
	this->setDatabaseValue(value.c_str());
	update = true;
      }
      free(tmpval);
    }
  }
  
  return 1;
}

void wsSubTag::setTag(string tag)
{
  chopQuotes(&tag);
  this->tag.assign(tag);
  this->setName(tag);
}

void wsSubTag::setValue(string value)
{
  if(this->isCommand(value)) {
    command = 1;
    command_output.assign("");
    chopBackQuotes(&value);
    if(split_buff != NULL) {
      free(split_buff);
    }
    split_buff = strdup(value.c_str());
    memset(argv, 0, sizeof(argv));
    this->splitCommand();
  } else {
    command = 0;
    chopQuotes(&value);
  }
  this->value.assign(value);
}

int wsSubTag::parse(string *in)
{
  char buf[1024];
  int filedes[2];
  int forkage;
  char *cmd = argv[0];

  if(command) {
    if(command_output == "") { /* Only run the external command once */

      if(pipe(filedes) == 0) {
	forkage = fork();
	if(forkage == -1) {
	  __wsmake_print_error("Couldn't fork! Quitting...\n");
	  exit(EXIT_FAILURE);
	}
	
	if(forkage == 0) {     /* Child  */
	  ::close(0);          /* close stdin */
	  ::close(1);          /* close stdout */
	  dup(filedes[0]);     /* read from the pipe */
	  dup(filedes[1]);     /* write to the pipe */
	  ::close(filedes[0]); /* close parent stdin */
	  ::close(filedes[1]); /* close parent stdout */
	  execvp(cmd, argv);   /* argv is populated in setValue */
	  exit(EXIT_SUCCESS);
	} else {               /* Parent */
	  ::close(filedes[1]); /* Not writing */
	  while(read(filedes[0], buf, sizeof(buf)) != 0) {
	    this->command_output+=buf;
	  }
	  ::close(filedes[0]);
	}
      }
    }
    return searchAndReplace(in, tag, command_output);
  } else {
    return searchAndReplace(in, tag, value);
  }
}

void wsSubTag::splitCommand()
{
  bool inquotes = false;
  bool indblquotes = false;
  bool lastws = false;      /* was last char whitespace */
  bool lastesc = false;     /* was last char escaped */
  char **ap = argv;         /* arg pointer */
  char *bp1 = split_buff;   /* buf arg pointer */
  char *bp2 = split_buff;   /* buf trav pointer */
  int  cnt = 0;

  while(*bp2 != '\0') {
    if(inquotes) {
      while((*bp2 != '\0')&&(*bp2 != '\'')) bp2++;
      if(*bp2 != '\0') { bp2++; }
      inquotes = false;
    } else if(indblquotes) {
      while((*bp2 != '\0')&&((*bp2 != '"')||(lastesc))) {
	if(*bp2 == '\\') {
	  lastesc = true;
	} else {
	  lastesc = false;
	}
	bp2++;
      }
      if(*bp2 != '\0') { bp2++; }
      indblquotes = false;
    } else {
      if((*bp2 == '\'')&&(!lastesc)) {
	inquotes = true;
	lastws = false;
	bp2++;
      } else if((*bp2 == '"')&&(!lastesc)) {
	indblquotes = true;
	lastws = false;
	bp2++;
      } else if(((*bp2 == ' ')||(*bp2 == '\t'))
		&&(!lastws)&&(cnt < MAX_ARGS)) {
	cnt++;
	*bp2++ = '\0';
	*ap++ = bp1;
	bp1 = bp2;
	lastws = true;
      } else {
	if(*bp2 == '\\') {
	  lastesc = true;
	} else {
	  lastesc = false;
	}
	bp2++;
	lastws = false;
      }
    }
  }

  *ap++ = bp1;
  *ap = NULL;
}

void wsSubTag::print(void) const
{
  __wsmake_print("%s:%s\n", tag.c_str(),value.c_str());
}

void wsSubTag::print(int debug) const
{
  __wsmake_print(debug, "%s:%s\n", tag.c_str(),value.c_str());
}

int wsSubTag::load(const string *buff)
{
  string tag;
  string value;

  splitString(&tag, &value, buff);

  this->setTag(tag);
  this->setValue(value);

  return 1;
}

int wsSubTag::load(const string *buff, const string *prefix)
{
  string tag;
  string value;

  this->prefix.assign(*prefix);
  splitString(&tag, &value, buff);

  this->setTag(tag);
  this->setValue(value);

  return 1;
}

int wsSubTag::load(const string *buff, const string *tagfront,
		   const string *tagback)
{
  string tag;
  string value;

  splitString(&tag, &value, buff);

  chopQuotes(&tag);
  this->setTag(*tagfront + tag + *tagback);
  this->setValue(value);

  return 1;
}

int wsSubTag::load(const string *buff, const string *tagfront,
		   const string *tagback, const string *prefix)
{
  string tag;
  string value;

  this->prefix.assign(*prefix);
  splitString(&tag, &value, buff);

  chopQuotes(&tag);
  this->setTag(*tagfront + tag + *tagback);
  this->setValue(value);

  return 1;
}

bool wsSubTag::isCommand(string value)
{
  unsigned stepper = 0;

  while((isspace(value[stepper])) &&  (stepper < value.length())) {
    stepper++;
  }

  if(value[stepper] == '`')
    return true;
      
  return false;
}
