/* $Id: wsDepend.cpp,v 1.2 2001/08/06 01:50:48 mike Exp $
 ***********************************************************************
 *         libwsmake - Core functionality library of wsmake            *
 *           Copyright (C) 1999,2000,2001 Michael Brownlow             *
 *                                                                     *
 * This program is free software; you can redistribute it and/or modify*
 * it under the terms of the GNU General Public License as published by*
 * the Free Software Foundation; either version 2 of the License, or   *
 * (at your option) any later version.                                 *
 *                                                                     *
 * This program is distributed in the hope that it will be useful,     *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of      *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the       *
 * GNU General Public License for more details.                        *
 *                                                                     *
 * You should have received a copy of the GNU General Public License   *
 * along with this program; if not, write to the Free Software         *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.           *
 *                                                                     *
 * For questions and comments, please email the author at:             *
 * mike@wsmake.org                                                     *
 ***********************************************************************/
#include <cerrno>
#include <sys/stat.h>

#include "wsDepend.h"

using namespace std;

wsDepend::wsDepend(string default_dir, string filename, wsPageGroup *pg)
  : ref_count(0), pagegroup(pg)
{
  this->setFilename(default_dir, filename);

  this->ref();

  if(!pagegroup->usingSourceTimestamp()) {    
    wsDBObject::load(pagegroup->getDatabaseFilename().c_str(),
		     pagegroup->getDatabaseFormat());
    this->setDatabaseKey(getDatabaseKey().c_str());
    this->setValueType(DBA_DT_LONG);
  }

  DBGPRINT(("O:created: wsDepend\n"));
}

wsDepend::~wsDepend()
{
  DBGPRINT(("O:destroyed: wsDepend\n"));
}

int wsDepend::sync(int force)
{
  long lastmod = 0;
  bool using_timestamp = pagegroup->usingSourceTimestamp();

  if(using_timestamp || this->ready()) {
    DBGPRINT(("S: syncing depend `%s'\n", this->getFilename().c_str()));
    
    // Does the page exist?
    if(!this->pageExists()) {
      DBGPRINT(("S: discard `%s' (no page)\n",
		this->getFilename().c_str()));
      __wsmake_print("warning: dependency doesn't exist:\n `%s'\n",
		     this->getFilename().c_str());
      updated = false;
      return 0;
    }

    if(!using_timestamp) {
      if(!this->dbKeyExists(this->getDatabaseKey().c_str())) {
	DBGPRINT(("S: update `%s' (no key)\n", this->getFilename().c_str()));
	// We set this to 0 now so that if the first "make" fails, it will
	// continue to try and make.
	this->setDatabaseKey(this->getDatabaseKey().c_str());
	this->setDatabaseValue(0l);
	updated = true;
	return 1;
      }
    }

    // Did they say to force an update?
    if(force) {
      DBGPRINT(("S: update (forced)\n"));
      updated = true;
      return 1;
    }

    if(!using_timestamp) {
      // Get lastmod now that we know a record exists for this page
      if(this->getDatabaseValue(&lastmod)==NULL) {
	__wsmake_print_error("key value could not be found! (`%s')\n",
			     this->getFilename().c_str());
	updated = false;
	return 0;
      }

      if(lastmod < this->getLastMod()) {
	DBGPRINT(("S: update `%s' (file updated) %ld < %ld\n",
		  this->getFilename().c_str(), lastmod,
		  this->getLastMod()));
	updated = true;
	return 1;
      }
    }
  } else {
    __wsmake_print_error("database isn't ready\n");
    updated = false;
    return 0;
  }

  updated = false;
  return 0;
}

void wsDepend::make(void)
{
  if(!pagegroup->usingSourceTimestamp()) {
    this->setDatabaseValue(this->getLastMod());
  }
}

void wsDepend::setFilename(string default_dir, string filename)
{
  if(filename[0] != '/') {
    this->filename.assign
      (collapsePath((default_dir+"/"+filename).c_str(), NULL)
       );
  } else {
    this->filename.assign(collapsePath(filename.c_str(), NULL));
  }
}

int wsDepend::pageExists(void)
{
  struct stat statbuf;
  int err;

  err = stat(this->getFilename().c_str(), &statbuf);

  if(err == -1) {
    if(errno == ENOENT) return 0;
  }

  return 1;
}

time_t wsDepend::getLastMod(void) const
{
  struct stat statbuf;
  int err;

  err = stat(this->getFilename().c_str(), &statbuf);

  if(!err)
    return statbuf.st_mtime;
  else
    return -1;
}

void wsDepend::print(void) const
{
  __wsmake_print("%s\n", this->getFilename().c_str());
}

void wsDepend::print(int debug) const
{
  __wsmake_print(debug, "%s\n", this->getFilename().c_str());
}

wsDepend *wsDepend::findOrCreate(string default_dir,
				 string filename,
				 wsPageGroup *pagegroup)
{
  wsDepend *d = NULL;
  string fname = filename;

  if(fname[0] != '/') {
    fname.assign(collapsePath((default_dir+"/"+fname).c_str(), NULL));
  } else {
    fname.assign(collapsePath(fname.c_str(), NULL));
  }

  if(wsDBObject::keyInCache((KEY_DEPEND + fname).c_str())) {
    d = (wsDepend *) wsDBObject::dbobWithKey((KEY_DEPEND + fname).c_str());
    d->ref();
    return d;
  } else {
    return new wsDepend(default_dir, filename, pagegroup);
  }
}
