/* $Id: wsDBObject.cpp,v 1.2 2001/09/04 01:11:40 mike Exp $
 ***********************************************************************
 *         libwsmake - Core functionality library of wsmake            *
 *           Copyright (C) 1999,2000,2001 Michael Brownlow             *
 *                                                                     *
 * This program is free software; you can redistribute it and/or modify*
 * it under the terms of the GNU General Public License as published by*
 * the Free Software Foundation; either version 2 of the License, or   *
 * (at your option) any later version.                                 *
 *                                                                     *
 * This program is distributed in the hope that it will be useful,     *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of      *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the       *
 * GNU General Public License for more details.                        *
 *                                                                     *
 * You should have received a copy of the GNU General Public License   *
 * along with this program; if not, write to the Free Software         *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.           *
 *                                                                     *
 * For questions and comments, please email the author at:             *
 * mike@wsmake.org                                                     *
 ***********************************************************************/
#include <cstdlib>

#include "wsDBObject.h"

using namespace std;

list< wsDBObject * > wsDBObject::cache;

wsDBObject::wsDBObject(void)
  : key(NULL), exists(0), dba(NULL)
{
  this->setValueType(DBA_DT_UNKNOWN);
  wsDBObject::cache.push_back(this);

  DBGPRINT(("O:created: wsDBObject\n"));
}

wsDBObject::wsDBObject(const char *filename, int db_type)
  : key(NULL), exists(0), dba(NULL)
{
  this->setValueType(DBA_DT_UNKNOWN);
  wsDBObject::cache.remove(this);
  if(!this->load(filename, db_type)) {
    __wsmake_print_error("could not load database `%s'\n", filename);
  }

  DBGPRINT(("O:created: wsDBObject `%s'\n", filename));
}

wsDBObject::~wsDBObject()
{
  if(this->key!=NULL) {
    free(this->key);
    this->key = NULL;
  }
  this->close();

  DBGPRINT(("O:destroyed: wsDBObject\n"));
}

int wsDBObject::load(const char *filename, int db_type)
{
  if(db_type == DBA_DB_UNKNOWN) {
    return 1;
  }

  if((dba = dba_open(dba, filename, db_type)) == NULL) {
    return 0;
  }

  return 1;
}

int wsDBObject::close()
{
  if(ready()) {
    dba_close(dba);
  }
  dba = NULL;
  if(this->key != NULL) {
    free(this->key);
    this->key = NULL;
  }
  return 1;
}

void wsDBObject::setDatabaseKey(const char *key)
{
  if(this->key != NULL) {
    free(this->key);
  }

  this->key = (char *)malloc(strlen(key)+1);
  memset(this->key,0,strlen(key)+1);
  strncpy(this->key,key,strlen(key));

  this->exists = this->dbKeyExists(this->key);
}

char *wsDBObject::getDatabaseKey(char *buf)
{
  buf = (char *)malloc(strlen(this->key)+1);
  memset(buf,0,strlen(key)+1);
  strncpy(buf,this->key,strlen(this->key));

  return buf;
}

void wsDBObject::setDatabaseValue(const char *value)
{
  if(this->ready()) {
    if(!this->exists) {
      dba_add(this->dba, DBA_DT_STRING, this->key, value);
      exists = this->dbKeyExists(this->key);
    } else {
      dba_set(this->dba, DBA_DT_STRING, this->key, value);
    }
    this->setValueType(DBA_DT_STRING);
    DBGPRINT(("got value (string): `%s'\n", value));
  } else {
    DBGPRINT(("database not ready!!! value not set\n"));
  }
}

void wsDBObject::setDatabaseValue(const long value)
{
  if(this->ready()) {
    if(!this->exists) {
      dba_add(this->dba, DBA_DT_LONG, this->key, value);
      exists = this->dbKeyExists(this->key);
    } else {
      dba_set(this->dba, DBA_DT_LONG, this->key, value);
    }
    this->setValueType(DBA_DT_LONG);
    DBGPRINT(("set value (long): `%ld'\n", value));
  } else {
    DBGPRINT(("database not ready!!! value not set\n"));
  }
}

void *wsDBObject::getDatabaseValue(void *buf)
{
  if(this->ready()) {
    // Make sure key is set; we don't worry about buf because it
    // crosses types.
    if((this->key!=NULL)&&(this->type!=DBA_DT_UNKNOWN)) {
      if(!this->dbKeyExists(this->key)) {
	__wsmake_print_error("can't get key yet, it isn't in database.\n");
	return NULL;
      }
      switch(this->type) {
      case DBA_DT_LONG :
	dba_get(this->dba, this->type, this->key, buf);
	DBGPRINT(("got value (long): `%ld'\n", *(long *)buf));
	break;
      case DBA_DT_STRING :
	buf = dba_get(this->dba, this->type, this->key, buf);
	DBGPRINT(("got value (string): `%s'\n", (char *)buf));
	break;
      }
      return buf;
    } else {
      __wsmake_print_error("can't get key yet, key is NULL or type is not "
			   "set. type was %d.\n", this->type);
    }
  } else {
    DBGPRINT(("database not ready!!! value not retrieved\n"));
  }

  return NULL;
}

int wsDBObject::dbKeyExists(const char *testkey)
{
  if(this->ready()) {
    return dba_exists(dba, testkey);
  } else {
    DBGPRINT(("database not ready!!! couldn't test existence\n"));
  }

  return 0;
}

int wsDBObject::keyInCache(const char *testkey)
{
  list< wsDBObject * >::const_iterator i;

  for(i=wsDBObject::cache.begin(); i!=wsDBObject::cache.end();
      i++) {
    if((*i)->key != NULL) {
      if(!strcmp((*i)->key, testkey)) {
	return 1;
      }
    }
  }

  return 0;
}

wsDBObject *wsDBObject::dbobWithKey(const char *testkey)
{
  list< wsDBObject * >::const_iterator i;

  for(i=wsDBObject::cache.begin(); i!=wsDBObject::cache.end(); i++) {
    if((*i)->key != NULL) {
      if(!strcmp((*i)->key, testkey)) {
	return *i;
      }
    }
  }

  return NULL;
}
