/***********************************************************************
 *                libdba - A database agent library                    *
 *             Copyright (C) 2000,2001 Michael Brownlow                *
 *                                                                     *
 * This program is free software; you can redistribute it and/or modify*
 * it under the terms of the GNU General Public License as published by*
 * the Free Software Foundation; either version 2 of the License, or   *
 * (at your option) any later version.                                 *
 *                                                                     *
 * This program is distributed in the hope that it will be useful,     *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of      *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the       *
 * GNU General Public License for more details.                        *
 *                                                                     *
 * You should have received a copy of the GNU General Public License   *
 * along with this program; if not, write to the Free Software         *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.           *
 *                                                                     *
 * For questions and comments, please email the author at:             *
 * mike@wsmake.org                                                     *
 ***********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include "dba_pool.h"
#include "dba_csv.h"
#include "dba_db.h"
#include "dba_list.h"

/* DBA Pool */
static DBA *__dba_p_start = NULL;
static DBA *__dba_p_end = NULL;
static int __dba_p_c = 0;
static int __dba_debugging = 0;

int __dba_init __P((int debugging))
{
  __dba_p_c = 0;

#ifdef HAVE_TARGET_CSV
  if(__dba_csv_init()!=DBA_INIT_SUCCEED) {
    __dba_print_error("error in CSV initialization");
    return DBA_INIT_FAIL;
  }
#endif

#ifdef HAVE_TARGET_DB
  if(__dba_db_init()!=DBA_INIT_SUCCEED) {
    __dba_print_error("error in DB initialization");
    return DBA_INIT_FAIL; /* 1 for DB failure */
  }
#endif

  __dba_debugging = debugging;

  return DBA_INIT_SUCCEED;
}

void __dba_exit __P(())
{
  DBA *a,*b;

  a = __dba_p_start;

  /* Go through pools and clean them up */
  while(a!=NULL) {
    __dba_print_debug("closing DB agent %#x\n", a);
    b = a->next;
    a->close(a);
    free(a);
    a = b;
  }

  __dba_print_debug("agent pool freed\n");

}

int __dba_add_db __P((DBA *dba))
{
  /* New database, let's add it */
  switch(dba->db_type) {
#ifdef HAVE_TARGET_CSV
  case DBA_DB_CSV :
    if(__dba_csv_register(dba)!=1) return 0;
    break;
#endif
#ifdef HAVE_TARGET_DB
  case DBA_DB_DB :
    if(__dba_db_register(dba)!=1) return 0;
    break;
#endif
  default :
    __dba_print_error("database type %s not supported\n",
                      __dba_db_string(dba->db_type));
  }

  if(!dba->open(dba)) {
    __dba_print_error("error opening database\n");
    return 0;
  }

  /* Add the database agent to the list */
  __dba_print_debug("adding agent to master list\n");
  __dba_l_add(&__dba_p_start,&__dba_p_end,dba);
  __dba_p_c++;

  return 1;
}

int __dba_remove_db __P((DBA *dba))
{
  __dba_print_debug("removing agent %#x from master list\n", dba);
  if(__dba_l_remove(&__dba_p_start,&__dba_p_end,dba) == DBA_SUCCEED) {
    __dba_p_c--;
    return DBA_SUCCEED;
  }

  return DBA_FAIL;
}

char *__dba_db_string(int db_type)
{
  switch(db_type) {
    case DBA_DB_CSV :
      return "DBA_DB_CSV (Comma Separated Values)";
      break;
    case DBA_DB_DB :
      return "DBA_DB_DB (Berkeley DB)";
      break;
    default :
      return "Unknown";
      break;
  }

  return NULL;
}

DBA *__dba_find __P((const char *filename))
{
  DBA *a;

  a = __dba_p_start;

  while(a!=NULL) {
    if(!strcmp(a->filename, filename)) {
      return a;
    }
    a = a->next;
  }

  return NULL;
}

void __dba_print_error __P((const char *format, ...))
{
  va_list ap;

  va_start(ap, format);
  __dba_print_stderr("error",format,&ap);
  va_end(ap);
}

void __dba_print_debug __P((const char *format, ...))
{
  va_list ap;

  if(__dba_debugging) {
    va_start(ap, format);
    __dba_print_stderr("debug",format,&ap);
    va_end(ap);
  }
}

void __dba_print_warning __P((const char *format, ...))
{
  va_list ap;

  va_start(ap, format);
  __dba_print_stderr("warning",format,&ap);
  va_end(ap);
}

void __dba_print_stderr
__P((const char *type, const char *format, va_list *ap))
{
  fprintf(stderr, "%s %s: ", PACKAGE, type);
  vfprintf(stderr, format, *ap);
}

void __dba_print_string_info
__P((const char *string_name, const char string[]))
{
  __dba_print_debug("%s: '%s' (size = %d) (char %d is x%x)\n", string_name,
		    string, strlen(string)+1, strlen(string)+1,
		    string[strlen(string)]);
}

void *__dba_realloc __P((void *ptr, size_t size))
{
  register void *value = realloc (ptr, size);
  if (value == 0) {
    __dba_print_error("virtual memory exhausted");
  }
  return value;
}

void *__dba_malloc __P((size_t size))
{
  register void *value = malloc (size);
  if (value == 0) {
    __dba_print_error("virtual memory exhausted");
  }
  return value;
}

void *__dba_free(void *item)
{
  if(item != NULL) {
    free(item);
    item = NULL;
  }
  return item;
}

