/*
	mysql C++ wrapper library

	Author: Roland Haenel <rh@ginster.net>

	This program is in the public domain.
	Distribute and use it freely.
*/

#include <unistd.h>
#include <stdarg.h>
#include <time.h>

#include "database.h"

Database::Database()
{
	connected = false;	// No connection yet
	strcpy(error, "No connection established");
}

Database::~Database()
{
	disconnect();		// Disconnect if connected to database
}

int Database::status()
{
	if (connected == false) return DB_CONNECTION_NONE;
	return DB_CONNECTION_OK;
}

int Database::init()
{
	return DB_COMMAND_OK;
}

char *Database::errorMessage()
{
	if (connected == false) return error;
	return mysql_error(&mysql);
}

int Database::connect(const char *host, char *port, const char *db, const char* user, const char* pass)
{
	char *unused = port;

/* ORIGINAL
	if (mysql_connect(&mysql, host, NULL, NULL) == NULL) {
		strcpy(error, "Connect to database failed");
		return DB_ERROR;
	}
*/
	if (mysql_connect(&mysql, host, user, pass) == NULL) {
                strcpy(error, "Connect to database failed");
                return DB_ERROR;
        }

	if (mysql_select_db(&mysql, db)) {
		mysql_close(&mysql);
		strcpy(error, "No such database");	
		return DB_ERROR;
	}

	unused++;			// Just to suppress a compiler warning
	connected = true;
	return DB_CONNECTION_OK;
}

void Database::disconnect()
{
	if (connected == false) return;
	mysql_close(&mysql);
	connected = false;
}

int Database::reset()
{
	return DB_COMMAND_OK;
}

DBResult *Database::exec(char *sqlFormat, ...)
{
	va_list ap;
	char sqlCommand[5000];

	va_start(ap, sqlFormat);
	vsnprintf(sqlCommand, 5000, sqlFormat, ap);
	va_end(ap);

	return new DBResult(&mysql, sqlCommand);
}

void Database::exec(DBResult *res, char *sqlFormat, ...)
{
	va_list ap;
	char sqlCommand[5000];

	va_start(ap, sqlFormat);
	vsnprintf(sqlCommand, 5000, sqlFormat, ap);
	va_end(ap);

	res->init(&mysql, sqlCommand);
}

// ------------------- Database result implementation ------------------

DBResult::DBResult()
{
	result     = NULL;
	haveError  = false;
}

DBResult::DBResult(MYSQL *mysql, char *query)
{
	result     = NULL;
	haveError  = false;
	init(mysql, query);
}

DBResult::~DBResult()
{
	if (result != NULL)			// Free memory resources
		mysql_free_result(result);
}

void DBResult::init(MYSQL *mysql, char *query)
{
	if (result != NULL) {
		mysql_free_result(result);
		result = NULL;
	}
	if (mysql_query(mysql, query) == 0) {		// query OK
		result = mysql_store_result(mysql);
		if (result == NULL) {			// empty query
			if (mysql_num_fields(mysql) == 0)
				haveError = false;
			else
				haveError = true;
		} else haveError  = false;
	} else haveError  = true;
}

int DBResult::status()
{
	if (haveError)      return DB_ERROR;
	if (result == NULL) return DB_COMMAND_OK;
	return DB_TUPLES_OK;
}

int DBResult::nrTuples()
{
	if (result == NULL) return 0;
	return mysql_num_rows(result);
}

int DBResult::nrFields()
{
	if (result == NULL) return 0;
	return mysql_num_fields(result);
}

char *DBResult::fieldName(int n)
{
	MYSQL_FIELD *field;

	if (result == NULL) return NULL;
	mysql_field_seek(result, n);
	field = mysql_fetch_field(result);
	if (field == NULL) return NULL;
	return field->name;
}

int DBResult::fieldSize(int n)
{
	MYSQL_FIELD *field;

	if (result == NULL) return 0;
	mysql_field_seek(result, n);
	field = mysql_fetch_field(result);
	if (field == NULL) return 0;
	return field->length;
}

int DBResult::fieldSize(char *name)
{
	int i;

	if (result == NULL) return 0;
	for (i = 0; i < nrFields(); i++)
		if (!strcmp(name, fieldName(i)))
			return fieldSize(i);
	return 0;
}

void DBResult::seekTuple(int tuple)
{
	if (result == NULL) return;
	mysql_data_seek(result, tuple);
}

char **DBResult::getTuple()
{
	MYSQL_ROW row;

	if (result == NULL) return NULL;
	row = mysql_fetch_row(result);
	return row;
}

char **DBResult::getTuple(int tuple)
{
	seekTuple(tuple);
	return getTuple();
}
