/* This for emacs -*-Mode: linux-c;-*- */
/*****************************************************************************
==============================================================================
    gasNews --- retrieves news from the net and shows it in your desktop
    Copyright (C) 1998-1999 Mauricio Julio Fernndez Pradier

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
==============================================================================
*****************************************************************************/

#ifdef HAVE_CONFIG_H
#    include <config.h>
#endif
#ifdef WITH_DMALLOC
#    include <dmalloc.h>
#endif

#include "Downloader.h"
#include "NewsList.h"
#include <stdio.h>
#include "Item.h"
#include <unistd.h>
#include <getopt.h>
#include "common.h"
#include "debug.h"
#include "gui.h"
#include "auto.h"
#include "ctypes.h"
#include "log.h"
#include "NewsTicker.h"

#include "util.h"
#include "NewsParsers.h"
#include "asNewsParser.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


int       usage();
int       copyright();

static void retrieve_bug_info(NewsList *);
static void retrieve_release_info(NewsList *);
static void retrieve_program_info(char *proxy);
static int parse_options (int argc, char *argv[] );
static void set_default_method();


/* if set will cause asNews to retrieve bug/release info from my site */
static int bug_info = 0;  
static int release_info = 0;

static int uses_gui = 0;
static int ticker = 1;

char *in_file= NULL;

Mode mode;
char *proxy = NULL;

DownMode downmode;

void 
set_default_method() 
{
 char *p = NULL;
 int i;
 parse_proxy_env(&p, &i);
 if (p) { 
	 downmode = MPROXY;
	 free(p);	
 }
 else downmode = MNETWORK;
}

int main(int argc, char *argv[])
{

 mode = FRESHMEAT;
 downmode = MNETWORK;
 /* set defaults */
 set_default_method();

 parse_options ( argc, argv );

 if ( ticker ) {
	 GNewsTicker *nt;
	 gtk_init(&argc, &argv);
	 
	 nt = new GNewsTicker("pop");
	 
	 gtk_widget_show ( GTK_WIDGET(nt->get_widget()) );
	 
	 gtk_main();
	 exit(0);
 }

 RetrieveInfo *reader;
 InfoSource *InS = NULL;
 
 if ( downmode == MFILE && !in_file ) {
	 usage();
	 exit(-1);
 }
 
 if ( bug_info || release_info ) {
	 retrieve_program_info( proxy );
	 exit(0);
 }
    
 switch(downmode) {
 case MPROXY:
	 /* downmode set to MPROXY if 
	    1. http_proxy environment var. set
	    2. used -p (--proxy) option
	 */
	 InS = create_netInfoSource(mode, proxy, 1, &downmode);
	 break;
 case MNETWORK:
	 /* if 
	    1. http_proxy not set && no -i nor -p option
	    2. explicitly specified with -n
	 */
	 InS = create_netInfoSource(mode, NULL, 0, &downmode);
	 break;
 case MFILE:
	 InS = new InfoSource ( in_file, NULL, NULL, 0, 0 );
	 break;
 };

 switch(downmode) {
 case MFILE:
	 reader = new FileRetrieve(InS);
	 break;
 case MPROXY:
	 reader = new NetProxyRetrieve(InS);
	 break;
 case MNETWORK:
	 reader = new NetRetrieve(InS);
	 break;
 default:
	 reader = new FileRetrieve(InS);
 }

 NewsParser *parser = create_NewsParser(mode, reader);
	 
 NewsList *li = parser->build_list();
	 
 GtkWidget *main_window;
 gtk_set_locale ();
 gtk_init (&argc, &argv);
 
 main_window = create_main_window ( parser->get_name() );
 display_news_list ( li );
 gtk_widget_show (main_window);
 
 gtk_main ();
	 
 delete li;
 delete parser;
 
 if(proxy)
	 free(proxy);

 if (in_file)
	 free(in_file);

}


int 
parse_options (int argc, char *argv[] )
{
 
 int option;
 struct option long_options[] =
 {
	 {"mode",      1, 0, 'm'},
	 {"input",     1, 0, 'i'},
	 {"help",      0, 0, 'h'},
	 {"proxy",     1, 0, 'p'},
	 {"verbose",   0, 0, 'v'},
	 {"network",   0, 0, 'n'},
	 {"copyright", 0, 0, 'c'},
	 {"version",   0, 0, 'V'},
	 {"bugs",      0, 0, 'B'},
	 {"release",   0, 0, 'R'},
	 {"gui",       0, 0, 'g'},
	 {"ticker",    0, 0, 'T'},
	 { 0 , 0, 0, 0 },
 };

 while ( 
	(option=getopt_long(argc, argv, "BRncgm:hTvi:p:V", 
			    long_options, NULL)) 
	!= EOF ) {
	 switch (option) {
	 case 'B':
		 bug_info = 1;
		 ticker = uses_gui = 0;
		 break;
	 case 'R':
		 release_info = 1;
		 ticker = uses_gui = 0;
		 break;
	 case 'c':
		 copyright();
		 exit(0);
	 case 'h':
		 usage();
		 exit(0);
		 break;
	 case 'T':
		 ticker = 1;
		 uses_gui = 0;
		 break;
	 case 'g':
		 ticker = 0;
		 uses_gui = 1;
		 break;
	 case 'v':
		 verbose++;
		 break;
	 case 'V':
		 fprintf ( stdout, "asNews v%s\n", VERSION);
		 exit(0);
		 break;
	 case 'p':
		 proxy = strdup (optarg);
		 downmode = MPROXY;
		 break;
	 case 'i':
		 if(in_file) free(in_file);
		 in_file = strdup ( optarg );
		 downmode = MFILE;
		 break;
	 case 'n':
		 downmode = MNETWORK;
		 break;
	 case 'm':
		 if ( which_news_server(optarg, &mode)== -1 ) {
			 log_error("Unknown news server: %s.", optarg);
			 exit(-1);
		 }
		 break;
	 }
 }

 if ( optind < argc ) {
	 if ( in_file ) free ( in_file );
	 in_file = strdup ( argv[optind++] );
	 downmode = MFILE;
	 
 }

 return 0;
}


int  copyright()
{
 fprintf(stdout, "gasNews v%s Copyright (c) 1998-1999 Maurice Fernndez\n\n", 
	 VERSION);
 fprintf(stdout,  
	 "\tThis program is free software; you can redistribute it and/or modify\n"
	 "\tit under the terms of the GNU General Public License as published by\n" 
	 "\tthe Free Software Foundation; either version 2 of the License, or\n" 
	 "\t(at your option) any later version.\n\n"

	 "\tThis program is distributed in the hope that it will be useful,\n"
	 "\tbut WITHOUT ANY WARRANTY; without even the implied warranty of\n"
	 "\tMERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
	 "\tGNU General Public License for more details.\n\n"
	    
	 "\tYou should have received a copy of the GNU General Public License\n"
	 "\talong with this program; if not, write to the Free Software\n"
	 "\tFoundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307\n\n");

 return 0;
}


int usage()
{
 fprintf(stdout, "gasNews %s Copyright (c) 1998-1999 Maurice Fernndez\n", 
	 VERSION);
 fprintf(stdout, 
	 "Usage: gasNews [OPTIONS] [OUTPUT PATH]\n");
 fprintf(stdout, 
	 "  Options: \n"
	 "    -B --bugs\t\tretrieve info about bugs in this version\n"
	 "    -c --copyright\tdisplay copyright information\n"
	 "    -g --gui\t\tshow the news in a window\n"
	 "    -h --help\t\tdisplay this help\n"
	 "    -i --input=FILE\tuse file FILE instead of connecting to the news server\n"
	 "    -m --mode=MODE\twhich news to download. MODE may be"
	 " freshmeat,\n\t\t\t linuxtoday, segfault, slashdot, kdenews, linuxapps\n"
	 "\t\t\t bbc, ceo, 7am, geeknews, linuxdev, icewalkers\n"
	 "\t\t\t appwatch, hotwired, mozilla\n"
	 "    -n --network\tconnect to the server instead of reading from a file\n"
	 "    -p --proxy=PROXY:PORT\tuse proxy PROXY, connecting to port PORT\n"
	 "    -R --release\tretrieve info about last release of asNews\n"
	 "    -T --ticker\t\tuse the experimental NewsTicker GUI\n"
	 "    -v --verbose\tgive more info and warnings (try -vv)\n"
	 "    -V --version\tgive version info\n\n"
	 "  Examples:\n"
	 "    gasNews -m slashdot --gui  (assumes http_proxy is set)\n"
	 "    gasNews -vvv -T\n\n"
);
 fprintf(stdout, 
	 "  Default options are:\n");
 if (!getenv("http_proxy"))
	 fprintf(stdout, 
		 "    get news from a file (filename must be specified)\n");
 else
	 fprintf(stdout, 
		 "    get news using proxy %s.\n", getenv("http_proxy") );
 fprintf(stdout, "\n"
	 "Report bugs to batsman@geocities.com\n");
 fprintf(stdout, "More info at http://www.geocities.com/SiliconValley/Vista/2075/new/asNews.html\n");
 return 0;
}


void 
retrieve_program_info(char *proxy)
{
 RetrieveInfo *reader;
 InfoSource *InS = NULL;
 char *proxy_name = NULL;
 int proxy_port = 0;
  
 fprintf(stdout, "gasNews %s Copyright (c) 1998-1999 Maurice Fernndez\n", 
	 VERSION);

 switch(downmode) {
 case MPROXY:
	 parse_proxy_env(&proxy_name, &proxy_port);
	 if (proxy) {
		 if(proxy_name) free(proxy_name);
		 get_proxy_name(proxy, &proxy_name, &proxy_port);
	 }
		 
	 fprintf (stderr, 
		  "Using proxy %s on port %d.\n", proxy_name, proxy_port);
	 InS = new InfoSource ( MYULTRAMODE, "www.geocities.com",
				proxy_name, 80, proxy_port);
	
	 break;
 case MFILE:
	 InS = new InfoSource ( in_file, NULL, NULL, 0, 0);
	 break;

 case MNETWORK:
 default:
	 InS = new InfoSource ( MYULTRAMODE, "www.geocities.com", 
				NULL, 80, 0);
	 break;
 };

 switch(downmode) {
 case MPROXY:
	 reader = new NetProxyRetrieve(InS);
	 break;
 case MFILE:
	 reader = new FileRetrieve(InS);
	 break;
 case MNETWORK:
 default:
	 reader = new NetRetrieve(InS);
	 break;
 }

 NewsParser *parser = new asNewsParser(reader);
  
 NewsList *list = parser->build_list();
 if (release_info)
	 retrieve_release_info(list);
 if (bug_info)
	 retrieve_bug_info(list);
  
 delete list;
 delete parser;

}

void 
retrieve_bug_info(NewsList *li)
{

 fprintf( stdout, "Summary of known bugs in asNews version %s\n\n", VERSION );
 for (Item *it = li->get_first(); it != NULL; it = li->get_next()) {
	 if( !strcmp(it->get_field("category"), "BUG") 
	     && strstr(it->get_field("topic"), VERSION ) ) {
		 fprintf(stdout, "Date: %s, Fixed in %s\nDescription: %s\n\n",
			 it->get_field("cdate"), 
			 it->get_field("link"), 
			 it->get_field("title"));
	 }    
 }
 
}


void 
retrieve_release_info(NewsList *li)
{
 int i = 0;
  
 fprintf( stdout, "Latest releases of asNews:\n\n"  );
 for (Item *it = li->get_first(); (it != NULL) && (i<3);it = li->get_next()) {
	 if( !strcmp(it->get_field("category"), "RELEASE") ) {
		 fprintf(stdout, "Date: %s, Version: %s\nDescription: %s\n\n",
			 it->get_field("cdate"), 
			 it->get_field("topic"), 
			 it->get_field("title"));
		 i++;
	 }    
 }

}
