/* This for emacs -*-Mode: linux-c;-*- */
/*****************************************************************************
==============================================================================
    asNews --- retrieves news from the net and shows them in your desktop
    Copyright (C) 1998-1999 Mauricio Julio Fernndez Pradier

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
==============================================================================
*****************************************************************************/

#ifdef HAVE_CONFIG_H
#    include <config.h>
#endif
#ifdef WITH_DMALLOC
#    include <dmalloc.h>
#endif

#include "common.h"
#include "Item.h"
#include "dotfile.h"
#include "readline.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "ssplit.h"
#include "log.h"

static char *skip_comments ( FILE *fp);
static GenericHash *read_methods (FILE *fp);
static GenericHash *read_tasks(FILE *fp);
static Item *read_config(FILE *fp);
static int skip_section ( FILE *fp);
static void free_array ( char *p[], int n );

void
free_array(char *p[], int n) {
 for (int i=0; i<n; i++) {
	 free( p[i] );
	 p[i] = NULL;
 }

}

GenericHash *read_methods ( FILE *fp )
{
 int nfields = 0;
 char *l;
 char *debugl;
 DMethod *method;
 GenericHash *hash = new GenericHash();
 char ERRORMSG[] = "Incorrect download method specification: %s in \n  \"%s\"."; 
 
 while ( (l = skip_comments(fp)) ) {
	 char *fields[4];

	 debugl = strdup(l);
	 nfields = ssplit ( l, " \t", fields, 4, 1, 1); 
	 /* ignore leading delims, multiple delims --> 1 delim */
	 
	 if ( !strcmp(fields[0], "}") ) {
		 free(l);
		 break;
	 }
	 if (nfields < 2) {
		 free(l);
		 free(debugl);
		 continue;
	 }
	 switch(nfields) {
	 case 2:
		 if ( strcmp (fields[1], "network") ) {
			 log_error_long( ERRORMSG, fields[1], debugl );
			 free(l);
			 free(debugl);
			 continue;
		 }
		 method = new DMethod ( fields[0] );
		 break;
	 case 3:
		 if ( strcmp (fields[1], "file") ) {
			 log_error_long( ERRORMSG, fields[1], debugl );
			 free(l);
			 free(debugl);
			 continue;
		 }
		 method = new DMethod ( fields[0], fields[2] );
		 break;
	 case 4: 
		 if ( strcmp (fields[1], "proxy") ) {
			 log_error_long ( ERRORMSG, fields[1], debugl );
			 free(l);
			 free(debugl);
			 continue;
		 }
		 method = new DMethod ( fields[0], fields[2], atoi(fields[3]));
		 break;
	 default:
		 free(l);
		 free(debugl);
		 continue;
	 }
	 
	 hash->insert_field( method->name, method );
	 /* a copy is stored in the hash, so we delete method */
	 delete method;
	 free(l); 
	 /* don't forget to free the line we read (has \0 to separate tokens)*/
	 free(debugl); /* and the whole line */
 }

 
 if (l == NULL) {
	 log_error( "Unexpected EOF while parsing section \"methods\".");
	 return hash;
 } 

 return hash;
}


GenericHash *
read_tasks(FILE *fp)
{
 GenericHash *hash;
 char *l;
 char *debugl;
 char *fields[4];
 int nfields;
 
 hash = new GenericHash();
 while ( (l = skip_comments(fp)) ) {
	 debugl = strdup(l);
	 nfields = ssplit ( l, " \t", fields, 4, 1, 1); 
	 if ( !strcmp(fields[0], "}") ) {
		 free(l);
		 free(debugl);
		 break;
	 }
	 if (nfields != 4) {
		 log_error_long ("Incorrect task specification in\n  \"%s\".", 
				 debugl);
		 free(debugl);
		 free(l);
		 continue;
	 }
	 Item *nit = new Item();
	 char key[strlen(fields[0]) + strlen(fields[1]) + 2];
	 sprintf ( key, "%s-%s", fields[1], fields[0] );
	 nit->insert_field ( "task", fields[0]);
	 nit->insert_field ( "news", fields[1]);
	 nit->insert_field ( "method", fields[2]);
	 nit->insert_field ( "path", fields[3]);
	 if ( hash->insert_field ( key, nit ) == -1)
		 log_error_long("Warning: task \"%s\" defined twice, "
			 "in\n  \"%s\".", key, debugl);
	 
	 free(debugl);
	 free(l);
	 delete nit;
 }
 
 if (l == NULL) {
	 log_error ( "Unexpected EOF while parsing section \"task\".");
	 return hash;
 }
 
 return hash;     
}


Item *
read_config(FILE *fp)
{
 Item *hash;
 char *l;
 char *fields[4];
 int nfields;
 
 hash = new Item();
 while ( (l = skip_comments(fp)) ) {
	 nfields = ssplit ( l, " \t", fields, 4, 1, 1); 
	 if ( !strcmp(fields[0], "}") ) {		 
		 free(l);
		 break;
	 }
	 hash->insert_field(fields[0], fields[1]);
	 free(l);
 }
 
 if (l == NULL) {
	 log_error ( "Unexpected EOF while parsing section \"task\".");
	 return hash;
 }
 
 return hash;     

}

/* 
 * Skips lines of fp until one non-comment line has '}' as 1st token
 * Returns: 
 *   0 --> never fails   
 */
int
skip_section(FILE *fp)
{
 
 char *l;

 while(1) {
	 char *cbrace;
	 l = skip_comments(fp);
	 if (!l)   break;
	 
	 cbrace = strtok( l, " \t" );
	 if (!strcmp(cbrace, "}")) {
		 free(l);
		 break;
	 }
	 free(l);
 }
 return 0;
 
}


char * 
skip_comments( FILE *fp ) 
{
 char *line;
 do {
	 line = ReadLine(fp);
	 if ( !line || ( (*line != '#') && (*line!='\0') ) )
		 break;
	 free(line);
 } while (1);
 return line;
}



char *
get_dotfile_name(int i)
{
 char *home = NULL;
 int len;
 char *dname = NULL;

 home = getenv("HOME");
 if (!home) return NULL;
 len = strlen(home) + strlen("/.asNews/asNewsrc.1") + 1;
 dname = (char *) malloc(len);
 if (!dname) return NULL;

 strcpy(dname, home);
 char bu[strlen("/.asNews/asNewsrc.2")+1];
 sprintf ( bu, "/.asNews/asNewsrc.%d", i);
 strcat(dname, bu);
 return dname;
}


char *
get_dotfile_name()
{
 char *home = NULL;
 int len;
 char *dname = NULL;

 home = getenv("HOME");
 if (!home) return NULL;
 len = strlen(home) + strlen("/.asNews/asNewsrc") + 1;
 dname = (char *) malloc(len);
 if (!dname) return NULL;

 strcpy(dname, home);
 strcat(dname, "/.asNews/asNewsrc");
 return dname;
}


DotFile *
read_dotfile(char *fname)
{
 
 FILE *fp;
 char *ptr  = NULL;
 char *cbrace = NULL;
 char *line = NULL;
 char *configfile = NULL;
 

 GenericHash *methods = NULL;
 GenericHash *tasks = NULL;
 Item    *config = NULL;

 if (!fname) {
	 configfile = get_dotfile_name();
	 log_msg ( "Using default configuration file: %s.", configfile);
 }
 else
	 configfile = strdup(fname);

	 
 if (verbose) 
	 log_msg ("Reading configuration file %s.", 
		  configfile );	
 
 if ((fp = fopen(configfile, "r")) == NULL) {
	 log_error( "Configuration file \"%s\" not found.", configfile);
	 free(configfile);
	 exit(-1);
 }
 free(configfile);
 
 char ERRORMSG[] = "Section \"%s\" not found in configuration file.";
 do {
	 line = skip_comments(fp);

	 if (line == NULL) {
		 fclose(fp); 
		 if (!tasks || !methods) {
			 if (methods) delete methods;
			 else log_error( ERRORMSG, "methods");
			 if (tasks) delete tasks;
			 else log_error( ERRORMSG, "tasks");
			 return NULL;
		 }
		 DotFile *ret = new DotFile();
		 ret->dmethods = methods;
		 ret->tasks = tasks;
		 ret->config = config;
		 return ret;
	 } 
	 
	 /* get a section name */
	 ptr = strtok ( line, " \t" );
	 cbrace = strtok (NULL, " \t");
	 /* check if this starts a section ('{' on the same line) */
	 if (strcmp(cbrace, "{")) {
		 log_error ( "Error while reading the configuration file: " 
			     "'{' expected after %s.", line );
		 free(line);
		 continue; /* ignore line */
	 }
	 
	 if (!strcmp(ptr, "methods")) {
		 if (methods) delete methods;
		 methods = read_methods (fp);		 
	 }
	 else if (!strcmp(ptr, "tasks")) {
		 if(tasks) delete tasks;
		 tasks = read_tasks (fp);
	 }
	 else if (!strcmp(ptr, "config")) {
		 if(config) delete config;
		 config = read_config (fp);
	 }
	 else {	
		 /* section name not recognized */
		 skip_section(fp);
	 } 
 
	 free(line);
 } while(1); 
 
 return NULL; /* will never get here */
}


DMethod::DMethod ( char *name, char *file )
{
  downmode = MFILE;
  this->name = strdup(name);
  file_name = strdup(file);
  host_name = NULL;
  port = 0;
}

DMethod::DMethod ( char *name, char *host, int port )
{
  this->name = strdup(name);
  host_name  = strdup(host);
  this->port = port;
  downmode = MPROXY;
  file_name = NULL;
}

DMethod::DMethod ( char *name )
{
 this->name = strdup (name);
 host_name = NULL;
 port = 0;
 downmode = MNETWORK;
 file_name = NULL;
}

DMethod::DMethod ( const DMethod &o )
{
  name = strdup(o.name);
  downmode = o.downmode;
  port = o.port;
  if ( o.file_name ) 
    file_name = strdup(o.file_name);
  if ( o.host_name )
    host_name = strdup(o.host_name);
  
}



DMethod::DMethod ( const DMethod *o )
{
  name = strdup(o->name);
  downmode = o->downmode;
  port = o->port;
  if ( o->file_name ) 
    file_name = strdup(o->file_name);
  else
	  file_name = NULL;
  if ( o->host_name )
    host_name = strdup(o->host_name);
  else
	  host_name = NULL;
}


void
DMethod::destroy() 
{
  if (file_name)
    free(file_name);
  if (host_name)
    free(host_name);
  free(name);

}
