
#ifdef HAVE_CONFIG_H
#    include <config.h>
#endif
#ifdef WITH_DMALLOC
#    include <dmalloc.h>
#endif

#include "common.h"
#include "auto.h"
#include "dotfile.h"
#include "Item.h"
#include <stdlib.h>
#include "Downloader.h"
#include <string.h>
#include "ctypes.h"
#include "log.h"

#include "NewsParsers.h"

static GenericHash *dmethods;
static GenericHash *tasks;
static Item *config;
static void get_period();
static void handle_tasks();
static void run_task ( gpointer key, gpointer value, gpointer data );



int 
read_configuration()
{
 char *field;
 if (!config) {
	 log_error ( "Section \"config\" not found.");
	 return -1;
 }
 get_period();
 
 field = config->get_field("verbose");
 if ( field ) {
	 verbose = atoi(field);
 }
 field = config->get_field("error-log");
 if ( field ) {
	 if (stderr_log) 
		 free(stderr_log);
	 stderr_log = strdup(field);
 }
 field = config->get_field("log");
 if ( field ) {
	 if ( stdout_log )
		 free ( stdout_log );
	 stdout_log = strdup(field);
 }
 return 0;
}

void
get_period()
{
 if (!config) {
	 period = 60; 
	 return; 
 }
 char *cad = config->get_field("period");
 if (!cad) {
	 period = 60;
	 return;
 }
 int p = atoi(cad);
 period = (p>60)?p:60;
 
}


int
handle_dotfile(DotFile *dfile)
{
 if (!dfile) {
	config = NULL;
	tasks = NULL;
	dmethods = NULL;
	return -1;
 } 
 dmethods = dfile->dmethods;
 tasks = dfile->tasks;
 config = dfile->config;

 handle_tasks();
 return 0;
}

int 
init_dotfile ( DotFile *dfile )
{
 if (!dfile) {
	 config = NULL;
	 tasks = NULL;
	 dmethods = NULL;
	 return -1;
 } 

 
 dmethods = dfile->dmethods;
 tasks = dfile->tasks;
 config = dfile->config;
 return 0;
}


void handle_tasks()
{
 tasks->foreach( run_task, NULL );
 
}


void run_task ( gpointer key, gpointer value, gpointer data )
{
 log_msg ( "Handling task \"%s\".", (char *)key);
 Item *task = (Item *) value;
 char *ofile;
 char *type;
 char *method;
 char *news;
 int dump = 0;
 
 ofile = task->get_field("path");
 type = task->get_field("task");
 method = task->get_field("method");
 news = task->get_field("news");

 if ( verbose )
	 log_msg ( "Task: %s Method: %s News: %s Path: %s.",
		type, method, news, ofile );
	 
 InfoSource *InS;
 RetrieveInfo *reader;
 CreateFile *cfile;
 NewsParser *parser;
 
 InS = create_InfoSource( method, news );
 if (!InS) return;
 reader = create_RetrieveInfo ( method, InS );
 if (!reader) {
	 delete InS;
	 return;
 }
 dump = ( !strcmp(type, "dump") );
 if ( dump ) {
	 int status;
	 FILE *fp;
	 
	 status = reader->retrieve();
	 if ( status == -1 ) {
		 log_error ( "Error while retrieving news." );
		 delete reader; /* reader deletes InS when deleted  */
		 return;
	 }
	 
	 if ( (fp = fopen(ofile, "w")) == NULL ) { 
		 status = -1;
		 log_error ( "Could not open destination file %s.",
			     ofile );
		 delete reader;
		 return;
	 }
	 	 
	 char *d = reader->get_data();
	 if (d) 
		 fprintf ( fp, "%s", reader->get_data() );
	 fclose (fp);
	 delete reader;
 }
 else {
	 int status = 0;
	 cfile = create_CreateFile ( type );
	 if (!cfile) {
		 log_error ("Output type \"%s\" not supported.",type);
		 delete reader;
		 return;
	 }
	 parser = create_NewsParser ( news, reader );
	 if (!parser) {
		 delete cfile;
		 delete reader;
		 return;
	 }
	 NewsList *li = parser->build_list(); 
	 status = cfile->do_file ( ofile, li );
	 if (status == -1) 
		 log_error( "Could not create file(s) in %s.", ofile );
	 delete li;
	 delete parser; /* parser deletes reader, which deletes InS */
 }

}


InfoSource *
create_InfoSource ( char *mname, char *news)
{
 int proxy_port = 0;
 char *proxy_name = NULL;
 int isproxy = 0;
 InfoSource *InS = NULL;
 DMethod *method;
 Mode mode;

 method = (DMethod *) dmethods->get_field(mname);
 if (!method) {
	 log_error("Download method \"%s\" non existent.", mname );
	 return NULL;
 }
 isproxy = (method->downmode == MPROXY);
 if (isproxy) {
	 proxy_port = method->port;
	 proxy_name = method->host_name;
 }
 if ( method->downmode == MFILE ) {
	 InS = new InfoSource ( method->file_name, NULL, NULL, 0, 0);
	 return InS;
 }
 mode = find_Mode(news);
 if (mode == -1) {
	 log_error ( "News server \"%s\" not supported.", news );
	 return NULL;
 }

 if (isproxy) {
	 DownMode ign;
	 char proxy[strlen(proxy_name)+20];
	 sprintf(proxy, "%s:%d", proxy_name, proxy_port);
	 InS = create_netInfoSource(mode, proxy, 0, &ign);
	 return InS;
 }
 /* direct download */
 DownMode ign;
 InS = create_netInfoSource(mode, NULL, 0, &ign);
 
 return InS;
 

}


RetrieveInfo *
create_RetrieveInfo ( char *mname, InfoSource *InS)
{
 DMethod *method = NULL;
 method = (DMethod *)dmethods->get_field(mname);
 if (!method) return NULL;
 RetrieveInfo *reader = NULL;

 switch(method->downmode) {
 case MFILE:
	 reader = new FileRetrieve(InS);
	 break;
 case MPROXY:
	 reader = new NetProxyRetrieve(InS);
	 break;
 case MNETWORK:
	 reader = new NetRetrieve(InS);
	 break;
 }
 
 return reader;
}

CreateFile *
create_CreateFile (char *type)
{
 CreateFile *cfile = NULL;
 if (!strcmp(type, "kde"))
	 cfile = new KDEFile();
 if (!strcmp(type, "afterstep"))
	 cfile = new ASFile();
 
 return cfile;
 
}



NewsParser *
create_NewsParser ( char *news, RetrieveInfo *r)
{

 Mode mode = find_Mode(news);
 return create_NewsParser(mode, r);

}
