/* This for emacs -*-Mode: linux-c;-*- */
/*****************************************************************************
==============================================================================
    asNews --- retrieves news from the net and shows them in your desktop
    Copyright (C) 1998-1999 Mauricio Julio Fernndez Pradier

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
==============================================================================
*****************************************************************************/

#ifdef HAVE_CONFIG_H
#    include <config.h>
#endif
#ifdef WITH_DMALLOC
#    include <dmalloc.h>
#endif

#include "Downloader.h"
#include "NewsList.h"
#include "stdio.h"
#include "CreateFile.h"
#include "Item.h"
#include <unistd.h>
#include <getopt.h>
#include "common.h"
#include "debug.h"
#include "auto.h"
#include "ctypes.h"
#include "daemon.h"
#include "log.h"

#include "util.h"
#include "NewsParsers.h"
#include "asNewsParser.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


int       usage();
int       copyright();

static void retrieve_bug_info(NewsList *);
static void retrieve_release_info(NewsList *);
static void retrieve_program_info(char *proxy);
static int parse_options (int argc, char *argv[] );
static void enter_daemon_mode();
static void set_default_method();


/* if set will cause asNews to retrieve bug/release info from my site */
static int bug_info = 0;  
static int release_info = 0;

//static int uses_gui = 0;
static int dump_mode = 0;
static int normal_mode = 1;


char *in_file= NULL;
char *out_dir = NULL;

Mode mode;
char *proxy = NULL;

DownMode downmode;
OutputType out_type;


void 
set_default_method() 
{
 char *p = NULL;
 int i;
 parse_proxy_env(&p, &i);
 if (p) { 
	 downmode = MPROXY;
	 free(p);	
 }
 else downmode = MNETWORK;
}

int main(int argc, char *argv[])
{

 mode = FRESHMEAT;
 downmode = MNETWORK;
 out_type = AS;
 /* set defaults */
 set_default_method();

 parse_options ( argc, argv );

 if (daemon_mode) {
	 enter_daemon_mode();
	 exit(0);
 }

 RetrieveInfo *reader;
 InfoSource *InS = NULL;
 
 if ( downmode == MFILE && !in_file ) {
	 usage();
	 exit(-1);
 }
 
 if ( bug_info || release_info ) {
	 retrieve_program_info( proxy );
	 exit(0);
 }
 
 if ( !out_dir && !dump_mode) {
	 usage();
	 exit (-1);
 }
    
 switch(downmode) {
 case MPROXY:
	 /* downmode set to MPROXY if 
	    1. http_proxy environment var. set
	    2. used -p (--proxy) option
	 */
	 InS = create_netInfoSource(mode, proxy, 1, &downmode);
	 break;
 case MNETWORK:
	 /* if 
	    1. http_proxy not set && no -i nor -p option
	    2. explicitly specified with -n
	 */
	 InS = create_netInfoSource(mode, NULL, 0, &downmode);
	 break;
 case MFILE:
	 InS = new InfoSource ( in_file, NULL, NULL, 0, 0 );
	 break;
 };

 switch(downmode) {
 case MFILE:
	 reader = new FileRetrieve(InS);
	 break;
 case MPROXY:
	 reader = new NetProxyRetrieve(InS);
	 break;
 case MNETWORK:
	 reader = new NetRetrieve(InS);
	 break;
 default:
	 reader = new FileRetrieve(InS);
 }

 if ( dump_mode ) {
	 int status;

	 status = reader->retrieve();
	 if ( status == -1 ) {
		 log_error ( "Error while retrieving news." );
		 exit(-1);
	 }
	 fprintf ( stdout, reader->get_data() );
 }

 CreateFile *cfile = NULL;

 if ( normal_mode ) {
	 switch ( out_type ) {
	 case AS:
		 cfile = new ASFile();
		 break;
	 case KDE:
		 cfile = new KDEFile();
		 break;
	 }
 }
 if ( normal_mode ) {
	 NewsParser *parser = create_NewsParser(mode, reader);
	 
	 NewsList *li = parser->build_list();
	 
	 cfile->do_file ( out_dir, li );

	 delete li;
	 delete parser;
 }

 if(proxy)
	 free(proxy);
 if (cfile)
	 delete cfile;
 if ( dump_mode )
	 delete reader;
 if (in_file)
	 free(in_file);
 if (out_dir) 
	 free(out_dir);
}


void 
enter_daemon_mode()
{
 
 DotFile *dfile = NULL;
 
 dfile = read_dotfile( dotfile_name );
 if (dfile == NULL) {
	 fprintf ( stderr, 
		   "Error while reading the configuration file.\n");
	 exit(-1);
 }
 init_dotfile(dfile);
 if ( read_configuration() == -1 ) {
	 fprintf ( stderr,  "Could not find section \"config\".\n");
	 exit(-1);
 }
 daemonize(0, 1);	
 log_mode(1);
 freopen("/dev/null", "r", stdin); 
 if (stdout_log) {
	 if ( freopen( stdout_log, "a", stdout) == NULL ) 
		 freopen( "/dev/null", "a", stdout );
 }
 else
	 freopen ( "/dev/null", "a", stdout );
 if ( stderr_log ) {
	 if ( freopen( stderr_log, "a", stderr) == NULL ) 
		 freopen( "/dev/null", "a", stderr );
 }
 else 
	 freopen ( "/dev/null", "a", stderr );

 /*  this way we can set daemon_mode = 0 from GDB  */
 /*  or from a signal handler, so everything gets cleaned OK  */
 /*   if asNews is killed */
 while ( daemon_mode ) {
	 handle_dotfile(dfile);
	 delete dfile;
	 for (int num=0; num<10; num++) {
		 char *fnam = get_dotfile_name(num);
		 if (!fnam) continue;
		 int fd = open ( fnam, O_RDONLY );
		 if (fd == -1) {
			 free(fnam);
			 continue;
		 }
		 close(fd);
		 log_msg ( "Reading configuration file %s.", fnam); 
		 if ((dfile = read_dotfile( fnam ))== NULL) {
			 log_error( "Error while reading configuration file %s.",
				    fnam );
			 free(fnam);
			 continue;
		 }
		 handle_dotfile(dfile);
		 delete dfile;
		 free(fnam);
	 }
	 log_msg ( "Sleeping... period: %d.", period );
	 fflush(stdout);
	 fflush(stderr);
	 sleep(period*60);
	 log_msg ( "Reading configuration file %s.", 
		   dotfile_name?dotfile_name:"~/.asNews/asNewsrc" );
	 if ( (dfile = read_dotfile( dotfile_name )) == NULL ) {
		 log_error( "Error while reading main configuration file.");
		 exit(-1);
	 }
	 init_dotfile (dfile);
	 if ( read_configuration() == -1 ) {
		 log_error ("Could not set program configuration.");
		 exit(-1);
	 }
 }
}


int 
parse_options (int argc, char *argv[] )
{
 
 int option;
 struct option long_options[] =
 {
	 {"mode",      1, 0, 'm'},
	 {"input",     1, 0, 'i'},
	 {"output",    1, 0, 'o'},
	 {"help",      0, 0, 'h'},
	 {"proxy",     1, 0, 'p'},
	 {"verbose",   0, 0, 'v'},
	 {"network",   0, 0, 'n'},
	 {"copyright", 0, 0, 'c'},
	 {"type",      1, 0, 't'},
	 {"version",   0, 0, 'V'},
	 {"bugs",      0, 0, 'B'},
	 {"release",   0, 0, 'R'},
	 {"dump",      0, 0, 'd'},
	 {"daemon",    2, 0, 'D'},
	 { 0 , 0, 0, 0 },
 };

 while ( 
	(option=getopt_long(argc, argv, "BRncD::dm:ht:vi:p:o:V", 
			    long_options, NULL)) 
	!= EOF ) {
	 switch (option) {
	 case 'B':
		 bug_info = 1;
		 break;
	 case 'R':
		 release_info = 1;
		 break;
	 case 'd':
		 dump_mode = 1;
		 normal_mode = 0;
		 break;
	 case 'c':
		 copyright();
		 exit(0);
	 case 'h':
		 usage();
		 exit(0);
		 break;
	 case 'v':
		 verbose++;
		 break;
	 case 'V':
		 fprintf ( stdout, "asNews v%s\n", VERSION);
		 exit(0);
		 break;
	 case 'D':
		 if ( optarg ) {
			 if (dotfile_name) free(dotfile_name);
			 dotfile_name = strdup(optarg);
		 }
		 daemon_mode = 1;
		 break;
	 case 'p':
		 proxy = strdup (optarg);
		 downmode = MPROXY;
		 break;
	 case 'i':
		 if(in_file) free(in_file);
		 in_file = strdup ( optarg );
		 downmode = MFILE;
		 break;
	 case 'o':
		 if(out_dir) free(out_dir);
		 out_dir = strdup ( optarg );
		 break;
	 case 'n':
		 downmode = MNETWORK;
		 break;
	 case 't':
		 if ( !strcmp ( "kde", optarg) )
			 out_type = KDE;
		 if ( !strcmp ( "afterstep", optarg) )
			 out_type = AS;
		 break;
	 case 'm':
		 if ( which_news_server(optarg, &mode)== -1 ) {
			 log_error("Unknown news server: %s.", optarg);
			 exit(-1);
		 }
		 break;
	 }
 }

 if ( optind < argc ) {
	 if ( (optind - argc) >= 2 ) { 
		 if ( in_file ) free ( in_file );
		 in_file = strdup ( argv[optind++] );
		 downmode = MFILE;
		 if (argv[optind] != NULL) {
			 if ( out_dir ) free (out_dir);
			 out_dir = strdup ( argv[optind] );
		 }
	 }
	 else {
		 if (out_dir) free (out_dir);
		 out_dir = strdup ( argv[optind] );
	 }
 }

 return 0;
}


int  copyright()
{
 fprintf(stdout, "asNews v%s Copyright (c) 1998-1999 Maurice Fernndez\n\n", 
	 VERSION);
 fprintf(stdout,  
	 "\tThis program is free software; you can redistribute it and/or modify\n"
	 "\tit under the terms of the GNU General Public License as published by\n" 
	 "\tthe Free Software Foundation; either version 2 of the License, or\n" 
	 "\t(at your option) any later version.\n\n"

	 "\tThis program is distributed in the hope that it will be useful,\n"
	 "\tbut WITHOUT ANY WARRANTY; without even the implied warranty of\n"
	 "\tMERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
	 "\tGNU General Public License for more details.\n\n"
	    
	 "\tYou should have received a copy of the GNU General Public License\n"
	 "\talong with this program; if not, write to the Free Software\n"
	 "\tFoundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307\n\n");

 return 0;
}


int usage()
{
 fprintf(stdout, "asNews %s Copyright (c) 1998-1999 Maurice Fernndez\n", 
	 VERSION);
 fprintf(stdout, 
	 "Usage: asNews [OPTIONS] [OUTPUT PATH]\n");
 fprintf(stdout, 
	 "  Options: \n"
	 "    -B --bugs\t\tretrieve info about bugs in this version\n"
	 "    -c --copyright\tdisplay copyright information\n"
	 "    -D --daemon[=FILE]\tenter daemon mode\n"
	 "    -d --dump\t\tdump the news\n"
	 "    -h --help\t\tdisplay this help\n"
	 "    -i --input=FILE\tuse file FILE instead of connecting to the news server\n"
	 "    -m --mode=MODE\twhich news to download. MODE may be"
	 " freshmeat,\n\t\t\t linuxtoday, segfault, slashdot, kdenews, linuxapps\n"
	 "\t\t\t bbc, ceo, 7am, geeknews, linuxdev, icewalkers\n"
	 "\t\t\t appwatch, hotwired, mozilla\n"
	 "    -n --network\tconnect to the server instead of reading from a file\n"
	 "    -p --proxy=PROXY:PORT\tuse proxy PROXY, connecting to port PORT\n"
	 "    -o --output=PATH\tcreate output files in PATH\n"
	 "    -R --release\tretrieve info about last release of asNews\n"
	 "    -t --type=TYPE\tcreate output files for TYPE. \n" 
	 "\t\t\t TYPE may be kde or afterstep\n"
	 "    -v --verbose\tgive more info and warnings (try -vv)\n"
	 "    -V --version\tgive version info\n\n"
	 "  Examples:\n"
	 "    asNews -p localhost:8080 -m slashdot /tmp/tst.slashdot\n"
	 "    asNews -n -m segfault ~/GNUStep/AfterStep/Library/start/Segfault/\n"
	 "    asNews -BR -n \n"
	 "    asNews -n --dump\n"
	 "    asNews -D\n\n"
);
 fprintf(stdout, 
	 "  Default options are:\n"
	 "    --mode=freshmeat\n"
	 "    --type=afterstep\n");
 if (!getenv("http_proxy"))
	 fprintf(stdout, 
		 "    get news from a file (filename must be specified)\n");
 else
	 fprintf(stdout, 
		 "    get news using proxy %s.\n", getenv("http_proxy") );
 fprintf(stdout, "\n"
	 "Report bugs to batsman@geocities.com\n");
 fprintf(stdout, "More info at http://www.geocities.com/SiliconValley/Vista/2075/new/asNews.html\n");
 return 0;
}


void 
retrieve_program_info(char *proxy)
{
 RetrieveInfo *reader;
 InfoSource *InS = NULL;
 char *proxy_name = NULL;
 int proxy_port = 0;
  
 fprintf(stdout, "asNews %s Copyright (c) 1998-1999 Maurice Fernndez\n", 
	 VERSION);

 switch(downmode) {
 case MPROXY:
	 parse_proxy_env(&proxy_name, &proxy_port);
	 if (proxy) {
		 if(proxy_name) free(proxy_name);
		 get_proxy_name(proxy, &proxy_name, &proxy_port);
	 }
		 
	 fprintf (stderr, 
		  "Using proxy %s on port %d.\n", proxy_name, proxy_port);
	 InS = new InfoSource ( MYULTRAMODE, "www.geocities.com",
				proxy_name, 80, proxy_port);
	
	 break;
 case MFILE:
	 InS = new InfoSource ( in_file, NULL, NULL, 0, 0);
	 break;

 case MNETWORK:
 default:
	 InS = new InfoSource ( MYULTRAMODE, "www.geocities.com", 
				NULL, 80, 0);
	 break;
 };

 switch(downmode) {
 case MPROXY:
	 reader = new NetProxyRetrieve(InS);
	 break;
 case MFILE:
	 reader = new FileRetrieve(InS);
	 break;
 case MNETWORK:
 default:
	 reader = new NetRetrieve(InS);
	 break;
 }

 NewsParser *parser = new asNewsParser(reader);
  
 NewsList *list = parser->build_list();
 if (release_info)
	 retrieve_release_info(list);
 if (bug_info)
	 retrieve_bug_info(list);
  
 delete list;
 delete parser;

}

void 
retrieve_bug_info(NewsList *li)
{

 fprintf( stdout, "Summary of known bugs in asNews version %s\n\n", VERSION );
 for (Item *it = li->get_first(); it != NULL; it = li->get_next()) {
	 if( !strcmp(it->get_field("category"), "BUG") 
	     && strstr(it->get_field("topic"), VERSION ) ) {
		 fprintf(stdout, "Date: %s, Fixed in %s\nDescription: %s\n\n",
			 it->get_field("cdate"), 
			 it->get_field("link"), 
			 it->get_field("title"));
	 }    
 }
 
}


void 
retrieve_release_info(NewsList *li)
{
 int i = 0;
  
 fprintf( stdout, "Latest releases of asNews:\n\n"  );
 for (Item *it = li->get_first(); (it != NULL) && (i<3);it = li->get_next()) {
	 if( !strcmp(it->get_field("category"), "RELEASE") ) {
		 fprintf(stdout, "Date: %s, Version: %s\nDescription: %s\n\n",
			 it->get_field("cdate"), 
			 it->get_field("topic"), 
			 it->get_field("title"));
		 i++;
	 }    
 }

}
