
#ifdef HAVE_CONFIG_H
#    include <config.h>
#endif
#ifdef WITH_DMALLOC
#    include <dmalloc.h>
#endif

#include "Item.h"
#include <string.h>
#include <stdlib.h>
#include "debug.h"
#include <string.h>
#include <glib.h>
#include <stdio.h>

static gint  compare_func (gpointer, gpointer);

static void  destroyTwoStrings (gpointer, gpointer, gpointer);
static GHashTable *duplicate_hash ( GHashTable *hash );
static void  insert_to_hash ( gpointer, gpointer, gpointer );

static void  destroyCloneable (gpointer, gpointer, gpointer);
static GHashTable *duplicate_generic_hash ( GHashTable *hash );
static void  insert_to_generic_hash ( gpointer, gpointer, gpointer );
static void print_field( gpointer, gpointer, gpointer);
static void inc_counter( gpointer, gpointer, gpointer);
static void call_foreach( gpointer, gpointer, gpointer);
static void call_foreach_generic( gpointer, gpointer, gpointer);

void 
call_foreach( gpointer key, gpointer value, gpointer data)
{
  void **arg = (void **)data;
  void (*func)(gpointer, gpointer, gpointer);
  void *param;
  TwoStrings *hval = (TwoStrings *)value;
  func = ( void (*)(gpointer, gpointer, gpointer) )arg[0];
  param = arg[1];
  
  func ( key, hval->value, param );
}

void 
call_foreach_generic( gpointer key, gpointer value, gpointer data)
{
  void **arg = (void **)data;
  void (*func)(gpointer, gpointer, gpointer);
  void *param;
  HashValue *hval = (HashValue *)value;
  func = ( void (*)(gpointer, gpointer, gpointer) )arg[0];
  param = arg[1];
  
  func ( key, hval->value, param );
}


void
inc_counter ( gpointer key, gpointer value, gpointer data )
{
  int *num = (int *)data;
  (*num)++;
}


/* Will be used as a hash function given a char *; I've just invented it */
guint 
str_hash(gpointer data)
{
  char *pt; 
  int i;
  guint ret;
  guint sum;

  sum = 0;
  pt = (char *)data;
  ret = i = strlen ( pt );
  for (int k=0; k < i/2-1; k+=2) {
	  ret += (pt[k] * pt[k+1]) ^ pt[i-2*k-1] + pt[k] - pt[k+1];
	  sum += pt[k] - pt[k+1]  ^ pt[i-k-1];
  }
  ret += pt[i-2] + pt[i-1];
  sum += pt[i-2] - pt[i-1];
  ret ^= sum; 
  return ret % 1024;
}

void  
insert_to_hash ( gpointer key, gpointer value, gpointer hash)
{
 TwoStrings *hval = (TwoStrings *) value;
 TwoStrings *nh = (TwoStrings *) malloc(sizeof(TwoStrings));
 nh->key = strdup (hval->key);
 nh->value = strdup (hval->value);
 g_hash_table_insert ( (GHashTable *)hash, nh->key, nh );

}

GHashTable *duplicate_hash ( GHashTable *hash )
{
  GHashTable *h;

  h = g_hash_table_new ( str_hash, compare_func );
  g_hash_table_foreach ( hash, insert_to_hash, h );

  return h;
}


void 
destroyTwoStrings (gpointer key, 
	 gpointer value, gpointer data)
{
  TwoStrings *hval = (TwoStrings *)value;
  free(hval->key);
  free(hval->value);
  free(hval);
}


void 
print_field( gpointer key, gpointer value, gpointer data)
{
 TwoStrings *hval = (TwoStrings *)value;
 
 fprintf ( (FILE *)data, "Key: %s\t\tValue: %s\n", 
	   hval->key, hval->value );

}

void  
insert_to_generic_hash ( gpointer key, gpointer value, gpointer hash)
{
 HashValue *pt = new HashValue ((HashValue *) value);

 g_hash_table_insert ( (GHashTable *)hash, pt->key, pt );
 
}

GHashTable *duplicate_generic_hash ( GHashTable *hash )
{
  GHashTable *h;

  h = g_hash_table_new ( str_hash, compare_func );
  g_hash_table_foreach ( hash, insert_to_generic_hash, h );

  return h;
}

void 
destroyCloneable (gpointer key, 
		  gpointer value, gpointer data)
{
  Cloneable *c;
  
  c = (Cloneable *)value;
  delete c; 
}


/* returns 1 if keys equal (NOT -1 if a<b, 0 if a==b, 1 if a>b, as usual )*/
gint 
compare_func(gpointer a, gpointer b)
{
  char *p1, *p2;

  p1 = (char *) a;
  p2 = (char *) b;
  return !strcmp( p1, p2 );
}


/**************************************************************************
***************************************************************************
 
                          Item  METHODS

***************************************************************************
***************************************************************************/

void
Item::foreach ( void (*func)(gpointer, gpointer, gpointer), void *data )
{
  if (!func) return;
  void *arg[2];
  arg[0] = func;
  arg[1] = data;
  g_hash_table_foreach ( hash, call_foreach, arg );
}

int 
Item::count_fields()
{
  int num;
  num = 0;
  g_hash_table_foreach ( hash, inc_counter, &num );
  return num;
}

void 
Item::print(FILE *fd)
{
 g_hash_table_foreach( hash, print_field, (void *)fd );
}

char *
Item::get_field ( char *name )
{
  TwoStrings *hval;
  hval = (TwoStrings *)g_hash_table_lookup ( hash, name );
  if (!hval) return NULL;
  return hval->value;
}

int
Item::insert_field ( char *name, char *data )
{
 Cloneable *pt;
 pt = (Cloneable *)g_hash_table_lookup ( hash, name );
 if (pt) return -1;
 TwoStrings *hval = (TwoStrings *) malloc(sizeof(TwoStrings));
 hval->key = strdup( name );
 hval->value = strdup( data );
 g_hash_table_insert ( hash, hval->key, hval );
 return 0;
}


int 
Item::delete_field ( char *name )
{
  TwoStrings *pt;
  
  pt = (TwoStrings *) g_hash_table_lookup ( hash, name);
  if (pt) {
    g_hash_table_remove( hash, name );
    free(pt->key);
    free(pt->value);
    free(pt);
  }
  return 0;
}


Item::Item()
{ 
  hash = g_hash_table_new ( str_hash, compare_func );
}

Item::Item(const Item *ob)
{
  hash = duplicate_hash(ob->hash);
 
}

Item::Item(const Item &ob)
{
  hash = duplicate_hash(ob.hash);
}




Item::~Item()
{
  g_hash_table_foreach( hash, destroyTwoStrings, NULL );
  g_hash_table_destroy( hash );

}


/**************************************************************************
***************************************************************************
 
                          GenericHash  METHODS

***************************************************************************
***************************************************************************/


Cloneable *
GenericHash::get_field ( char *name )
{
 HashValue *hval = (HashValue *)g_hash_table_lookup ( hash, name );
 if (!hval) return NULL;
 return hval->value;
}

int
GenericHash::insert_field ( char *name, Cloneable *data )
{
 HashValue *hval;
 Cloneable *pt;
 pt = (Cloneable *)g_hash_table_lookup ( hash, name );
 if (pt) return -1;
 hval = new HashValue (name, data);
 g_hash_table_insert ( hash, hval->key, hval);
 return 0;
}


int 
GenericHash::delete_field ( char *name )
{
  Cloneable *pt;
  
  pt = (Cloneable *) g_hash_table_lookup ( hash, name);
  if (pt) {
	  g_hash_table_remove( hash, name );
	  delete pt;
  }
  return 0;
}



void
GenericHash::foreach ( void (*func)(gpointer, gpointer, gpointer), void *data )
{
  if (!func) return;

  void *arg[2];
  arg[0] = func;
  arg[1] = data;
  g_hash_table_foreach ( hash, call_foreach_generic, arg );
}

int 
GenericHash::count_fields()
{
  int num;
  num = 0;
  g_hash_table_foreach ( hash, inc_counter, &num );
  return num;
}


GenericHash::GenericHash()
{ 
  hash = g_hash_table_new ( str_hash, compare_func );
}

GenericHash::GenericHash(const GenericHash *ob)
{
  hash = duplicate_generic_hash(ob->hash);
 
}

GenericHash::GenericHash(const GenericHash &ob)
{
  hash = duplicate_generic_hash(ob.hash);
}

void
GenericHash::destroy()
{
  g_hash_table_foreach( hash, destroyCloneable, NULL );
  g_hash_table_destroy( hash );

}
