using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Data;
using System.IO;
using System.Xml;

namespace Minesweeper.NET
{
	public enum GameStatus { NOTSTARTED, INPROGRESS, WON, LOST }
	public enum TileValue
	{
			MINE, NOMINE, MINECLEARED, ONEADJMINE, TWOADJMINE, THREEADJMINE, FOURADJMINE,
		FIVEADJMINE, SIXADJMINE, SEVENADJMINE, EIGHTADJMINE 
	}

	public enum TileMask
	{
		CLEAR, FLAG, QUESTIONMARK
	}


	
	public delegate void TileClickedEventHandler(Object sender, MouseEventArgs e);
	public delegate void GameOverEventHandler(Object sender, EventArgs e);


	public class Tile
	{
		

		public const int TILESIZEX = 18;		// width of each tile
		public const int TILESIZEY  = 18;	// height of each tile
		
		
		private static Font fntButtonFont = null;
		private static Bitmap bmpUnselected = null,
			bmpDepressed = null,
			bmpFlag = null,
			bmpMine = null,
			bmpXMine = null,
			bmpDetonatedMine = null;

		private System.Windows.Forms.Label lblTile;		// the label control that graphically
														// represents the tile

		private TileValue iValue;		// represents the value of the tile.  Can be:
								//		MINE
								//		NOMIINE
								//		A value between 1 and 8, indicating how many
								//		mines are surrounding it
								//		MINECLEARED

		private TileMask iMask;		// represents the value of the tile.  Can be:
								//		FLAG
								//		QUESTIONMARK
								//		CLEAR
	
		private int iX, iY;		// the X and Y coodinates of the mine


		public Tile()
		{
			CreateFontsAndBitmaps();		// create the fonts and bitmaps, if needed

			// by default, create a NOMINE, CLEAR tile
			iValue = TileValue.NOMINE;
			iMask = TileMask.CLEAR;

			lblTile = new System.Windows.Forms.Label();			
			lblTile.Size = new System.Drawing.Size(TILESIZEX, TILESIZEY);					
			lblTile.Name = X.ToString() + "!" + Y.ToString();			
			lblTile.MouseDown += new MouseEventHandler(this.TileClicked);			
			lblTile.Font = fntButtonFont;
			lblTile.TextAlign = ContentAlignment.MiddleCenter;
			lblTile.BackgroundImage = bmpUnselected;
			lblTile.Text = "";
		}


		private void CreateFontsAndBitmaps()
		{
			// creates the fonts and bitmaps, if needed
			// creat the font, if needed
			if (fntButtonFont == null)
				fntButtonFont = new Font(FontFamily.GenericSansSerif, 9, FontStyle.Bold);

			/********* create the image bitmaps, if needed *******************/
			SolidBrush blackBrush = new SolidBrush(Color.Black);
			SolidBrush whiteBrush = new SolidBrush(Color.White);
			SolidBrush redBrush = new SolidBrush(Color.Red);
			
			if (bmpUnselected == null)
			{
				bmpUnselected = new Bitmap(TILESIZEX, TILESIZEY);
				Graphics imgTile = Graphics.FromImage(bmpUnselected);

				imgTile.FillRectangle(new SolidBrush(Color.FromArgb(220,220,220)),
					0, 0, TILESIZEX, TILESIZEY);
				imgTile.FillRectangle(whiteBrush, 0, 0, TILESIZEX, 2);
				imgTile.FillRectangle(whiteBrush, 0, 0, 2, TILESIZEY);
				imgTile.FillRectangle(new SolidBrush(Color.FromArgb(135,135,135)), TILESIZEX-2, 0, TILESIZEX, TILESIZEY);
				imgTile.FillRectangle(new SolidBrush(Color.FromArgb(135,135,135)), 0, TILESIZEY-2, TILESIZEX, TILESIZEY);
			}

			if (bmpDepressed == null)
			{
				bmpDepressed = new Bitmap(TILESIZEX, TILESIZEY);
				Graphics imgTileDepressed = Graphics.FromImage(bmpDepressed);

				imgTileDepressed.FillRectangle(new SolidBrush(Color.FromArgb(135,135,135)),
					0, 0, TILESIZEX, TILESIZEY);
				imgTileDepressed.FillRectangle(new SolidBrush(Color.FromArgb(190,190,190)),
					1, 1, TILESIZEX-1, TILESIZEY-1);
			}

			if (bmpFlag == null)
			{
				bmpFlag = new Bitmap(TILESIZEX, TILESIZEY);
				Graphics imgTileFlag = Graphics.FromImage(bmpFlag);

				imgTileFlag.FillRectangle(new SolidBrush(Color.FromArgb(220,220,220)),
					0, 0, TILESIZEX, TILESIZEY);
				imgTileFlag.FillRectangle(whiteBrush, 0, 0, TILESIZEX, 2);
				imgTileFlag.FillRectangle(whiteBrush, 0, 0, 2, TILESIZEY);
				imgTileFlag.FillRectangle(new SolidBrush(Color.FromArgb(135,135,135)), TILESIZEX-2, 0, TILESIZEX, TILESIZEY);
				imgTileFlag.FillRectangle(new SolidBrush(Color.FromArgb(135,135,135)), 0, TILESIZEY-2, TILESIZEX, TILESIZEY);

				
				imgTileFlag.FillRectangle(blackBrush, 4, TILESIZEY-5, TILESIZEX-8, 2);
				imgTileFlag.FillRectangle(blackBrush, 6, TILESIZEY-7, TILESIZEX-12, 2);
				imgTileFlag.FillRectangle(blackBrush, TILESIZEX / 2, 4, 2, TILESIZEY-8);

				Point t1 = new Point(4,6), t2 = new Point(TILESIZEX/2+1,3), t3 = new Point(TILESIZEX/2+1,9);
				Point[] trianglePoints = { t1, t2, t3 };				
				imgTileFlag.FillPolygon(redBrush, trianglePoints);
			}

			if (bmpMine == null)
			{
				bmpMine = new Bitmap(TILESIZEX, TILESIZEY);
				Graphics imgTileMine = Graphics.FromImage(bmpMine);

				imgTileMine.FillRectangle(new SolidBrush(Color.FromArgb(135,135,135)),
					0, 0, TILESIZEX, TILESIZEY);
				imgTileMine.FillRectangle(new SolidBrush(Color.FromArgb(190,190,190)),
					1, 1, TILESIZEX-1, TILESIZEY-1);
				imgTileMine.FillEllipse(blackBrush, 5, 5, TILESIZEX-10, TILESIZEY-10);
				imgTileMine.FillRectangle(blackBrush, 3, TILESIZEY / 2, TILESIZEX-6, 1);
				imgTileMine.FillRectangle(blackBrush, TILESIZEX/2, 3, 1, TILESIZEY-6);
				imgTileMine.DrawLine(new Pen(blackBrush, 1), new Point(4,4), new Point(TILESIZEX-4, TILESIZEY-4));
				imgTileMine.DrawLine(new Pen(blackBrush, 1), new Point(TILESIZEX-4,4), new Point(4, TILESIZEY-4));
			}

			if (bmpXMine == null)
			{
				bmpXMine = new Bitmap(TILESIZEX, TILESIZEY);
				Graphics imgTileXMine = Graphics.FromImage(bmpXMine);

				imgTileXMine.FillRectangle(new SolidBrush(Color.FromArgb(135,135,135)),
					0, 0, TILESIZEX, TILESIZEY);
				imgTileXMine.FillRectangle(new SolidBrush(Color.FromArgb(190,190,190)),
					1, 1, TILESIZEX-1, TILESIZEY-1);
				imgTileXMine.FillEllipse(blackBrush, 5, 5, TILESIZEX-10, TILESIZEY-10);
				imgTileXMine.FillRectangle(blackBrush, 3, TILESIZEY / 2, TILESIZEX-6, 1);
				imgTileXMine.FillRectangle(blackBrush, TILESIZEX/2, 3, 1, TILESIZEY-6);
				imgTileXMine.DrawLine(new Pen(blackBrush, 1), new Point(4,4), new Point(TILESIZEX-4, TILESIZEY-4));
				imgTileXMine.DrawLine(new Pen(blackBrush, 1), new Point(TILESIZEX-4,4), new Point(4, TILESIZEY-4));

				// draw the X
				imgTileXMine.DrawLine(new Pen(redBrush, 2), new Point(2,2), new Point(TILESIZEX-2, TILESIZEY-2));
				imgTileXMine.DrawLine(new Pen(redBrush, 2), new Point(TILESIZEX-2,2), new Point(2, TILESIZEY-2));
			}
			
			if (bmpDetonatedMine == null)
			{
				bmpDetonatedMine = new Bitmap(TILESIZEX, TILESIZEY);
				Graphics imgTileDetMine = Graphics.FromImage(bmpDetonatedMine);

				imgTileDetMine.FillRectangle(new SolidBrush(Color.FromArgb(135,135,135)),
					0, 0, TILESIZEX, TILESIZEY);
				imgTileDetMine.FillRectangle(redBrush, 1, 1, TILESIZEX-1, TILESIZEY-1);
				imgTileDetMine.FillEllipse(blackBrush, 5, 5, TILESIZEX-10, TILESIZEY-10);
				imgTileDetMine.FillRectangle(blackBrush, 3, TILESIZEY / 2, TILESIZEX-6, 1);
				imgTileDetMine.FillRectangle(blackBrush, TILESIZEX/2, 3, 1, TILESIZEY-6);
				imgTileDetMine.DrawLine(new Pen(blackBrush, 1), new Point(4,4), new Point(TILESIZEX-4, TILESIZEY-4));
				imgTileDetMine.DrawLine(new Pen(blackBrush, 1), new Point(TILESIZEX-4,4), new Point(4, TILESIZEY-4));
			}
			/*******************************************************************/
		}




		public void ResetTile()
		{
			lblTile.Location = new System.Drawing.Point(iX * TILESIZEX, iY * TILESIZEY + frmMain.HEADERHEIGHT);

			lblTile.BackgroundImage = bmpUnselected;
			lblTile.Text = "";
		}

		
		// create an event
		public event TileClickedEventHandler TileWasClicked;

		private void TileClicked(Object sender, MouseEventArgs e)
		{
			TileWasClicked(this, e);
		}


		public System.Windows.Forms.Label TileLabel 
		{
			get { return lblTile; }
		}

		public int X 
		{
			set 
			{
				iX = value;
			}
			get
			{
				return iX;
			}
		}

		public int Y 
		{
			set 
			{
				iY = value;
			}
			get
			{
				return iY;
			}
		}

		public TileValue Value
		{
			set 
			{
				iValue = value;
			}
			get 
			{
				return iValue;
			}
		}

		public TileMask Mask
		{
			set 
			{
				iMask = value;
			}
			get 
			{
				return iMask;
			}
		}

		public void ClearTile()
		{
			lblTile.BackgroundImage = bmpUnselected;
			lblTile.Text = "";
		}


		public void DisplayEndGameTile()
		{
			// this function displays tiles as they should appear at the end of a
			// Minesweeper game

			// if we have a correctly placed flag, do nothing
			if (Mask == TileMask.FLAG && Value == TileValue.MINE)
			{
				// do nothing
				return;
			}

			// if we have an INCORRECTLY placed flag, show the mine with
			// an X through it
			if (Mask == TileMask.FLAG && Value != TileValue.MINE)
			{
				lblTile.BackgroundImage = bmpXMine;
				lblTile.Text = "";
				return;
			}


			// if we have a mine that has not been covered, show the mine
			if (Mask == TileMask.CLEAR && Value == TileValue.MINE)
			{
				// we only want to show the mine if it is not the one that was clicked
				if (lblTile.BackgroundImage != bmpDetonatedMine)
				{
					lblTile.Text = "";
					lblTile.BackgroundImage = bmpMine;
				}
				return;
			}
		}



		public void DisplayTile()
		{
			// display the tile based upon its value and mask
			if (iMask != TileMask.CLEAR)
			{
				switch (iMask)
				{
					case TileMask.FLAG:
						lblTile.BackgroundImage = bmpFlag;
						lblTile.Text = "";
						break;

					case TileMask.QUESTIONMARK:
						lblTile.BackgroundImage = bmpUnselected;						
						lblTile.ForeColor = Color.Black;
						lblTile.Text = "?";
						break;
				}
			}
			else
			{
				switch (iValue)
				{
					case TileValue.MINE:
						// how to display a mine
						lblTile.BackgroundImage = bmpDetonatedMine;						
						break;

					case TileValue.NOMINE:
						lblTile.BackgroundImage = bmpUnselected;
						lblTile.Text = "";
						break;

					case TileValue.MINECLEARED:
						lblTile.BackgroundImage = bmpDepressed;
						break;


					case TileValue.ONEADJMINE:
						lblTile.BackgroundImage = bmpDepressed;
						lblTile.ForeColor = Color.Blue;
						lblTile.Text = "1";
						break;

					case TileValue.TWOADJMINE:
						lblTile.BackgroundImage = bmpDepressed;
						lblTile.ForeColor = Color.Green;
						lblTile.Text = "2";
						break;

					case TileValue.THREEADJMINE:
						lblTile.BackgroundImage = bmpDepressed;
						lblTile.ForeColor = Color.Red;
						lblTile.Text = "3";
						break;

					case TileValue.FOURADJMINE:
						lblTile.BackgroundImage = bmpDepressed;
						lblTile.ForeColor = Color.Black;
						lblTile.Text = "4";
						break;

					case TileValue.FIVEADJMINE:
						lblTile.BackgroundImage = bmpDepressed;
						lblTile.ForeColor = Color.DarkMagenta;
						lblTile.Text = "5";
						break;
				
					case TileValue.SIXADJMINE:
						lblTile.BackgroundImage = bmpDepressed;
						lblTile.ForeColor = Color.DarkOrange;
						lblTile.Text = "6";
						break;
				
					case TileValue.SEVENADJMINE:
						lblTile.BackgroundImage = bmpDepressed;
						lblTile.ForeColor = Color.Firebrick;
						lblTile.Text = "7";
						break;
				
					case TileValue.EIGHTADJMINE:
						lblTile.BackgroundImage = bmpDepressed;
						lblTile.ForeColor = Color.Brown;
						lblTile.Text = "8";
						break;
				}			
			}
		}
	}


	public class Board
	{
		// define constant COLUMNS/ROW/MINE counts
		public const int BEGINNERDEFAULTMINECOUNT = 10;
		public const int BEGINNERDEFAULTCOLUMNS = 9;
		public const int BEGINNERDEFAULTROWS = 9;

		public const int INTERMEDIATEDEFAULTMINECOUNT = 40;
		public const int INTERMEDIATEDEFAULTCOLUMNS = 16;
		public const int INTERMEDIATEDEFAULTROWS = 16;

		public const int EXPERTDEFAULTMINECOUNT = 99;
		public const int EXPERTDEFAULTCOLUMNS = 30;
		public const int EXPERTDEFAULTROWS = 16;

		public const int MAXCOLUMNS=30;
		public const int MAXROWS=30;
		public const int MAXMINECOUNT = 99;



		private Tile [] boardArray;		// the array of Tiles that make up the board
		private int iColumns, iRows;		// how many columns and rows in the board
		private int iMineCount;				// how many mines on the board
		private int iTotalMines;
		private GameStatus progress;	// what's the status of the game?

		private System.Windows.Forms.Label lblMineCount;	// the label for showing the mine count
		private System.Windows.Forms.Label lblTimer;	// the label for showing the time elapsed

		private System.Windows.Forms.Timer tmrSecondCount;	// a counter to count down the seconds


		public Board(int iMines, int iCols, int iRow)
		{
			iColumns = iCols;
			iRows = iRow;
			iTotalMines = iMineCount = iMines;
			progress = GameStatus.NOTSTARTED;

			BuildBoard();

			InstantiateControls();
		}

		public event GameOverEventHandler GameOver;


		public GameStatus Status
		{
			get 
			{
				return progress;
			}
		}


		private void InstantiateControls()
		{
			this.lblMineCount = new System.Windows.Forms.Label();
			this.lblTimer = new System.Windows.Forms.Label();
			this.tmrSecondCount = new System.Windows.Forms.Timer();


			// 
			// lblMineCount
			// 
			this.lblMineCount.BackColor = System.Drawing.Color.Black;
			this.lblMineCount.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
			this.lblMineCount.Font = new System.Drawing.Font("Courier New", 21F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.lblMineCount.ForeColor = System.Drawing.Color.Red;
			this.lblMineCount.Name = "lblMineCount";
			this.lblMineCount.Size = new System.Drawing.Size(50, 50);
			this.lblMineCount.TabIndex = 0;
			this.lblMineCount.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
			this.lblMineCount.Text = iMineCount.ToString();

			// 
			// lblTimer
			// 
			this.lblTimer.BackColor = System.Drawing.Color.Black;
			this.lblTimer.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
			this.lblTimer.Font = new System.Drawing.Font("Courier New", 15F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.lblTimer.ForeColor = System.Drawing.Color.Red;
			this.lblTimer.Name = "lblMineCount";
			this.lblTimer.Size = new System.Drawing.Size(50, 50);			
			this.lblTimer.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
			this.lblTimer.Location = new System.Drawing.Point(iColumns * Tile.TILESIZEX - lblTimer.Width, 0);
			this.lblTimer.Text = "0";

			// 
			// tmrSecondCount
			// 
			this.tmrSecondCount.Interval = 1000;
			this.tmrSecondCount.Enabled = false;		// don't start the timer just yet			
			this.tmrSecondCount.Tick += new EventHandler(this.TimerTickEvent);
		}


		private void TimerTickEvent(Object sender, EventArgs e)
		{
			lblTimer.Text = (Convert.ToInt32(lblTimer.Text) + 1).ToString();
		}


		public int Time
		{
			get { return Convert.ToInt32(this.lblTimer.Text); }
		}

		public int Columns 
		{
			get 
			{
				return iColumns;
			}
		}
		
		public int Rows
		{
			get 
			{
				return iRows;
			}
		}


		public void AttachTilesToForm(System.Windows.Forms.Form frmTemp)
		{
			// add the Tile labels and board controls to the frmMain form
			int BOARDSIZE = MAXROWS * MAXCOLUMNS;

			for (int iLoop=0; iLoop < BOARDSIZE; iLoop++)
				frmTemp.Controls.Add(boardArray[iLoop].TileLabel);

			frmTemp.Controls.Add(lblMineCount);
			frmTemp.Controls.Add(lblTimer);

			// show the board
			/*for (int iLoop=0; iLoop < iColumns*iRows; iLoop++) 
				if (boardArray[iLoop].Value == TileValue.MINE)
					boardArray[iLoop].DisplayTile();
					*/
		}


		
		private void BuildBoard()
		{
			int BOARDSIZE = MAXROWS * MAXCOLUMNS;
			int iLoop;
			
			// begin by instantiating the board array
			boardArray = new Tile[BOARDSIZE];
			for (iLoop=0; iLoop < BOARDSIZE; iLoop++)
			{
				boardArray[iLoop] = new Tile();
				boardArray[iLoop].TileWasClicked += new TileClickedEventHandler(this.TileClicked);
			}

			PopulateBoardArray();
		}


		public void RebuildBoard(int Mines, int Columns, int Rows)
		{
			iMineCount = iTotalMines = Mines;
			iColumns = Columns;
			iRows = Rows;

			progress = GameStatus.NOTSTARTED;		// reset the game progress

			// stop the time
			this.tmrSecondCount.Stop();
			this.tmrSecondCount.Enabled = false;

			this.lblTimer.Location = new System.Drawing.Point(iColumns * Tile.TILESIZEX - lblTimer.Width, 0);

			PopulateBoardArray();

			this.lblMineCount.Text = iMineCount.ToString();
			this.lblTimer.Text = "0";			
		}



		private void PopulateBoardArray()
		{
			int BOARDSIZE = iRows * iColumns;
			int iLoop;

			// reshift the board
			for (iLoop=0; iLoop < BOARDSIZE; iLoop++)
			{
				boardArray[iLoop].TileLabel.Enabled = true;
				boardArray[iLoop].TileLabel.Visible = true;

				boardArray[iLoop].Value = TileValue.NOMINE;
				boardArray[iLoop].Mask = TileMask.CLEAR;

				boardArray[iLoop].X = iLoop % iColumns;
				boardArray[iLoop].Y = iLoop / iColumns;

				boardArray[iLoop].ResetTile();
			}


			// cancel out the rest of the tiles
			for (iLoop=BOARDSIZE; iLoop < MAXROWS * MAXCOLUMNS; iLoop++)
			{
				boardArray[iLoop].TileLabel.Visible = false;
				boardArray[iLoop].TileLabel.Enabled = false;				
			}
			

			// place MINECOUNT random mines
			Random rndGen = new Random();

			int iMinesPlaced = 0, iPos;
			while (iMinesPlaced < iMineCount)
			{
				do
				{
					iPos = rndGen.Next(0, BOARDSIZE - 1);
				} while (boardArray[iPos].Value != TileValue.NOMINE);

				boardArray[iPos].Value = TileValue.MINE;
				iMinesPlaced++;
			}

			//DumpArrayToMsgBox();
			// now place all of the mine counts

			for (iLoop=0; iLoop < BOARDSIZE; iLoop++)
				if (boardArray[iLoop].Value == TileValue.NOMINE)
					// we aren't sitting on a mine, so
					// find how many mines are around us
					boardArray[iLoop].Value = MineCount(iLoop);
		
			//DumpArrayToMsgBox();
		}


		private void DumpArrayToMsgBox()
		{
			String s = "";
			int iMC = 0;
			for (int iLoop=0; iLoop < iColumns*iRows; iLoop++)
			{
				if (iLoop % iColumns == 0) s += "\n";
				switch (boardArray[iLoop].Value)
				{
					case TileValue.MINE:
						// how to display a mine
						s += "M";
						iMC++;
						break;

					case TileValue.NOMINE:
						s += "0";
						break;

					case TileValue.MINECLEARED:
						s += "~";
						break;


					case TileValue.ONEADJMINE:
						s += "1";
						break;

					case TileValue.TWOADJMINE:
						s += "2";
						break;

					case TileValue.THREEADJMINE:
						s += "3";
						break;

					case TileValue.FOURADJMINE:
						s += "4";
						break;

					case TileValue.FIVEADJMINE:
						s += "5";
						break;
				
					case TileValue.SIXADJMINE:
						s += "6";
						break;
				
					case TileValue.SEVENADJMINE:
						s += "7";
						break;
				
					case TileValue.EIGHTADJMINE:
						s += "8";
						break;
				}				
			}

			MessageBox.Show(s + "\nMC = " + iMC.ToString());

		}

		private TileValue MineCount(int iPos)
		{
			// in this function we need to return how many mines
			// are surrounding the current position on the board
			int iMineCount = 0;
			int X, Y;

			X = iPos % iColumns;
			Y = iPos / iColumns;

			// check our left spots
			if (X != 0) 
			{
				if (boardArray[iPos-1].Value == TileValue.MINE)
					iMineCount++;

				// check upper-left
				if (Y != 0)
					if (boardArray[iPos-1-iColumns].Value == TileValue.MINE)
						iMineCount++;

				// check bottom-right
				if (Y < iRows-1)
				{
					if (boardArray[iPos-1+iColumns].Value == TileValue.MINE)
						iMineCount++;
				}
			}

			// check our right spots
			if (X < iColumns-1)
			{
				if (boardArray[iPos+1].Value == TileValue.MINE)
					iMineCount++;

				// check upper-right
				if (Y != 0)
					if (boardArray[iPos+1-iColumns].Value == TileValue.MINE)
						iMineCount++;

				// check bottom-right
				if (Y < iRows-1)
				{
					if (boardArray[iPos+1+iColumns].Value == TileValue.MINE)
						iMineCount++;
				}
			}

			// check above us
			if (Y != 0)
				if (boardArray[iPos-iColumns].Value == TileValue.MINE)
					iMineCount++;

			// check below us
			if (Y < iRows-1)
			{
				if (boardArray[iPos+iColumns].Value == TileValue.MINE)
					iMineCount++;
			}



			switch(iMineCount)
			{
				case 0:
					return TileValue.NOMINE;
					
				case 1:
					return TileValue.ONEADJMINE;
					
				case 2:
					return TileValue.TWOADJMINE;
					
				case 3:
					return TileValue.THREEADJMINE;
					
				case 4:
					return TileValue.FOURADJMINE;
					
				case 5:
					return TileValue.FIVEADJMINE;
					
				case 6:
					return TileValue.SIXADJMINE;
					
				case 7:
					return TileValue.SEVENADJMINE;
					
				case 8:
					return TileValue.EIGHTADJMINE;					
				
				default:
					return TileValue.NOMINE;					
			}
		}


		public void RevealAllMines()
		{
			// loop through an display all of the mines
			for (int iLoop=0; iLoop < iColumns * iRows; iLoop++)				
				boardArray[iLoop].DisplayEndGameTile();
		}

		private void TileClicked(Object sender, MouseEventArgs e)
		{
			// find out what button was clicked
			Tile tmpTile = (Tile) sender;
			
			// if the game is over or we are clicking a cleared mine, we don't need to do anything
			if (progress == GameStatus.LOST || progress == GameStatus.WON || tmpTile.Value == TileValue.MINECLEARED) 
				return;
			
			// if the game hasn't been started yet, start the timer
			if (progress == GameStatus.NOTSTARTED) StartGame();


			// what do we do?
			int iPos = tmpTile.Y*iColumns + tmpTile.X; 
							
			// was the right or left mouse button clicked?
			if (e.Button == MouseButtons.Right)
			{
				// what we want to do depends on the tile's current state
				if (boardArray[iPos].Mask != TileMask.FLAG && boardArray[iPos].Mask != TileMask.QUESTIONMARK)
				{
					// make it a flag					
					boardArray[iPos].Mask = TileMask.FLAG;
					boardArray[iPos].DisplayTile();

					// decrement the number of mines left
					iMineCount--;

					// update the display
					lblMineCount.Text = iMineCount.ToString();

					// Check to see if the user has won
					// they can only win when iMineCount is 0
					if (iMineCount == 0)
					{						
						if (AllMinesCorrectlyPlaced() && AllNonMinesCleared())
							EndGame(true);
					}
				}
				else if (boardArray[iPos].Mask == TileMask.FLAG)
				{
					// make it a question mark					
					boardArray[iPos].Mask = TileMask.QUESTIONMARK;
					boardArray[iPos].DisplayTile();

					// decrement the number of mines left
					iMineCount++;

					// update the display
					lblMineCount.Text = iMineCount.ToString();
				} 
				else if (boardArray[iPos].Mask == TileMask.QUESTIONMARK)
				{
					// return the board to its normal state
					tmpTile.Mask = TileMask.CLEAR;
					boardArray[iPos].ClearTile();
				}				
			}
			else if (e.Button == MouseButtons.Left)
			{
				// if we're sitting on a flag, a left-click should do nothing
				if (tmpTile.Mask == TileMask.FLAG) return;


				if (boardArray[iPos].Mask == TileMask.QUESTIONMARK)
				{
					boardArray[iPos].TileLabel.Text = "";
					boardArray[iPos].Mask = TileMask.CLEAR;
					boardArray[iPos].DisplayTile();
				}

				boardArray[iPos].DisplayTile();		// display the clicked tile

				// what if the tile is a mine or a NOMINE tile?
				if (boardArray[iPos].Value == TileValue.MINE)
				{
					// game over, the person lost
					EndGame(false);
				}
				else 
				{
					if (boardArray[iPos].Value == TileValue.NOMINE)
					{
						// we must clear out all of the needed tiles
						ClearOutNoMines(iPos);
					}

					if (iMineCount == 0)
					{						
						if (AllMinesCorrectlyPlaced() && AllNonMinesCleared())
							EndGame(true);
					}

					// regardless of what happened, I need to mark the mine as cleared
					boardArray[iPos].Value = TileValue.MINECLEARED;
					
					if (NothingButMinesLeft())
					{
						ReplaceMinesWithFlags();
						EndGame(true);
					}
				}
			}
		}



		private bool NothingButMinesLeft()
		{
			// this function needs to check to see if all of the
			// remaining tiles contain nothing but mines
			for (int iLoop=0; iLoop < iColumns*iRows; iLoop++)
				// if we come across an uncovered tile that is NOT a mine,
				// then return false
				if (boardArray[iLoop].Value != TileValue.MINECLEARED)
					if (boardArray[iLoop].Value != TileValue.MINE)
						return false;

			// if we've gotten this far, then we only have mines left to clear
			return true;
		}


		private void ReplaceMinesWithFlags()
		{
			// we need to loop through and replace all of our mine tiles with flag tiles
			// this happens if the user has unclicked mines left (and nothing else)
			for (int iLoop=0; iLoop < iColumns*iRows; iLoop++)
				if (boardArray[iLoop].Value == TileValue.MINE)
				{
					boardArray[iLoop].Mask = TileMask.FLAG;
					boardArray[iLoop].DisplayTile();
				}
		}



		private bool AllNonMinesCleared()
		{
			// check to make sure all of the non-mines have been cleared
			for (int iLoop=0; iLoop < iColumns*iRows; iLoop++)
				if (boardArray[iLoop].Value != TileValue.MINE && boardArray[iLoop].Value != TileValue.MINECLEARED)
					return false;

			// if we get this far, then all that's left are mines (cleared or not)
			return true;
		}
        

		private bool AllMinesCorrectlyPlaced()
		{
			// check to see if all of our mines are correctly placed
			bool winner = true;
			int iCount = 0;

			for (int iLoop=0; iLoop < iColumns*iRows; iLoop++)
				if ((boardArray[iLoop].Value != TileValue.MINE && boardArray[iLoop].Mask == TileMask.FLAG) ||
					(boardArray[iLoop].Value == TileValue.MINE && boardArray[iLoop].Mask != TileMask.FLAG))
						winner = false;
				else
						iCount++;

			return winner && iCount == (iRows*iColumns);
		}


		private void StartGame()
		{
			progress = GameStatus.INPROGRESS;

			this.tmrSecondCount.Enabled = true;
			this.tmrSecondCount.Start();
		}


		private void EndGame(bool bolWon)
		{
			// did we win or lose this game?
			if (bolWon)
			{
				progress = GameStatus.WON;
				this.lblMineCount.Text = "0";		// reset the number of mines left
			}
			else
				progress = GameStatus.LOST;


			RevealAllMines();		

			this.tmrSecondCount.Stop();
			this.tmrSecondCount.Enabled = false;

			// raise the end game event
			GameOver(this, new EventArgs());
		}



		private void ClearOutNoMines(int iPos)
		{
			// this function needs to clear out the current mine and any adjacent mines
			int X,Y;

			// now, check adjacent mines
			X = iPos % iColumns;
			Y = iPos / iColumns;

			if (boardArray[iPos].Value == TileValue.NOMINE)
			{
				// set as mine cleared
				boardArray[iPos].Value = TileValue.MINECLEARED;
				if (boardArray[iPos].Mask == TileMask.QUESTIONMARK)
					// if we're on a question mark, we must clear out the text value
					boardArray[iPos].TileLabel.Text = "";
				boardArray[iPos].Mask = TileMask.CLEAR;

				// apply UI effects
				boardArray[iPos].DisplayTile();

				// check our left spots
				if (X != 0) 
				{
					ClearOutNoMines(iPos-1);

					if (Y != 0)
						ClearOutNoMines(iPos-1-iColumns);

					// check bottom-right
					if (Y < iRows-1)
						ClearOutNoMines(iPos-1+iColumns);
				}

				// check our right spots
				if (X < iColumns-1)
				{
					ClearOutNoMines(iPos+1);

					// check upper-right
					if (Y != 0)
						ClearOutNoMines(iPos+1-iColumns);

					// check bottom-right
					if (Y < iRows-1)
						ClearOutNoMines(iPos+1+iColumns);
				}

				// check above us
				if (Y != 0)
					ClearOutNoMines(iPos-iColumns);

				// check below us
				if (Y < iRows-1)
					ClearOutNoMines(iPos+iColumns);
			} 
			else if (boardArray[iPos].Value != TileValue.MINE && boardArray[iPos].Value != TileValue.MINECLEARED)
			{
				// display the tile that borders the mine
				boardArray[iPos].Mask = TileMask.CLEAR;
				boardArray[iPos].DisplayTile();
				boardArray[iPos].Value = TileValue.MINECLEARED;
			}

		}
	}


	
	/// <summary>
	/// Summary description for Form1.
	/// </summary>
	public class frmMain : System.Windows.Forms.Form
	{
		private System.Windows.Forms.MainMenu mainMenu1;
		private System.Windows.Forms.MenuItem mnuGameNew;
		private System.Windows.Forms.MenuItem mnuGameBeginner;
		private System.Windows.Forms.MenuItem mnuGameIntermediate;
		private System.Windows.Forms.MenuItem mnuGameExpert;
		private System.Windows.Forms.MenuItem mnuGameCustom;
		private System.Windows.Forms.MenuItem mnuGame;
		private System.Windows.Forms.MenuItem mnuHelp;
		private System.Windows.Forms.MenuItem mnuSeparator2;
		private System.Windows.Forms.MenuItem mnuSeparator1;
		private System.ComponentModel.IContainer components;		
		

		private Board board;
		private System.Windows.Forms.Button btnNewGame;

		private Bitmap bmpSmiley, bmpFrowny, bmpGlasses;

		// define graphic constants
		public const int HEADERHEIGHT = 50;

		private int iRows, iColumns, iMines;
		private System.Windows.Forms.MenuItem menuItem1;
		private System.Windows.Forms.MenuItem mnuGameBestTimes;
		private System.Windows.Forms.MenuItem mnuGameExit;


		private String strBestTimeBeginner, strBestTimeIntermediate, strBestTimeExpert;
		private int iBestTimeBeginner, iBestTimeIntermediate, iBestTimeExpert;
		private System.Windows.Forms.MenuItem menuItem2;
		private System.Windows.Forms.MenuItem lblHelpAbout;


		public frmMain()
		{
			//
			// Required for Windows Form Designer support
			//
			InitializeComponent();

			// draw the smiley/frowny faces
			DrawFaces();
			
			// set the height and width of the window
			BuildBoard();

			// get our best times
			GetBestTimes();
		}


		private void GetBestTimes()
		{
			// read in the best times
			String strFilePath = Environment.CurrentDirectory + "\\scores.xml";

			if (!File.Exists(strFilePath))
			{
				// file doesn't exist, set values to null
				iBestTimeBeginner = iBestTimeIntermediate = iBestTimeExpert = 0;
				strBestTimeBeginner = strBestTimeIntermediate = strBestTimeExpert = "";
			} 
			else 
			{
				// load the document
				XmlDocument xmlScores = new XmlDocument();
				xmlScores.PreserveWhitespace = true;
				xmlScores.Load(strFilePath);
				
				
				XmlNode root = xmlScores.FirstChild;

				int iChildLoop;

				if (root.HasChildNodes)
				{
					for (int iLoop=0; iLoop < root.ChildNodes.Count; iLoop++)
						switch (root.ChildNodes[iLoop].Name)
						{
							case "beginner":
								if (root.ChildNodes[iLoop].HasChildNodes)
									for (iChildLoop=0; iChildLoop < root.ChildNodes[iLoop].ChildNodes.Count; iChildLoop++)
										if (root.ChildNodes[iLoop].ChildNodes[iChildLoop].Name == "time")
											iBestTimeBeginner = Convert.ToInt32(root.ChildNodes[iLoop].ChildNodes[iChildLoop].InnerText);
										else if (root.ChildNodes[iLoop].ChildNodes[iChildLoop].Name == "name")
											strBestTimeBeginner = root.ChildNodes[iLoop].ChildNodes[iChildLoop].InnerText;
								break;

							case "intermediate":
								if (root.ChildNodes[iLoop].HasChildNodes)
									for (iChildLoop=0; iChildLoop < root.ChildNodes[iLoop].ChildNodes.Count; iChildLoop++)
										if (root.ChildNodes[iLoop].ChildNodes[iChildLoop].Name == "time")
											iBestTimeIntermediate = Convert.ToInt32(root.ChildNodes[iLoop].ChildNodes[iChildLoop].InnerText);
										else if (root.ChildNodes[iLoop].ChildNodes[iChildLoop].Name == "name")
											strBestTimeIntermediate = root.ChildNodes[iLoop].ChildNodes[iChildLoop].InnerText;
								break;

							case "expert":
								if (root.ChildNodes[iLoop].HasChildNodes)
									for (iChildLoop=0; iChildLoop < root.ChildNodes[iLoop].ChildNodes.Count; iChildLoop++)
										if (root.ChildNodes[iLoop].ChildNodes[iChildLoop].Name == "time")
											iBestTimeExpert = Convert.ToInt32(root.ChildNodes[iLoop].ChildNodes[iChildLoop].InnerText);
										else if (root.ChildNodes[iLoop].ChildNodes[iChildLoop].Name == "name")
											strBestTimeExpert = root.ChildNodes[iLoop].ChildNodes[iChildLoop].InnerText;
								break;
						}	
				}
			}

		}


		private void DrawFaces()
		{
			// draw smile
			bmpSmiley = new Bitmap(this.btnNewGame.Width, this.btnNewGame.Height);
			Graphics imgTile = Graphics.FromImage(bmpSmiley);

			imgTile.FillRectangle(new SolidBrush(Color.LightGray), 0, 0, 40, 40);
			imgTile.DrawEllipse(new Pen(Color.Black), 4, 4, 30, 30);
			imgTile.FillEllipse(new SolidBrush(Color.Yellow), 5, 5, 28, 28);
			imgTile.FillEllipse(new SolidBrush(Color.Black), 12, 15, 4, 4);
			imgTile.FillEllipse(new SolidBrush(Color.Black), 23, 15, 4, 4);

			Point[] aPoints = new Point[4];
			aPoints[0].X = 9; aPoints[0].Y = 23;
			aPoints[1].X = 15; aPoints[1].Y = 27;
			aPoints[2].X = 23; aPoints[2].Y = 27;
			aPoints[3].X = 28; aPoints[3].Y = 23;

			imgTile.DrawCurve(new Pen(Color.Black), aPoints);


			// draw frown
			bmpFrowny = new Bitmap(this.btnNewGame.Width, this.btnNewGame.Height);
			Graphics imgFrownTile = Graphics.FromImage(bmpFrowny);

			imgFrownTile.FillRectangle(new SolidBrush(Color.LightGray), 0, 0, 40, 40);
			imgFrownTile.DrawEllipse(new Pen(Color.Black), 4, 4, 30, 30);
			imgFrownTile.FillEllipse(new SolidBrush(Color.Yellow), 5, 5, 28, 28);
			
			imgFrownTile.DrawLine(new Pen(Color.Black), new Point(13, 15), new Point(10, 9));
			imgFrownTile.DrawLine(new Pen(Color.Black), new Point(10, 15), new Point(13, 9));
			imgFrownTile.DrawLine(new Pen(Color.Black), new Point(23, 15), new Point(27, 9));
			imgFrownTile.DrawLine(new Pen(Color.Black), new Point(27, 15), new Point(23, 9));
			
			aPoints[0].X = 9; aPoints[0].Y = 27;
			aPoints[1].X = 15; aPoints[1].Y = 23;
			aPoints[2].X = 23; aPoints[2].Y = 23;
			aPoints[3].X = 28; aPoints[3].Y = 27;

			imgFrownTile.DrawCurve(new Pen(Color.Black), aPoints);


			// draw glasses
			bmpGlasses = new Bitmap(this.btnNewGame.Width, this.btnNewGame.Height);
			Graphics imgTileGlasses = Graphics.FromImage(bmpGlasses);

			imgTileGlasses.FillRectangle(new SolidBrush(Color.LightGray), 0, 0, 40, 40);
			imgTileGlasses.DrawEllipse(new Pen(Color.Black), 4, 4, 30, 30);
			imgTileGlasses.FillEllipse(new SolidBrush(Color.Yellow), 5, 5, 28, 28);
			imgTileGlasses.FillEllipse(new SolidBrush(Color.Black), 11, 13, 8, 8);
			imgTileGlasses.FillEllipse(new SolidBrush(Color.Black), 21, 13, 8, 8);
			
			aPoints[0].X = 9; aPoints[0].Y = 23;
			aPoints[1].X = 15; aPoints[1].Y = 27;
			aPoints[2].X = 23; aPoints[2].Y = 27;
			aPoints[3].X = 28; aPoints[3].Y = 23;

			imgTileGlasses.DrawCurve(new Pen(Color.Black), aPoints);
			imgTileGlasses.DrawLine(new Pen(Color.Black, 2), new Point(12,17), new Point(25,17));
			imgTileGlasses.DrawLine(new Pen(Color.Black, 2), new Point(5,14), new Point(15,17));
			imgTileGlasses.DrawLine(new Pen(Color.Black, 2), new Point(25,17), new Point(34,14));
			
		}


		private void GameOver(Object sender, EventArgs e)
		{
			if (board.Status == GameStatus.LOST)
				// show the frowny
				this.btnNewGame.BackgroundImage = bmpFrowny;
			else if (board.Status == GameStatus.WON)
			{				
				this.btnNewGame.BackgroundImage = bmpGlasses;

				// did we set a new record?
				// first, what type of game is this

				bool bolSetRecordTime = false;
				String strDifficulty = "";
				
				if (mnuGameBeginner.Checked)
				{
					if (board.Time < iBestTimeBeginner || iBestTimeBeginner == 0)
					{
						// we set a new record
						bolSetRecordTime = true;
						strDifficulty = "beginner";
					}
				} 
				else if (mnuGameIntermediate.Checked)
				{
					if (board.Time < iBestTimeIntermediate || iBestTimeIntermediate == 0)
					{
						// we set a new record
						bolSetRecordTime = true;
						strDifficulty = "intermediate";
					}
				}
				else if (mnuGameExpert.Checked)
				{
					if (board.Time < iBestTimeExpert  || iBestTimeExpert == 0)
					{
						// we set a new record
						bolSetRecordTime = true;
						strDifficulty = "expert";
					}
				}

				// display the SetRecord dialog box, if needed
				if (bolSetRecordTime)
				{
					frmSetRecord frmRecord = new frmSetRecord();
					frmRecord.Time = board.Time;
					frmRecord.SetLabelValues();
					frmRecord.Difficulty = strDifficulty;

					frmRecord.ShowDialog(this);

					// update the score
					switch (strDifficulty)
					{
						case "beginner":
							iBestTimeBeginner = board.Time;
							strBestTimeBeginner = frmRecord.WinnersName;
							break;

						case "intermediate":
							iBestTimeIntermediate = board.Time;
							strBestTimeIntermediate = frmRecord.WinnersName;
							break;

						case "expert":
							iBestTimeExpert = board.Time;
							strBestTimeExpert = frmRecord.WinnersName;
							break;
					}
				}
			}
		}


		private void CheckMenu()
		{
			if (iColumns == Board.BEGINNERDEFAULTCOLUMNS &&
				iRows == Board.BEGINNERDEFAULTROWS &&
				iMines == Board.BEGINNERDEFAULTMINECOUNT)
			{
				mnuGameBeginner.Checked = true;
				mnuGameIntermediate.Checked = false;
				mnuGameExpert.Checked = false;
				mnuGameCustom.Checked = false;
			}
			else if (iColumns == Board.INTERMEDIATEDEFAULTCOLUMNS &&
				iRows == Board.INTERMEDIATEDEFAULTROWS &&
				iMines == Board.INTERMEDIATEDEFAULTMINECOUNT)
			{
				mnuGameBeginner.Checked = false;
				mnuGameIntermediate.Checked = true;
				mnuGameExpert.Checked = false;
				mnuGameCustom.Checked = false;
			}
			else if (iColumns == Board.EXPERTDEFAULTCOLUMNS &&
				iRows == Board.EXPERTDEFAULTROWS &&
				iMines == Board.EXPERTDEFAULTMINECOUNT)
			{
				mnuGameBeginner.Checked = false;
				mnuGameIntermediate.Checked = false;
				mnuGameExpert.Checked = true;
				mnuGameCustom.Checked = false;
			}
			else 
			{
				mnuGameBeginner.Checked = false;
				mnuGameIntermediate.Checked = false;
				mnuGameExpert.Checked = false;
				mnuGameCustom.Checked = true;
			}
		}


		private void BuildBoard()
		{
			// play a beginner board by default
			iColumns = Board.BEGINNERDEFAULTCOLUMNS;
			iRows = Board.BEGINNERDEFAULTROWS;
			iMines = Board.BEGINNERDEFAULTMINECOUNT;

			CheckMenu();

			ResizeWindow();			


			// create an instance of the board
			board = new Board(iMines, iColumns, iRows);

			board.GameOver += new GameOverEventHandler(this.GameOver);

			board.AttachTilesToForm(this);

			btnNewGame.BackgroundImage = bmpSmiley;
		}


		private void ResizeWindow()
		{
			// Calculate the total size we'll need
			Size s = new Size(iColumns * Tile.TILESIZEX + 8, (iRows+2) * Tile.TILESIZEY + 10 + HEADERHEIGHT);

			// resize the window
			this.MaximumSize = s;
			this.MinimumSize = s;

			this.Width = s.Width;;
			this.Height = s.Height;

			// set the location of the New Game button
			this.btnNewGame.Location = new System.Drawing.Point(this.Width / 2 - this.btnNewGame.Width / 2-4, 4);
		}


		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if (components != null) 
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.mnuGameIntermediate = new System.Windows.Forms.MenuItem();
			this.mnuHelp = new System.Windows.Forms.MenuItem();
			this.mnuSeparator1 = new System.Windows.Forms.MenuItem();
			this.mainMenu1 = new System.Windows.Forms.MainMenu();
			this.mnuGame = new System.Windows.Forms.MenuItem();
			this.mnuGameNew = new System.Windows.Forms.MenuItem();
			this.mnuSeparator2 = new System.Windows.Forms.MenuItem();
			this.mnuGameBeginner = new System.Windows.Forms.MenuItem();
			this.mnuGameExpert = new System.Windows.Forms.MenuItem();
			this.mnuGameCustom = new System.Windows.Forms.MenuItem();
			this.mnuGameBestTimes = new System.Windows.Forms.MenuItem();
			this.menuItem1 = new System.Windows.Forms.MenuItem();
			this.mnuGameExit = new System.Windows.Forms.MenuItem();
			this.btnNewGame = new System.Windows.Forms.Button();
			this.menuItem2 = new System.Windows.Forms.MenuItem();
			this.lblHelpAbout = new System.Windows.Forms.MenuItem();
			this.SuspendLayout();
			// 
			// mnuGameIntermediate
			// 
			this.mnuGameIntermediate.Index = 3;
			this.mnuGameIntermediate.Text = "&Intermediate";
			this.mnuGameIntermediate.Click += new System.EventHandler(this.mnuGameIntermediate_Click);
			// 
			// mnuHelp
			// 
			this.mnuHelp.Index = 1;
			this.mnuHelp.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					this.menuItem2,
																					this.lblHelpAbout});
			this.mnuHelp.Text = "Help";
			// 
			// mnuSeparator1
			// 
			this.mnuSeparator1.Index = 6;
			this.mnuSeparator1.Text = "-";
			// 
			// mainMenu1
			// 
			this.mainMenu1.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					  this.mnuGame,
																					  this.mnuHelp});
			// 
			// mnuGame
			// 
			this.mnuGame.Index = 0;
			this.mnuGame.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					this.mnuGameNew,
																					this.mnuSeparator2,
																					this.mnuGameBeginner,
																					this.mnuGameIntermediate,
																					this.mnuGameExpert,
																					this.mnuGameCustom,
																					this.mnuSeparator1,
																					this.mnuGameBestTimes,
																					this.menuItem1,
																					this.mnuGameExit});
			this.mnuGame.Text = "Game";
			// 
			// mnuGameNew
			// 
			this.mnuGameNew.Index = 0;
			this.mnuGameNew.Shortcut = System.Windows.Forms.Shortcut.F2;
			this.mnuGameNew.Text = "&New";
			this.mnuGameNew.Click += new System.EventHandler(this.mnuGameNew_Click);
			// 
			// mnuSeparator2
			// 
			this.mnuSeparator2.Index = 1;
			this.mnuSeparator2.Text = "-";
			// 
			// mnuGameBeginner
			// 
			this.mnuGameBeginner.Index = 2;
			this.mnuGameBeginner.Text = "&Beginner";
			this.mnuGameBeginner.Click += new System.EventHandler(this.mnuGameBeginner_Click);
			// 
			// mnuGameExpert
			// 
			this.mnuGameExpert.Index = 4;
			this.mnuGameExpert.Text = "&Expert";
			this.mnuGameExpert.Click += new System.EventHandler(this.mnuGameExpert_Click);
			// 
			// mnuGameCustom
			// 
			this.mnuGameCustom.Index = 5;
			this.mnuGameCustom.Text = "&Custom...";
			this.mnuGameCustom.Click += new System.EventHandler(this.mnuGameCustom_Click);
			// 
			// mnuGameBestTimes
			// 
			this.mnuGameBestTimes.Index = 7;
			this.mnuGameBestTimes.Text = "Best &Times...";
			this.mnuGameBestTimes.Click += new System.EventHandler(this.menuItem2_Click);
			// 
			// menuItem1
			// 
			this.menuItem1.Index = 8;
			this.menuItem1.Text = "-";
			// 
			// mnuGameExit
			// 
			this.mnuGameExit.Index = 9;
			this.mnuGameExit.Text = "E&xit";
			this.mnuGameExit.Click += new System.EventHandler(this.mnuGameExit_Click);
			// 
			// btnNewGame
			// 
			this.btnNewGame.BackColor = System.Drawing.Color.LightGray;
			this.btnNewGame.Location = new System.Drawing.Point(168, 8);
			this.btnNewGame.Name = "btnNewGame";
			this.btnNewGame.Size = new System.Drawing.Size(40, 40);
			this.btnNewGame.TabIndex = 0;
			this.btnNewGame.Click += new System.EventHandler(this.btnNewGame_Click);
			// 
			// menuItem2
			// 
			this.menuItem2.Index = 0;
			this.menuItem2.Text = "-";
			// 
			// lblHelpAbout
			// 
			this.lblHelpAbout.Index = 1;
			this.lblHelpAbout.Text = "&About Minesweeper...";
			this.lblHelpAbout.Click += new System.EventHandler(this.lblHelpAbout_Click);
			// 
			// frmMain
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			this.ClientSize = new System.Drawing.Size(292, 273);
			this.Controls.AddRange(new System.Windows.Forms.Control[] {
																		  this.btnNewGame});
			this.MaximizeBox = false;
			this.Menu = this.mainMenu1;
			this.Name = "frmMain";
			this.Text = "Minesweeper.NET";
			this.ResumeLayout(false);

		}
		#endregion

		/// <summary>
		/// The main entry point for the application.
		/// </summary>
		[STAThread]
		static void Main() 
		{
			Application.Run(new frmMain());
		}


		private void StartNewGame()
		{
			bool bolResizedWindow = false;
			
			CheckMenu();

			if (iColumns < board.Columns && iRows < board.Rows)
			{
				ResizeWindow();
				bolResizedWindow = true;
			}

			// start a new game when this guy is clicked
			board.RebuildBoard(iMines, iColumns, iRows);

			// slap on the smiley face to the button
			this.btnNewGame.BackgroundImage = bmpSmiley;

			if (!bolResizedWindow) ResizeWindow();			
		}
		
		private void btnNewGame_Click(object sender, System.EventArgs e)
		{
			// start a new game
			StartNewGame();
		}

		private void mnuGameNew_Click(object sender, System.EventArgs e)
		{
			// start a new game
			StartNewGame();
		}

		private void mnuGameBeginner_Click(object sender, System.EventArgs e)
		{
			// configure a new game for the beginner
			iMines = Board.BEGINNERDEFAULTMINECOUNT;
			iColumns = Board.BEGINNERDEFAULTCOLUMNS;
			iRows = Board.BEGINNERDEFAULTROWS;
			
			StartNewGame();
		}

		private void mnuGameIntermediate_Click(object sender, System.EventArgs e)
		{
			// configure a new game for the intermediate
			iMines = Board.INTERMEDIATEDEFAULTMINECOUNT;
			iColumns = Board.INTERMEDIATEDEFAULTCOLUMNS;
			iRows = Board.INTERMEDIATEDEFAULTROWS;

			StartNewGame();
		}

		private void mnuGameExpert_Click(object sender, System.EventArgs e)
		{
			// configure a new game for the expert
			iMines = Board.EXPERTDEFAULTMINECOUNT;
			iColumns = Board.EXPERTDEFAULTCOLUMNS;
			iRows = Board.EXPERTDEFAULTROWS;

			StartNewGame();
		}

		private void mnuGameCustom_Click(object sender, System.EventArgs e)
		{
			frmCustomField fCust = new frmCustomField();
			if (fCust.ShowDialog(this) == DialogResult.OK)
			{
				iRows = fCust.getRowsValue();
				iColumns = fCust.getColumnsValue();
				iMines = fCust.getMinesValue();

				StartNewGame();
			}

			
		}

		private void mnuGameExit_Click(object sender, System.EventArgs e)
		{
			Application.Exit();
		}

		private void menuItem2_Click(object sender, System.EventArgs e)
		{
			frmBestTimes fBest = new frmBestTimes();

			fBest.BestTimeBeginner = iBestTimeBeginner;
			fBest.BestTimeIntermediate = iBestTimeIntermediate;
			fBest.BestTimeExpert = iBestTimeExpert;

			fBest.BestTimeBeginnerName = strBestTimeBeginner;
			fBest.BestTimeIntermediateName = strBestTimeIntermediate;
			fBest.BestTimeExpertName = strBestTimeExpert;

			fBest.SetLableValues();

			fBest.ShowDialog(this);
		}

		private void lblHelpAbout_Click(object sender, System.EventArgs e)
		{
			frmAbout fAbout = new frmAbout();
			fAbout.ShowDialog(this);

		}


	}
}
